package ru.yandex.direct.core.entity.keyword.processing;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.advq.query.ast.Word;
import ru.yandex.advq.query.ast.WordKind;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.libs.keywordutils.helper.ParseKeywordCache;
import ru.yandex.direct.libs.keywordutils.helper.SingleKeywordsCache;
import ru.yandex.direct.libs.keywordutils.inclusion.model.SingleKeywordWithLemmas;
import ru.yandex.direct.libs.keywordutils.model.AnyKeyword;
import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;
import ru.yandex.direct.libs.keywordutils.model.OrderedKeyword;
import ru.yandex.direct.libs.keywordutils.model.SingleKeyword;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.core.entity.keyword.processing.MinusKeywordsDeduplicator.deduplicateSubKeywords;
import static ru.yandex.direct.libs.keywordutils.parser.SplitCompoundWordExpressionTransform.splitRawWordRegexp;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class KeywordNormalizer {

    private final SingleKeywordsCache singleKeywordsCache;

    private final StopWordService stopWordService;

    private final ParseKeywordCache parseKeywordCache;

    @Autowired
    public KeywordNormalizer(StopWordService stopWordService,
                             SingleKeywordsCache singleKeywordsCache,
                             ParseKeywordCache parseKeywordCache) {
        this.singleKeywordsCache = singleKeywordsCache;
        this.stopWordService = stopWordService;
        this.parseKeywordCache = parseKeywordCache;
    }

    public NormalizedKeyword normalizeKeyword(Keyword originalKeyword) {
        return normalizeKeyword(originalKeyword, true);
    }

    /**
     * Нормализация фразы.
     * <p>
     * Учитывается, что слова могут оканчиваться на точку, эта точка убирается.
     *
     * @param originalKeyword   исходная фраза
     * @param needDeduplication нужно ли удалять дублирующиеся слова
     * @return нормализованная фраза
     */
    public NormalizedKeyword normalizeKeyword(Keyword originalKeyword, boolean needDeduplication) {
        List<SingleKeyword> singleSubKeywords =
                StreamEx.of(originalKeyword.getAllKeywords())
                        .select(SingleKeyword.class)
                        .toList();
        List<OrderedKeyword> orderedSubKeywords =
                StreamEx.of(originalKeyword.getAllKeywords())
                        .select(OrderedKeyword.class)
                        .toList();

        // разделяем составные слова
        singleSubKeywords = flatMap(singleSubKeywords, this::splitSingleKeyword);
        orderedSubKeywords = mapList(orderedSubKeywords, this::splitOrderedKeyword);

        if (needDeduplication) {
            // Удаляем дублирующиеся слова в исходных фразах.
            singleSubKeywords = deduplicateSubKeywords(singleKeywordsCache, singleSubKeywords, SingleKeyword.class);
            orderedSubKeywords = deduplicateSubKeywords(singleKeywordsCache, orderedSubKeywords, OrderedKeyword.class);
        }
        List<SingleKeyword> originalSingleSubKeywords = new ArrayList<>(singleSubKeywords);
        List<OrderedKeyword> originalOrderedSubKeywords = new ArrayList<>(orderedSubKeywords);

        // подготавливаем слова. приводим все слова к нижнему регистру,
        // так как от этого может зависеть определение стоп-слов
        singleSubKeywords = mapList(singleSubKeywords, this::prepareNormalization);
        orderedSubKeywords = mapList(orderedSubKeywords, this::prepareNormalization);

        // заменяем за переделами квадратных скобок
        // все незафиксированные слова на их лексемы, не являющиеся стоп-словами
        List<SingleKeyword> firstLemmaSingleSubKeywords = StreamEx.of(singleSubKeywords)
                .flatCollection(this::replaceByFirstLemmaOmitStopWords)
                .toList();

        // заменяем в квадратных скобках все незафиксированные слова на их первые леммы
        List<OrderedKeyword> firstLemmaOrderedSubKeywords = StreamEx.of(orderedSubKeywords)
                .map(this::replaceByFirstLemmaPreserveStopWords)
                .toList();

        // удаляем знак "+" у слов, не являющихся стоп-словами
        firstLemmaSingleSubKeywords =
                removePlusFromNonStopWordsInSingleKeywords(firstLemmaSingleSubKeywords);
        firstLemmaOrderedSubKeywords =
                removePlusFromNonStopWordsInOrderedKeywords(firstLemmaOrderedSubKeywords);

        List<NormalizedWord<AnyKeyword>> allSortedSubKeywords = new ArrayList<>(
                StreamEx.zip(originalSingleSubKeywords, firstLemmaSingleSubKeywords, NormalizedWord<AnyKeyword>::new)
                        // удаляем все незафиксированные знаком "+" или "!" стоп-слова за пределами кавычек и
                        // квадратных скобок
                        .remove(stopWordShouldBeRemoved(originalKeyword.isQuoted()))
                        .toList());
        allSortedSubKeywords.addAll(
                StreamEx.zip(originalOrderedSubKeywords, firstLemmaOrderedSubKeywords, NormalizedWord<AnyKeyword>::new)
                        .toList());
        // сортируем
        Collections.sort(allSortedSubKeywords);

        return new NormalizedKeyword(originalKeyword, allSortedSubKeywords);
    }

    private Predicate<NormalizedWord<AnyKeyword>> stopWordShouldBeRemoved(boolean isKeywordQuoted) {
        return subword -> {
            SingleKeyword normalizedSubword = (SingleKeyword) subword.getNormalizedWord();
            return !isKeywordQuoted && stopWordService.isStopWord(normalizedSubword.getWord().getText()) &&
                    normalizedSubword.getWord().getKind() == WordKind.RAW;
        };
    }

    public String normalizeKeyword(String keyword) {
        return normalizeKeyword(keyword, true);
    }

    public String normalizeKeyword(String keyword, boolean needDeduplication) {
        Keyword parsedKeyword = parseKeywordCache.parse(keyword.trim());
        return normalizeKeyword(parsedKeyword, needDeduplication).getNormalized().toString();
    }

    public List<String> normalizeKeywords(List<String> keywords) {
        return mapList(keywords, this::normalizeKeyword);
    }

    public KeywordWithMinuses normalizeKeywordWithMinuses(KeywordWithMinuses keywordWithMinuses) {
        return normalizeKeywordWithMinuses(keywordWithMinuses, null, null)
                .toKeywordWithMinuses();
    }

    /**
     * Нормализует саму фразу, а также минус-слова.
     * <p>
     * (!) Поддерживает только ключевые фразы с минус-словами, НЕ с минус-фразами.
     * То есть в каждой минус-фразе {@link KeywordWithMinuses#getMinusKeywords()}
     * должно быть только одно слово.
     *
     * @param keywordWithMinuses    исходная фраза с минус-словами
     * @param normalizeMinusWordMap мапа для мемоизации нормализации минус-слов. Если null - мемоизация не требуется.
     * @param normalizeKeywordMap   мапа для мемоизации нормализации ключевой фразы. Если null - мемоизация не
     *                              требуется.
     * @return нормализованная фраза с минус-словами
     */
    public NormalizedKeywordWithMinuses normalizeKeywordWithMinuses(KeywordWithMinuses keywordWithMinuses,
                                                                    @Nullable Map<SingleKeyword,
                                                                            List<NormalizedWord<SingleKeyword>>> normalizeMinusWordMap,
                                                                    @Nullable Map<Keyword, NormalizedKeyword> normalizeKeywordMap) {
        checkAllMinusKeywordsAreSingle(keywordWithMinuses);

        NormalizedKeyword plusPhraseContainer =
                normalizeKeywordMemoized(keywordWithMinuses.getKeyword(), normalizeKeywordMap);

        // Нормализуем каждое отдельное минус слово,
        // удаляем дубликаты минус-слов и сортируем лексикографически.
        return new NormalizedKeywordWithMinuses(plusPhraseContainer,
                StreamEx.of(keywordWithMinuses.getMinusKeywords())
                        .map(minusKeyword -> (SingleKeyword) minusKeyword.getAllKeywords().get(0))
                        .flatCollection(t -> normalizeMinusWordMemoized(t, normalizeMinusWordMap))
                        .distinct()
                        .sorted()
                        .toList());
    }

    private List<NormalizedWord<SingleKeyword>> normalizeMinusWordMemoized(SingleKeyword keyword,
                                                                           @Nullable Map<SingleKeyword,
                                                                                   List<NormalizedWord<SingleKeyword>>> resultMapping) {
        if (resultMapping == null) {
            return normalizeMinusWord(keyword);
        }
        return resultMapping.computeIfAbsent(keyword, this::normalizeMinusWord);
    }

    private NormalizedKeyword normalizeKeywordMemoized(Keyword keyword,
                                                       @Nullable Map<Keyword, NormalizedKeyword> resultMapping) {
        if (resultMapping == null) {
            return normalizeKeyword(keyword);
        }
        return resultMapping.computeIfAbsent(keyword, this::normalizeKeyword);
    }

    /**
     * Аналог {@link #normalizeKeywordWithMinuses(KeywordWithMinuses)} с перехватом исключений.
     */
    @Nullable
    public KeywordWithMinuses safeNormalizeKeywordWithMinuses(@Nullable KeywordWithMinuses keywordWithMinuses) {
        if (keywordWithMinuses == null) {
            return null;
        }
        try {
            return normalizeKeywordWithMinuses(keywordWithMinuses);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * Нормализация одного минус-слова.
     */
    private List<NormalizedWord<SingleKeyword>> normalizeMinusWord(SingleKeyword minusWord) {
        List<SingleKeyword> originalMinusWords = splitSingleKeyword(minusWord);
        List<SingleKeyword> minusWords = mapList(originalMinusWords, this::prepareNormalization);

        // фиксируем стоп-слово оператором "!", если у стоп-слова нет "+"
        minusWords = mapList(minusWords, this::addFixedToStopWord);

        // заменяем все слова, не зафиксированные операторами "!" и "+", на их первые леммы
        List<SingleKeyword> firstLemmaSingleKeywords = flatMap(minusWords, this::replaceByFirstLemmaOmitStopWords);

        // удаляем знак "+" у слов, не являющихся стоп-словами
        return StreamEx.zip(originalMinusWords, removePlusFromNonStopWordsInSingleKeywords(firstLemmaSingleKeywords),
                NormalizedWord::new)
                .toList();
    }

    /**
     * Разделяет составное слово.
     */
    private List<SingleKeyword> splitSingleKeyword(SingleKeyword singleKeyword) {
        Word word = singleKeyword.getWord();
        return splitRawWordRegexp(word.getText())
                .stream()
                .map(text -> new SingleKeyword(new Word(word.getKind(), text)))
                .collect(Collectors.toList());
    }

    /**
     * Разделяет составные слова.
     */
    private OrderedKeyword splitOrderedKeyword(OrderedKeyword orderedKeyword) {
        List<SingleKeyword> newSingleKeywords = flatMap(orderedKeyword.getSingleKeywords(), this::splitSingleKeyword);
        return new OrderedKeyword(newSingleKeywords);
    }

    /**
     * Подготавливает фразу к нормализации. Приводит к нижнему регистру
     */
    private SingleKeyword prepareNormalization(SingleKeyword singleKeyword) {
        return toLowerCase(singleKeyword);
    }

    /**
     * Подготавливает фразу к нормализации. Приводит к нижнему регистру
     */
    private OrderedKeyword prepareNormalization(OrderedKeyword orderedKeyword) {
        List<SingleKeyword> newSingleKeywords = mapList(orderedKeyword.getSingleKeywords(), this::prepareNormalization);
        return new OrderedKeyword(newSingleKeywords);
    }

    private SingleKeyword toLowerCase(SingleKeyword singleKeyword) {
        Word oldWord = singleKeyword.getWord();
        Word newWord = new Word(oldWord.getKind(), oldWord.getText().toLowerCase());
        return new SingleKeyword(newWord);
    }

    /*
        При замене на лемму слова:
        Если слово не в квадратных скобках, то возвращаем первую лемму не являющейся стоп-словом
        Если все леммы - стоп-слова, то берем первую лемму
     */
    private List<SingleKeyword> replaceByFirstLemmaOmitStopWords(SingleKeyword singleKeyword) {
        return replaceByFirstLemma(singleKeyword, false);
    }

    private List<SingleKeyword> replaceByFirstLemmaPreserveStopWords(SingleKeyword singleKeyword) {
        return replaceByFirstLemma(singleKeyword, true);
    }

    private List<SingleKeyword> replaceByFirstLemma(SingleKeyword singleKeyword, boolean preserveStopWords) {
        boolean isFixed = singleKeyword.getWord().getKind() == WordKind.FIXED;
        boolean isPlusKind = singleKeyword.getWord().getKind() == WordKind.PLUS;
        List<SingleKeywordWithLemmas> keywordWithLemmas =
                singleKeywordsCache.singleKeywordsFrom(singleKeyword.getWord());
        return StreamEx.of(keywordWithLemmas)
                .map(kwl -> isFixed
                        || isPlusKind && stopWordService.isStopWord(singleKeyword.getWord().getText())
                        ? kwl.getNormalizedForm() : getNormalizedLemma(kwl.getLemmas(), preserveStopWords))
                .map(text -> new SingleKeyword(new Word(singleKeyword.getWord().getKind(), text)))
                .toList();
    }

    private OrderedKeyword replaceByFirstLemmaPreserveStopWords(OrderedKeyword orderedKeyword) {
        List<SingleKeyword> firstLemmaSingleKeywords = StreamEx.of(orderedKeyword.getSingleKeywords())
                .toFlatList(this::replaceByFirstLemmaPreserveStopWords);
        return new OrderedKeyword(firstLemmaSingleKeywords);
    }

    /*
       Если слово не в квадратных скобках, то возвращаем первую лемму не являющейся стоп-словом
       Если все леммы - стоп-слова, то берем первую лемму
     */
    private String getNormalizedLemma(List<String> lemmas, boolean preserveStopWords) {
        //TODO разобраться после релиза (или отрыва перла), почему в квадратных скобках и кавычках леммы-стоп-слова
        //обрабатываются по-разному
        //Пример из тестов:
        // у слова "бывшая" две леммы: быть (стоп-слово) и бывший
        //[бывшая жена] -> [быть жена]
        //"бывшая жена" -> "бывший жена"
        String firstLemma = lemmas.get(0);
        if (preserveStopWords) {
            return firstLemma;
        }
        return StreamEx.of(lemmas)
                .remove(stopWordService::isStopWord)
                .findFirst()
                .orElse(firstLemma);
    }

    private SingleKeyword addFixedToStopWord(SingleKeyword singleKeyword) {
        boolean isFixed = singleKeyword.getWord().getKind() == WordKind.FIXED;
        boolean isPlusKind = singleKeyword.getWord().getKind() == WordKind.PLUS;
        return !isFixed && !isPlusKind && stopWordService.isStopWord(singleKeyword.getWord().getText()) ?
                new SingleKeyword(new Word(WordKind.FIXED, singleKeyword.getWord().getText()))
                : singleKeyword;
    }

    private List<SingleKeyword> removePlusFromNonStopWordsInSingleKeywords(List<SingleKeyword> singleKeywords) {
        return mapList(singleKeywords, this::removePlusIfNotStopWord);
    }

    private List<OrderedKeyword> removePlusFromNonStopWordsInOrderedKeywords(List<OrderedKeyword> orderedKeyword) {

        return StreamEx.of(orderedKeyword)
                .map(ok -> {
                    List<SingleKeyword> singleKeywords = mapList(ok.getSingleKeywords(), this::removePlusIfNotStopWord);
                    return new OrderedKeyword(singleKeywords);
                })
                .toList();
    }

    private SingleKeyword removePlusIfNotStopWord(SingleKeyword singleKeyword) {
        if (!stopWordService.isStopWord(singleKeyword.getWord().getText()) &&
                singleKeyword.getWord().getKind() == WordKind.PLUS) {
            return new SingleKeyword(new Word(WordKind.RAW, singleKeyword.getWord().getText()));
        } else {
            return singleKeyword;
        }
    }

    private static void checkAllMinusKeywordsAreSingle(KeywordWithMinuses keywordWithMinuses) {
        for (ru.yandex.direct.libs.keywordutils.model.Keyword minusKeyword : keywordWithMinuses.getMinusKeywords()) {
            checkArgument(!minusKeyword.isQuoted(), "minusKeyword cannot be quoted");
            checkArgument(minusKeyword.getAllKeywords().size() == 1,
                    "minusKeyword has more than one AnyKeyword: %s", minusKeyword);
            checkArgument(minusKeyword.getAllKeywords().stream().allMatch(k -> k instanceof SingleKeyword),
                    "minusKeyword must contain only one SingleKeyword: %s", minusKeyword);
        }
    }
}
