package ru.yandex.direct.core.entity.keyword.processing;

import java.util.List;

import ru.yandex.advq.query.ast.Word;
import ru.yandex.advq.query.ast.WordKind;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.libs.keywordutils.model.AnyKeyword;
import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;
import ru.yandex.direct.libs.keywordutils.model.OrderedKeyword;
import ru.yandex.direct.libs.keywordutils.model.SingleKeyword;

import static org.apache.commons.lang3.StringUtils.join;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class KeywordProcessingUtils {

    private static final String SPEC_SYMBOLS_REGEXP = "[!+\\-\"\\[\\]]";
    private static final String SPACES_REGEXP = "\\s+";
    private static final String EMPTY_STRING = "";
    private static final String KEYWORDS_LENGTH_REPLACES_REGEXP = "-[!+]";

    public static String removeSpecSymbols(String keyword) {
        return keyword.replaceAll(SPEC_SYMBOLS_REGEXP, EMPTY_STRING);
    }

    public static int getLengthWithoutSpecSymbolsAndSpaces(List<String> keywords) {
        String joined = join(keywords.toArray());
        return removeSpaces(removeSpecSymbols(joined)).length();
    }

    public static int getLengthWithoutMinusExclamationAndSpaces(String keyword) {
        return keyword.trim().replaceAll("\\s+", " ")
                .replaceAll(KEYWORDS_LENGTH_REPLACES_REGEXP, "-").length();
    }

    public static String[] splitBySpaces(String keyword) {
        return keyword.split(SPACES_REGEXP);
    }

    public static String removeSpaces(String keyword) {
        return keyword.replaceAll(SPACES_REGEXP, EMPTY_STRING);
    }

    /**
     * Плюс слова из NormalizedKeyword, которые являются OrderedKeyword, но содержат одно слово, превращают в
     * SingleKeyword
     */
    public static NormalizedKeyword cleanLoneWordsFromBrackets(StopWordService stopWordService,
                                                               NormalizedKeyword keyword) {
        return new NormalizedKeyword(keyword.getOriginal(), mapList(keyword.getAllKeywords(),
                kw -> loneOrderedWordToSingle(stopWordService, kw)));
    }

    private static NormalizedWord<AnyKeyword> loneOrderedWordToSingle(StopWordService stopWordService,
                                                                      NormalizedWord<AnyKeyword> subword) {
        return new NormalizedWord<>(loneOrderedWordToSingle(stopWordService, subword.getOriginalWord()),
                loneOrderedWordToSingle(stopWordService, subword.getNormalizedWord()));
    }

    /**
     * Превращаем одиночные слова в квадратных скобках из OrderedKeyword в SingleKeyword, остальные без изменений.
     * Если это одиночное слово являлось стоп-словом и не было явно зафиксировано, оно фиксируется оператором +
     */
    private static AnyKeyword loneOrderedWordToSingle(StopWordService stopWordService, AnyKeyword anyKeyword) {
        if (!(anyKeyword instanceof OrderedKeyword)
                || ((OrderedKeyword) anyKeyword).getSingleKeywords().size() != 1) {
            return anyKeyword;
        }
        SingleKeyword singleKeyword = ((OrderedKeyword) anyKeyword).getSingleKeywords().get(0);
        Word word = singleKeyword.getWord();
        if (stopWordService.isStopWord(word.getText())
                && !word.getKind().equals(WordKind.PLUS)
                && !word.getKind().equals(WordKind.FIXED)) {
            return new SingleKeyword(new Word(WordKind.PLUS, word.getText()));
        }
        return singleKeyword;
    }

    /**
     * Проверяем наличие стоп-слов в кавычках и удаляем "+"
     */

    public static KeywordWithMinuses cleanWordsInBracketsFromPlus(KeywordWithMinuses keywordWithMinuses) {
        if (keywordWithMinuses.getKeyword().isQuoted()) {
            List<AnyKeyword> anyKeywords =
                    mapList(keywordWithMinuses.getKeyword().getAllKeywords(), KeywordProcessingUtils::plusWordToRaw);
            return new KeywordWithMinuses(new Keyword(true, anyKeywords), keywordWithMinuses.getMinusKeywords());
        } else {
            return keywordWithMinuses;
        }
    }

    /**
     * Превращаем слово из стоп-слова в слово с WordKind.RAW
     */
    private static AnyKeyword plusWordToRaw(AnyKeyword anyKeyword) {
        if (anyKeyword instanceof OrderedKeyword) {
            List<SingleKeyword> anyKeywords =
                    mapList(((OrderedKeyword) anyKeyword).getSingleKeywords(), KeywordProcessingUtils::plusWordToRaw);
            return new OrderedKeyword(anyKeywords);

        }
        if (anyKeyword instanceof SingleKeyword) {
            return plusWordToRaw((SingleKeyword) anyKeyword);
        }
        return anyKeyword;
    }

    private static SingleKeyword plusWordToRaw(SingleKeyword singleKeyword) {
        Word word = singleKeyword.getWord();
        if (word.getKind().equals(WordKind.PLUS)) {
            return new SingleKeyword(new Word(WordKind.RAW, word.getText()));
        } else {
            return singleKeyword;
        }
    }
}
