package ru.yandex.direct.core.entity.keyword.processing;

import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.advq.query.ast.Expression;
import ru.yandex.advq.query.ast.ExpressionStopWordsTransform;
import ru.yandex.advq.query.ast.ExpressionVisitor;
import ru.yandex.advq.query.ast.Word;
import ru.yandex.advq.query.ast.WordKind;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.parser.KeywordParser;
import ru.yandex.direct.libs.keywordutils.parser.SplitCompoundWordExpressionTransform;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class KeywordStopwordsFixer {

    private final StopWordService stopWordService;

    @Autowired
    public KeywordStopwordsFixer(StopWordService stopWordService) {
        this.stopWordService = stopWordService;
    }

    /**
     * Принудительно фиксирует указанным оператором стоп-слова
     * (находящиеся за пределами квадратных скобок)
     * во всех фразах в списке, которые не заключены в кавычки.
     * Также разделяет сложные слова на составные части. Напр., санкт-петербург -> санкт петербург
     *
     * @param keywords список ключевых фраз (плюс/минус).
     * @param fixType  способ фиксации.
     * @return список фраз разделенными сложными словами и с принудительно зафиксированными стоп-словами.
     */
    public List<String> fixStopwords(List<String> keywords, WordKind fixType) {
        ExpressionVisitor<Expression> stopwordsFixingVisitor = getStopwordsFixingVisitor(fixType);
        return mapList(keywords, kw -> fixStopwords(kw, stopwordsFixingVisitor));
    }

    /**
     * Для каждой фразы в списке:
     * если фраза заключена в кавычки, то удаляет кавычки и фиксирует
     * указанным оператором стоп-слова (находящиеся за пределами квадратных скобок);
     * если фраза не заключена в кавычки, то возвращает ее как есть.
     *
     * @param keywords список ключевых фраз (плюс/минус).
     * @param fixType  способ фиксации.
     */
    public List<String> unquoteAndFixStopwords(List<String> keywords, WordKind fixType) {
        ExpressionVisitor<Expression> stopwordsFixingVisitor = getStopwordsFixingVisitor(fixType);
        return mapList(keywords, keyword -> unquoteAndFixStopwords(keyword, stopwordsFixingVisitor));
    }

    private ExpressionVisitor<Expression> getStopwordsFixingVisitor(WordKind fixType) {
        return new StopWordsFixingVisitor(fixType, stopWordService);
    }

    private static String unquoteAndFixStopwords(String keyword, ExpressionVisitor<Expression> stopwordsFixingVisitor) {
        if (!(keyword.startsWith("\"") && keyword.endsWith("\""))) {
            return keyword;
        }
        keyword = keyword.substring(1, keyword.length() - 1);
        return fixStopwords(keyword, stopwordsFixingVisitor);
    }

    private static String fixStopwords(String keyword, ExpressionVisitor<Expression> stopwordsFixingVisitor) {
        Expression expression = KeywordParser.buildExpression(keyword);
        expression = expression
                .accept(SplitCompoundWordExpressionTransform.INSTANCE)
                .accept(stopwordsFixingVisitor);
        Keyword keywordWithFixedStopWords = KeywordParser.parseExpression(expression);
        return keywordWithFixedStopWords.toString();
    }

    private static class StopWordsFixingVisitor extends ExpressionStopWordsTransform {
        private StopWordService stopWordService;
        private WordKind fixType;

        StopWordsFixingVisitor(WordKind fixType, StopWordService stopWordService) {
            this.stopWordService = stopWordService;
            this.fixType = fixType;
        }

        @Override
        public Expression visitRawStopWord(Word expr) {
            if (isInsideQuotedWords() || isInsideSquareBrackets()) {
                return expr;
            }
            return new Word(fixType, expr.getText());
        }

        @Override
        public Expression visitWord(Word expr) {
            if (expr.getKind() == WordKind.RAW && stopWordService.isStopWord(expr.getText())) {
                return visitRawStopWord(expr);
            } else {
                return expr;
            }
        }

    }
}
