package ru.yandex.direct.core.entity.keyword.processing;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints;
import ru.yandex.direct.libs.keywordutils.helper.SingleKeywordsCache;
import ru.yandex.direct.libs.keywordutils.inclusion.model.SingleKeywordWithLemmas;
import ru.yandex.direct.libs.keywordutils.model.AnyKeyword;
import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.model.OrderedKeyword;
import ru.yandex.direct.libs.keywordutils.model.SingleKeyword;
import ru.yandex.direct.libs.keywordutils.parser.KeywordParseFunction;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static ru.yandex.direct.libs.keywordutils.inclusion.KeywordInclusionUtils.isFirstIncludedInSecond;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public final class MinusKeywordsDeduplicator {

    private static final Comparator<SingleKeywordWithLemmas> SINGLE_WORD_COMPARATOR =
            new SingleKeywordByFirstLemmaComparator();

    private MinusKeywordsDeduplicator() {
    }

    /**
     * Алгоритм удаления дубликатов:<ol>
     * <li> сортируем слова, находящиеся за пределами квадратных скобок, в порядке,
     * в котором сортируются их первые леммы</li>
     * <li> сортируем последовательности слов в квадратных скобках,
     * в порядке, в котором они сортируются как обычные строки</li>
     * <li> далее сравниваем леммы всех слов (включая те, что в квадратных скобках) в двух минус-фразах последовательно:
     * если леммы каждого слова одной минус-фразы содержат все леммы соответствующих слов
     * второй минус-фразы, то вторая минус-фраза считается дублем и удаляется.</li>
     * </ol>
     * <p>
     * Из пункта 3 следует, что если количество слов во фразах отличается, то они не будут считаться дубликатами.
     * Фразы длиннее {@link MinusPhraseConstraints#WORDS_MAX_COUNT} слов не дедуплицируются
     * <p>
     * Сортировки слов не влияют на выходные минус-фразы.
     *
     * P.S. Для оптимизации, в начале алгоритма для каждой из фраз происходит
     * подбор всех возможных словосочетаний из лемм слов фразы. Это позволяет найти потенциально
     * дублирующиеся фразы по пересечениям таких словосочетаний и не проходить внутренний цикл по всем фразам.
     *
     * @param keywords список валидных минус-фраз
     * @return отсортированный список минус-фраз без "дубликатов"
     */
    public static List<String> removeDuplicatesStr(SingleKeywordsCache keywordsCache,
                                                   List<String> keywords,
                                                   KeywordParseFunction parseFunction) {
        List<Keyword> deduplicated = removeDuplicates(keywordsCache, mapList(keywords, t -> parseFunction.parse(t)));
        return mapList(deduplicated, Keyword::toString);
    }

    /**
     * @param keywords список валидных минус-фраз
     * @return отсортированный список минус-фраз без "дубликатов"
     */
    public static List<Keyword> removeDuplicates(SingleKeywordsCache keywordsCache,
                                                 List<Keyword> keywords) {
        return removeDuplicates(keywordsCache, keywords, true);
    }

    /**
     * Удаляет дубликаты слов внутри фразы (НЕ минус-фразы).
     * Если два слова имеют пересекающиеся леммы, то удаляется "широкое" слово.
     * Например "новый", "нова" -> "новый"
     */
    public static <T extends AnyKeyword> List<T> deduplicateSubKeywords(SingleKeywordsCache keywordsCache,
                                                                        List<T> anyKeywords, Class<T> clazz) {
        List<Keyword> keywords = StreamEx.of(anyKeywords)
                .map(sk -> new Keyword(List.of(sk)))
                .toList();
        List<Keyword> deduplicatedKeywords = removeDuplicates(keywordsCache, keywords, false);
        return StreamEx.of(deduplicatedKeywords)
                .map(k -> k.getAllKeywords().get(0))
                .select(clazz)
                .toList();
    }

    private static List<Keyword> removeDuplicates(SingleKeywordsCache keywordsCache, List<Keyword> keywords,
                                                  boolean removeNestedWord) {
        Int2ObjectOpenHashMap<IntSet> duplicates = getDuplicateIndexes(keywordsCache, keywords, removeNestedWord);
        Set<Integer> duplicatesForRemove = StreamEx.of(duplicates.values()).toFlatCollection(Function.identity(),
                HashSet::new);

        return EntryStream.of(keywords)
                .filterKeyValue((index, kw) -> !duplicatesForRemove.contains(index))
                .values().toList();
    }

    /**
     * @return Мапа: Индекс ключевой фразы в исходном списке ->
     * набор индексов, являющихся дубликатами фразы с индексом ключа.
     * Если у фразы не было дубликатов, то ее не будет в результирующей мапе.
     */
    public static Int2ObjectOpenHashMap<IntSet> getDuplicateIndexes(SingleKeywordsCache keywordsCache,
                                                                 List<Keyword> keywords) {
        return getDuplicateIndexes(keywordsCache, keywords, true);
    }

    /**
     * @param removeNestedWord параметр, отвечащющий на вопрос:
     * какое ключевое слово удалять при неполном совпадении слов.
     * Например, когда одно из слов фиксированное, а другое нет ('машина' и '!машина')
     * или слова имеют пересекающиеся леммы ('ухо' и 'уха').
     * Если removeNestedWord=true, то удаляется слово, которое "поглощается" другим словом
     * (в примерах это '!машина' и 'ухо').
     * Если removeNestedWord=false, то удаляется слово, которое "поглощает" другое слово
     * (в примерах это 'машина' и 'уха').
     *
     * @return Мапа: Индекс ключевой фразы в исходном списке ->
     * набор индексов, являющихся дубликатами фразы с индексом ключа.
     * Если у фразы не было дубликатов, то ее не будет в результирующей мапе.
     */
    private static Int2ObjectOpenHashMap<IntSet> getDuplicateIndexes(SingleKeywordsCache singleKeywordsCache,
                                                                     List<Keyword> keywords,
                                                                     boolean removeNestedWord) {
        try (TraceProfile profile = Trace.current().profile("minus_keywords:remove_duplicates", "", keywords.size())) {
            List<KeywordSortingWrapper> keywordWrappers = new ArrayList<>();
            Map<KeywordSortingWrapper, IntSet> keywordWrapperCombinationHashes = new HashMap<>();
            Int2ObjectOpenHashMap<IntArrayList> combinationHashKeywordIndexes = new Int2ObjectOpenHashMap<>();
            for (int i = 0; i < keywords.size(); i++) {
                var keywordWrapper = new KeywordSortingWrapper(keywords.get(i), singleKeywordsCache);
                keywordWrappers.add(keywordWrapper);
                if (keywordWrapper.singleList(singleKeywordsCache).size() <= MinusPhraseConstraints.WORDS_MAX_COUNT) {
                    IntSet keywordCombinationHashes = keywordWrapperCombinationHashes.computeIfAbsent(keywordWrapper,
                            t -> t.combineAllLemmas(singleKeywordsCache));
                    for (int keywordCombinationHash : keywordCombinationHashes) {
                        IntArrayList indexes = combinationHashKeywordIndexes.computeIfAbsent(keywordCombinationHash,
                                h -> new IntArrayList());
                        indexes.add(i);
                    }
                }
            }

            Int2ObjectOpenHashMap<IntSet> duplicates = new Int2ObjectOpenHashMap<>();
            IntSet duplicatesForRemove = new IntOpenHashSet();
            for (int i = 0; i < keywordWrappers.size(); i++) {
                if (duplicatesForRemove.contains(i)) {
                    continue;
                }
                KeywordSortingWrapper keywordWrapper1 = keywordWrappers.get(i);
                IntSet keywordCombinationHashes = keywordWrapperCombinationHashes.get(keywordWrapper1);
                if (keywordCombinationHashes == null) {
                    continue;
                }
                for (int keywordCombinationHash : keywordCombinationHashes) {
                    IntArrayList potentialKeywordIndexes = combinationHashKeywordIndexes.get(keywordCombinationHash);
                    for (int j : potentialKeywordIndexes) {
                        KeywordSortingWrapper keywordWrapper2 = keywordWrappers.get(j);
                        if (keywordWrapper1 == keywordWrapper2 || duplicatesForRemove.contains(j)) {
                            continue;
                        }

                        if (removeNestedWord && keywordWrapper2.isDuplicatedBy(keywordWrapper1, singleKeywordsCache)
                                || !removeNestedWord && keywordWrapper1.isDuplicatedBy(keywordWrapper2, singleKeywordsCache)) {
                            duplicatesForRemove.add(j);
                            duplicates.computeIfAbsent(i, x -> new IntOpenHashSet()).add(j);
                        }
                    }
                }
            }
            Int2ObjectOpenHashMap<IntSet> result = new Int2ObjectOpenHashMap<>();
            IntSet rootsOfDuplicates = new IntOpenHashSet(duplicates.keySet());
            rootsOfDuplicates.removeAll(duplicatesForRemove);
            for (int root : rootsOfDuplicates) {
                IntSet rootDuplicates = new IntOpenHashSet();
                collectDuplicates(root, duplicates, rootDuplicates);
                result.put(root, rootDuplicates);
            }
            return result;
        }
    }

    /**
     * Рекурсивно перебирает все исходящие из узла {@code root} ребра дерева {@code duplicatesTree}
     * с индексами дубликатов ключевых фраз, находящихся в вершинах дерева,
     * и собирает их в один набор {@code resultDuplicates}.<p>
     *
     * Например {@code duplicatesTree = {5 -> [4, 2], 4 -> [3], 2 -> [1, 0]}},<p>
     * {@code root=5},<p>
     * тогда {@code resultDuplicates = [4, 2, 3, 1, 0]}
     * @param root индекс ключевой фразы, для которого собираем дубликаты
     * @param duplicatesTree дерево дубликатов: где узел - индекс ключевой фразы, а потомки - индексы его дубликатов
     * @param resultDuplicates результирующий набор индексов дубликатов ключевой фразы с индексом {@code root}
     */
    private static void collectDuplicates(int root, Int2ObjectOpenHashMap<IntSet> duplicatesTree,
                                          IntSet resultDuplicates) {
        IntSet children = duplicatesTree.get(root);
        if (children == null) {
            return;
        }
        for (int child : children) {
            resultDuplicates.add(child);
            collectDuplicates(child, duplicatesTree, resultDuplicates);
        }
    }

    /**
     * Удалить removeKeywords из existingKeywords
     * @param existingKeywords существующий набор слов
     * @param removeKeywords набор на удаление
     * @return
     */
    public static List<String> removeStr(SingleKeywordsCache singleKeywordsCache,
                                         List<String> existingKeywords,
                                         List<String> removeKeywords,
                                         KeywordParseFunction parseFunction) {
        List<Keyword> deduplicated = remove(singleKeywordsCache, mapList(existingKeywords, t -> parseFunction.parse(t)),
                mapList(removeKeywords, t -> parseFunction.parse(t)));
        return mapList(deduplicated, Keyword::toString);
    }

    public static List<Keyword> remove(SingleKeywordsCache singleKeywordsCache, List<Keyword> existingKeywords,
                                       List<Keyword> removeKeywords) {
        try (TraceProfile profile = Trace.current().profile("minus_keywords:remove", "", existingKeywords.size())) {
            List<KeywordSortingWrapper> existingKeywordWrappers =
                    mapList(existingKeywords, keyword -> new KeywordSortingWrapper(keyword, singleKeywordsCache));
            List<KeywordSortingWrapper> removeKeywordWrappers =
                    mapList(removeKeywords, keyword -> new KeywordSortingWrapper(keyword, singleKeywordsCache));

            for (int i = 0; i < existingKeywordWrappers.size(); i++) {
                KeywordSortingWrapper keywordWrapper1 = existingKeywordWrappers.get(i);

                for (KeywordSortingWrapper keywordWrapper2 : removeKeywordWrappers) {
                    if (keywordWrapper1.isDuplicatedBy(keywordWrapper2, singleKeywordsCache)) {
                        existingKeywordWrappers.remove(i);
                        i--;
                        break;
                    }
                }
            }

            return mapList(existingKeywordWrappers, kww -> kww.keyword);
        }
    }

    private static class KeywordSortingWrapper {
        private final Keyword keyword;
        private List<SingleKeywordWithLemmas> sortedWords;
        private List<OrderedKeyword> sortedWordSequences;
        @Nullable
        private List<SingleKeywordWithLemmas> singleList;

        KeywordSortingWrapper(Keyword keyword, SingleKeywordsCache singleKeywordsCache) {
            this.keyword = keyword;

            sortedWords = StreamEx.of(keyword.getAllKeywords())
                    .select(SingleKeyword.class)
                    .flatCollection(kw -> singleKeywordsCache.singleKeywordsFrom(kw.getWord()))
                    .sorted(SINGLE_WORD_COMPARATOR)
                    .toList();

            sortedWordSequences = StreamEx.of(keyword.getAllKeywords())
                    .select(OrderedKeyword.class)
                    .sorted()
                    .toList();
        }

        @Nonnull
        private List<SingleKeywordWithLemmas> singleList(SingleKeywordsCache singleKeywordsCache) {
            if (singleList == null) {
                singleList = StreamEx.of(sortedWordSequences)
                        .flatCollection(OrderedKeyword::getSingleKeywords)
                        .flatCollection(kw -> singleKeywordsCache.singleKeywordsFrom(kw.getWord()))
                        .append(sortedWords)
                        .toList();
            }
            return singleList;
        }

        private boolean isDuplicatedBy(KeywordSortingWrapper otherKeywordWrapper,
                                       SingleKeywordsCache singleKeywordsCache) {
            // если другая ключевая фраза - точный набор слов, а данная - нет,
            // то другая точно не включает данную (данная не может быть дубликатом другой)
            if (otherKeywordWrapper.keyword.isQuoted() && !keyword.isQuoted()) {
                return false;
            }

            // для начала проверяем размеры - самый дешевый путь
            if (otherKeywordWrapper.sortedWords.size() != sortedWords.size()) {
                return false;
            }
            if (otherKeywordWrapper.sortedWordSequences.size() != sortedWordSequences.size()) {
                return false;
            }
            for (int i = 0; i < sortedWordSequences.size(); i++) {
                List<SingleKeyword> otherWordSequence =
                        otherKeywordWrapper.sortedWordSequences.get(i).getSingleKeywords();
                List<SingleKeyword> thisWordSequence = sortedWordSequences.get(i).getSingleKeywords();
                if (otherWordSequence.size() != thisWordSequence.size()) {
                    return false;
                }
            }

            // все размеры сходятся, дальше проверяем вхождение лемм
            List<SingleKeywordWithLemmas> allOtherWords = otherKeywordWrapper.singleList(singleKeywordsCache);
            List<SingleKeywordWithLemmas> allThisWords = this.singleList(singleKeywordsCache);

            for (int i = 0; i < allOtherWords.size(); i++) {
                SingleKeywordWithLemmas otherWord = allOtherWords.get(i);
                SingleKeywordWithLemmas thisWord = allThisWords.get(i);
                if (!isFirstIncludedInSecond(thisWord, otherWord)) {
                    return false;
                }
            }

            return true;
        }

        private IntSet combineAllLemmas(SingleKeywordsCache singleKeywordsCache) {
            return KeywordCombinator.combineAllLemmas(singleList(singleKeywordsCache));
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            KeywordSortingWrapper that = (KeywordSortingWrapper) o;
            return Objects.equals(keyword, that.keyword);
        }

        @Override
        public int hashCode() {
            return Objects.hash(keyword);
        }
    }
}
