package ru.yandex.direct.core.entity.keyword.processing;

import java.util.List;
import java.util.Objects;

import ru.yandex.direct.libs.keywordutils.model.AnyKeyword;
import ru.yandex.direct.libs.keywordutils.model.Keyword;

import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Контейнер для хранения ключевой фразы (без минус-слов) в исходном и нормализованном виде.
 */
public class NormalizedKeyword {
    /**
     * Исходная ключевая фраза.
     */
    private Keyword originalKeyword;
    /**
     * Список (отсортированный, без дубликатов) ключевых слов. Каждое слово хранится в исходном и нормализованном виде.
     */
    private List<NormalizedWord<AnyKeyword>> allKeywords;

    public NormalizedKeyword(List<NormalizedWord<AnyKeyword>> allKeywords) {
        this(null, allKeywords);
    }

    public NormalizedKeyword(Keyword originalKeyword, List<NormalizedWord<AnyKeyword>> allKeywords) {
        this.originalKeyword = originalKeyword;
        this.allKeywords = allKeywords;
    }

    public boolean isQuoted() {
        return originalKeyword != null && originalKeyword.isQuoted();
    }

    public static NormalizedKeyword from(Keyword normalizedKeyword) {
        return new NormalizedKeyword(mapList(normalizedKeyword.getAllKeywords(), NormalizedWord::new));
    }

    /**
     * Возвращает ключевую фразу в исходной форме. Если исходная форма неизвестна - возвращает нормализованную форму.
     *
     * @return ключевая фраза в исходной форме
     */
    public Keyword getOriginal() {
        return nvl(originalKeyword, getNormalized());
    }

    /**
     * Возвращает ключевую фразу в нормализованной форме.
     *
     * @return ключевая фраза в нормализованной форме
     */
    public Keyword getNormalized() {
        return new Keyword(isQuoted(), mapList(allKeywords, NormalizedWord::getNormalizedWord));
    }

    /**
     * @return ключевая фраза, состоящая из слов без дубликатов в исходной форме
     */
    public Keyword getDeduplicatedOriginal() {
        return new Keyword(isQuoted(), mapList(allKeywords, NormalizedWord::getOriginalWord));
    }

    /**
     * Возвращает список (отсортированный, без дубликатов) ключевых слов.
     *
     * @return список ключевых слов
     */
    public List<NormalizedWord<AnyKeyword>> getAllKeywords() {
        return allKeywords;
    }

    @Override
    public String toString() {
        return getNormalized().toString();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        NormalizedKeyword keyword = (NormalizedKeyword) o;
        return isQuoted() == keyword.isQuoted() &&
                Objects.equals(getNormalized(), keyword.getNormalized());
    }

    @Override
    public int hashCode() {
        return Objects.hash(isQuoted(), getNormalized());
    }
}
