package ru.yandex.direct.core.entity.keyword.processing;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;
import ru.yandex.direct.libs.keywordutils.model.SingleKeyword;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Контейнер для хранения ключевой фразы c минус-словами в исходном и нормализованном виде.
 */
public class NormalizedKeywordWithMinuses {
    /**
     * Ключевая фраза. Хранится в исходном и нормализованном виде.
     */
    private NormalizedKeyword keyword;
    /**
     * Список минус-слов. Каждое минус-слово хранится в исходном и нормализованном виде. Список отсортирован после
     * нормализации, но может потерять это свойство после расклейки.
     */
    private List<NormalizedWord<SingleKeyword>> minusWords;

    public NormalizedKeywordWithMinuses(NormalizedKeyword keyword, List<NormalizedWord<SingleKeyword>> minusWords) {
        this.keyword = keyword;
        this.minusWords = minusWords;
    }

    public NormalizedKeyword getKeyword() {
        return keyword;
    }

    public List<NormalizedWord<SingleKeyword>> getMinusWords() {
        return minusWords;
    }

    /**
     * Возвращает список минус-слов (отсортированный, без дубликатов, каждое минус-слово в исходном виде).
     *
     * @return список минус-слов
     */
    public List<SingleKeyword> getOriginalMinusWords() {
        return mapList(minusWords, NormalizedWord::getOriginalWord);
    }

    /**
     * Добавляет новые минус-слова после расклейки. После этого список минус-слов может оказаться неотсортированным.
     *
     * @param addedMinusWords список добавленных минус-слов
     * @return {@link NormalizedKeywordWithMinuses} с новыми минус-словами
     */
    public NormalizedKeywordWithMinuses appendMinuses(List<NormalizedWord<SingleKeyword>> addedMinusWords) {
        List<NormalizedWord<SingleKeyword>> newMinusWords = new ArrayList<>(minusWords);
        newMinusWords.addAll(addedMinusWords);
        return new NormalizedKeywordWithMinuses(keyword, newMinusWords);
    }

    public static NormalizedKeywordWithMinuses from(KeywordWithMinuses keywordWithMinuses) {
        return new NormalizedKeywordWithMinuses(
                new NormalizedKeyword(keywordWithMinuses.getKeyword(),
                        mapList(keywordWithMinuses.getKeyword().getAllKeywords(), NormalizedWord::new)),
                mapList(keywordWithMinuses.getMinusKeywords(),
                        mw -> new NormalizedWord<>((SingleKeyword) mw.getAllKeywords().get(0))));
    }

    /**
     * Возвращает объект {@link KeywordWithMinuses}, содержащий нормализованное ключевое слово с нормализованными
     * минус-словами.
     *
     * @return объект {@link KeywordWithMinuses}
     */
    public KeywordWithMinuses toKeywordWithMinuses() {
        return new KeywordWithMinuses(getKeyword().getNormalized(), mapList(getMinusWords(),
                nmw -> new Keyword(singletonList(nmw.getNormalizedWord()))));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        NormalizedKeywordWithMinuses that = (NormalizedKeywordWithMinuses) o;
        return Objects.equals(keyword, that.keyword) &&
                Objects.equals(minusWords, that.minusWords);
    }

    @Override
    public int hashCode() {
        return Objects.hash(keyword, minusWords);
    }

    @Override
    public String toString() {
        return toKeywordWithMinuses().toString();
    }
}
