package ru.yandex.direct.core.entity.keyword.processing.deduplication;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import static ru.yandex.direct.core.entity.keyword.processing.deduplication.KeywordDeduplicationUtils.checkDuplication;

public class AdGroupKeywordDeduplicationUtils {

    private AdGroupKeywordDeduplicationUtils() {
    }

    /**
     * Ищет дубликаты внутри списка модифицируемых фраз (добавляемых или обновляемых).
     * <p>
     * Возвращает мультимапу, ключами которой являются индексы продублированных фраз,
     * а значениями - все индексы их дубликатов. При этом в возвращаемой
     * мультимапе в качестве ключей присутствуют индексы всех продублированных фраз.
     * <p>
     * !!! Слова в передаваемых фразах должны быть нормализованы и сортированы.
     *
     * @param containers список модифицируемых фраз (добавляемых или обновляемых).
     * @return мультимапа, ключами которой являются индексы продублированных фраз,
     * а значениями - все индексы их дубликатов. При этом в возвращаемой
     * мультимапе в качестве ключей присутствуют индексы всех продублированных фраз.
     */
    public static Multimap<Integer, Integer> computeDuplicates(List<DuplicatingContainer> containers) {
        Map<Integer, List<DuplicatingContainer>> adGroupIndexToContainers = groupByAdGroupIndex(containers);

        Multimap<Integer, Integer> duplicates = HashMultimap.create();

        adGroupIndexToContainers.forEach((adGroupIndex, containersOfAdGroup) -> {
            containersOfAdGroup.forEach(container -> {
                List<Integer> duplicateIndexes = getDuplicateIndexes(container, containersOfAdGroup);
                duplicates.putAll(container.getIndex(), duplicateIndexes);
            });
        });
        return duplicates;
    }

    /**
     * Ищет дубликаты между списками существующих фраз и модифицируемых (добавляемых или обновляемых).
     * При добавлении в мапе существующих фраз должны быть все фразы,
     * а при обновлении только те, которые не изменяются запросом.
     * <p>
     * Возвращается мапа, ключами которой являются индексы модифицируемых фраз,
     * а значениями - индексы первого найденного дубликата среди существующих фраз.
     * <p>
     * !!! Слова в передаваемых фразах должны быть нормализованы и сортированы.
     *
     * @param newContainers      список новых фраз (добавляемых или обновляемых).
     * @param existingContainers список существующих фраз;
     *                           при добавлении в этом списке должны быть все фразы,
     *                           а при обновлении только те, которые не изменяются запросом.
     * @return мапа, ключами которой являются индексы модифицируемых фраз,
     * а значениями - индексы первого найденного дубликата среди существующих фраз.
     */
    public static Map<Integer, Integer> computeDuplicates(
            List<DuplicatingContainer> newContainers,
            List<DuplicatingContainer> existingContainers) {
        Map<Integer, List<DuplicatingContainer>> adGroupIndexToNewContainers = groupByAdGroupIndex(newContainers);
        Map<Integer, List<DuplicatingContainer>> adGroupIndexToExistingContainers = groupByAdGroupIndex(existingContainers);

        Map<Integer, Integer> duplicates = new HashMap<>();

        EntryStream.of(adGroupIndexToNewContainers)
                .filterKeys(adGroupIndexToExistingContainers::containsKey)
                .forKeyValue((adGroupIndex, newContainersOfAdGroup) -> {
                    List<DuplicatingContainer> existingContainersOfAdGroup =
                            adGroupIndexToExistingContainers.get(adGroupIndex);
                    newContainersOfAdGroup.forEach(newContainer -> {
                        DuplicatingContainer existingDuplicate =
                                getFirstDuplicateKeyword(newContainer, existingContainersOfAdGroup);
                        if (existingDuplicate != null) {
                            duplicates.put(newContainer.getIndex(), existingDuplicate.getIndex());
                        }
                    });
                });
        return duplicates;
    }

    private static Map<Integer, List<DuplicatingContainer>> groupByAdGroupIndex(
            List<DuplicatingContainer> containers) {
        return StreamEx.of(containers).groupingBy(DuplicatingContainer::getAdGroupIndex);
    }

    private static List<Integer> getDuplicateIndexes(DuplicatingContainer container,
                                                     Collection<DuplicatingContainer> containers) {
        return StreamEx.of(containers)
                .filter(c -> container.getAutotargeting() == c.getAutotargeting())
                .filter(t -> !container.getIndex().equals(t.getIndex())
                        && checkDuplication(container.getNormalizedKeywordWithMinuses(),
                        t.getNormalizedKeywordWithMinuses()))
                .map(DuplicatingContainer::getIndex)
                .toList();
    }

    private static DuplicatingContainer getFirstDuplicateKeyword(DuplicatingContainer container,
                                                                 Collection<DuplicatingContainer> containers) {
        return containers.stream()
                .filter(c -> container.getAutotargeting() == c.getAutotargeting())
                .filter(c -> checkDuplication(
                        container.getNormalizedKeywordWithMinuses(), c.getNormalizedKeywordWithMinuses()))
                .findFirst()
                .orElse(null);
    }
}
