package ru.yandex.direct.core.entity.keyword.processing.glue;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.keyword.processing.NormalizedKeyword;
import ru.yandex.direct.core.entity.keyword.processing.NormalizedKeywordWithMinuses;
import ru.yandex.direct.core.entity.keyword.processing.NormalizedWord;
import ru.yandex.direct.libs.keywordutils.helper.SingleKeywordsCache;
import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.model.SingleKeyword;

import static ru.yandex.direct.core.entity.keyword.processing.MinusKeywordsDeduplicator.getDuplicateIndexes;

@ParametersAreNonnullByDefault
public class KeywordGluer {
    private KeywordGluer() {
        // no instantiation
    }

    /**
     * Склеивает ключевые фразы по следующему принципу: если у двух или более фраз полностью совпадают ключевые слова
     * (но не обязательно минус-слова), то эти фразы склеиваются в одну, состояющую из общих ключевых слов, и
     * объединенных минус-слов из всех склеенных фраз.
     *
     * @param normalizedKeywordWithMinuses список ключевых фраз для склейки
     * @return список склеенных ключевых фраз
     */
    public static List<NormalizedKeywordWithMinuses> glueKeywords(
            SingleKeywordsCache keywordsCache,
            List<NormalizedKeywordWithMinuses> normalizedKeywordWithMinuses) {
        List<Keyword> originalKeywordsForDeduplicate = StreamEx.of(normalizedKeywordWithMinuses)
                .map(NormalizedKeywordWithMinuses::getKeyword)
                .map(NormalizedKeyword::getDeduplicatedOriginal)
                .toList();
        Int2ObjectOpenHashMap<IntSet> duplicates = getDuplicateIndexes(keywordsCache, originalKeywordsForDeduplicate);

        List<NormalizedKeyword> result = StreamEx.of(normalizedKeywordWithMinuses)
                .map(NormalizedKeywordWithMinuses::getKeyword)
                .toList();
        Map<NormalizedKeyword, Set<NormalizedWord<SingleKeyword>>> gluedParts = new HashMap<>();
        for (int thisKeywordIndex = 0; thisKeywordIndex < normalizedKeywordWithMinuses.size(); thisKeywordIndex++) {
            NormalizedKeywordWithMinuses thisKeywordWithMinuses = normalizedKeywordWithMinuses.get(thisKeywordIndex);
            NormalizedKeyword thisKeyword = thisKeywordWithMinuses.getKeyword();
            Set<NormalizedWord<SingleKeyword>> minusWords =
                    gluedParts.computeIfAbsent(thisKeyword, t -> new TreeSet<>());
            minusWords.addAll(thisKeywordWithMinuses.getMinusWords());
            for (int otherKeywordIndex : duplicates.getOrDefault(thisKeywordIndex, new IntOpenHashSet())) {
                NormalizedKeywordWithMinuses otherKeywordWithMinuses =
                        normalizedKeywordWithMinuses.get(otherKeywordIndex);
                result.set(otherKeywordIndex, thisKeyword);
                minusWords.addAll(otherKeywordWithMinuses.getMinusWords());
            }
        }
        return StreamEx.of(result)
                .map(nkw -> new NormalizedKeywordWithMinuses(nkw, new ArrayList<>(gluedParts.get(nkw))))
                .toList();
    }
}
