package ru.yandex.direct.core.entity.keyword.processing.unglue;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.advq.query.ast.Word;
import ru.yandex.advq.query.ast.WordKind;
import ru.yandex.direct.core.entity.keyword.processing.KeywordCombinator;
import ru.yandex.direct.core.entity.keyword.processing.KeywordProcessingUtils;
import ru.yandex.direct.core.entity.keyword.processing.NormalizedWord;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.libs.keywordutils.helper.SingleKeywordsCache;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;
import ru.yandex.direct.libs.keywordutils.inclusion.model.SingleKeywordWithLemmas;
import ru.yandex.direct.libs.keywordutils.model.AnyKeyword;
import ru.yandex.direct.libs.keywordutils.model.OrderedKeyword;
import ru.yandex.direct.libs.keywordutils.model.SingleKeyword;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.libs.keywordutils.inclusion.KeywordInclusionUtils.isFirstIncludedInSecond;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Расклейщик фраз.
 * <p>
 * Некоторые ключевые фразы могут иметь пересекающиеся множества поисковых запросов.
 * В этом случае они будут конфликтовать между собой на этой общей части запросов.
 * В такой ситуации выбор фразы и соответствующей ставки, по которой будет показано объявление,
 * не зависит от пользователя.
 * <p>
 * Пример:
 * Ключевые фразы: слон, конь.
 * Поисковые запросы, на которых фразы не конфликтуют: розовый слон, быстрый конь.
 * Поисковые запросы, на которых фразы конфликтуют: слон быстрый как конь, слон конь основные отличия.
 * <p>
 * В некоторых случаях множество поисковых запросов одной фразы может полностью включать в себя
 * множество поисковых запросов второй фразы. Такие фразы называются склеенными.
 * В этом случае можно отделить меньшее множество от большего: уточнить фразу, определяющую
 * большее множество, таким образом, чтобы из этого множества вычесть меньшее множество,
 * определенное второй фразой. Это можно сделать достаточно интуитивно в простом случае,
 * когда длины фраз отличаются на 1 слово, и при этом более длинная фраза поглощает короткую.
 * В этом случае короткой фразе добавляется то слово, на которое они отличаются, в качестве минус-слова.
 * Пример:
 * Ключевые фразы: слон, слон купить.
 * Поисковые запросы первой фразы: слон, большой слон, розовый слон, купить слона, купить большого слона.
 * Поисковые запросы второй фразы: купить слона, купить большого слона.
 * <p>
 * Фразы конфликтуют на всех поисковых запросах второй фразы, так как все они входят в множество
 * поисковых запросов первой фразы. Если же добавить к первой фразе минус-слово /купить/,
 * то конкуренция между этими двумя фразами будет полностью исключена.
 * <p>
 * В целом расклейка не обязательна и имеет очевидные ограничения. Например, когда фразы отличаются
 * более чем на одно слово, расклейка не производится, так как для этого потребовалось бы добавлять
 * дополнительные ключевые фразы. Расклейку делаем в тех случаях, когда выполняется интуитивно
 * для пользователя и не имеет неприятных побочных последствий.
 * <p>
 * Правила расклейки:<ul>
 * <li>фразы расклеиваются попарно</li>
 * <li>при расклейке двух фраз более короткой добавляется одно минус-слово</li>
 * <li>минус-слово добавляется ровно в той форме, в которой присутствует в более длинной фразе</li>
 * <li>зафиксированные стоп-слова добавляются в минус-слова всегда с оператором "!"</li>
 * </ul>
 * <p>
 * Правила применимости расклейки:<ul>
 * <li>ни одна из фраз не заключена в кавычки (флаг quoted)</li>
 * <li>количество одиночных слов отличается на 1</li>
 * <li>количество групп слов (в квадратных скобках) совпадает (может быть равно 0),
 * при этом единичные слова в квадратных скобках должны учитываться, как слова вне квадратных скобок</li>
 * <li>одиночные слова длинной фразы содержат все одиночные слова короткой фразы, при этом эквивалентность
 * слов проверяется просто по equals (с учетом операторов "!" и "+"), так как фраза приходит в
 * нормализованном виде</li>
 * <li>группы слов (в квадратных скобках) в обеих фразах попарно эквивалентны (приходят отсортированными),
 * эквивалентность групп определяется попарной эквивалентностью входящих в них одиночных слов в исходном
 * порядке; эквивалентность отдельных слов определяется так же, как и для одиночных слов, не входящих в группы;
 * по сути отсортированные группы слов можно проверять по equals в строковом представлении</li>
 * <li>минус-слова с оператором "!" сравниваются между собой по исходной форме, поэтому короткая фраза
 * может содержать фиксированные слова в разных формах (/-!слона -!слонов/); при этом в остальных случаях сравнение
 * делается по первой лемме; это сделано для того, чтобы исключить случаи, когда множества ключевых слов, заданных
 * минус-словами, пересекаются (/-слон -!слона/ или /-слон -слона/)</li>
 * </ul>
 */
@Service
public class KeywordUngluer {

    private final KeywordWithLemmasFactory keywordFactory;
    private final StopWordService stopWordService;
    private final SingleKeywordsCache singleKeywordsCache;

    public KeywordUngluer(KeywordWithLemmasFactory keywordFactory, StopWordService stopWordService,
                          SingleKeywordsCache singleKeywordsCache) {
        this.keywordFactory = keywordFactory;
        this.stopWordService = stopWordService;
        this.singleKeywordsCache = singleKeywordsCache;
    }

    /**
     * Дает ответ, какие фразы и каким образом должны быть расклеены.
     * Ключевые слова внутри фраз должны быть нормализованы и отсортированы одинаковым образом.
     * <p>
     * В рамках групп объявлений определяет, каким новым фразам и каким существующим необходимо
     * добавить минус-слово для исключения конкуренции в тех случаях, когда это возможно.
     * Минус-слова могут быть добавлены к существующим фразам из-за конкуренции с добавляемыми,
     * а могут быть добавлены к новым фразам из-за конкуренции с другими новыми фразами или существующими.
     * <p>
     * Фразы могут быть расклеены между собой только в рамках одной группы объявлений,
     * для этого каждая фраза подается на вход вместе с {@code adGroupIndex}.
     * <p>
     * Возвращаемый результат имеет "рекомендательный" характер, и может быть применен частично
     * с учетом ограничения на максимальную длину фраз или с учетом того, что полное
     * применение может привести к образованию новых дубликатов. И к новым и к существующим фразам
     * можно применять любые из перечисленных в результате минус-слов или все вместе.
     * <p>
     * В ходе алгоритма осуществляется проход по списку добавляемых слов
     * Факт того, что из данного newKeyword можно получить
     * другое новое слово otherNewKeyword добавлением одного элемента в plusWords, равносилен тому,
     * что у одной из "вырезанных" версий otherNewKeyword {@link #getKeywordInfoIndexMap} и у newKeyword совпадает
     * одна из соответствующих им хэшей {@link #createCombinedInfoHashes}.
     * <p>
     * Поэтому используется MultiMap newCutKeywordsInfoMapping, хранящий в качестве ключей
     * хеши {@link #createCombinedInfoHashes}, а в качестве значений - соответствующие им индексы
     * {@link KeywordForUnglue} в массиве newKeywordsContainers.
     * Обращение к данному мапу позволяет за О(1) по данному newKeyword найти все интересующие otherNewKeyword
     * Другие варианты "расклейки" между новыми и имеющимися словами обрабатываются аналогично с помощью других мультимапов
     *
     * @param newKeywordsContainers      список контейнеров с новыми ключевыми фразами c минус-словами.
     * @param existingKeywordsContainers список контейнеров с существующими ключевыми фразами с минус-словами.
     *                                   В случае добавления список должен содержать все существующие
     *                                   фразы в затронутых группах, а в случае обновления - все
     *                                   существующие фразы в затронутых группах, которые не затронуты
     *                                   самим обновлением.
     * @return список результатов для входного списка {@code newKeywordsContainers}.
     * Результаты возвращаются только для тех элементов, которые были расклеены,
     * то есть в общем случае размер выходного списка меньше или равен размеру входного.
     */
    public List<UnglueResult> unglue(List<UnglueContainer> newKeywordsContainers,
                                     List<UnglueContainer> existingKeywordsContainers) {
        try (TraceProfile profile = Trace.current().profile("keywords:unglue")) {
            checkArguments(newKeywordsContainers, existingKeywordsContainers);
            List<UnglueResult> results = new ArrayList<>();
            List<KeywordForUnglue> newKeywords = mapList(newKeywordsContainers,
                    uc -> KeywordForUnglue.build(stopWordService, keywordFactory, singleKeywordsCache, uc));
            List<KeywordForUnglue> existingKeywords = mapList(existingKeywordsContainers,
                    uc -> KeywordForUnglue.build(stopWordService, keywordFactory, singleKeywordsCache, uc));

            Set<Integer> keywordSizes = StreamEx.of(newKeywords)
                .map(keyword -> keyword.getPlusWords().size()).distinct()
                .toSet();
            keywordSizes.addAll(StreamEx.of(existingKeywords)
                    .map(keyword -> keyword.getPlusWords().size()).distinct()
                    .toSet());
            Int2ObjectOpenHashMap<IntArrayList> newCutKeywordsInfoMapping =
                    getKeywordInfoIndexMap(newKeywords, keywordSizes,true);
            Int2ObjectOpenHashMap<IntArrayList> existingCutKeywordsInfoMapping =
                    getKeywordInfoIndexMap(existingKeywords, keywordSizes,true);
            Int2ObjectOpenHashMap<IntArrayList> existingKeywordsInfoMapping =
                    getKeywordInfoIndexMap(existingKeywords, keywordSizes, false);

            for (KeywordForUnglue newKeyword : newKeywords) {
                if (newKeyword.isQuoted()) {
                    continue;
                }

                List<NormalizedWord<SingleKeyword>> addedMinusWords = new ArrayList<>();
                Map<Integer, NormalizedWord<SingleKeyword>> addedMinusWordsToExisting = new HashMap<>();
                IntSet keywordInfoHashes = createCombinedInfoHashes(newKeyword);
                for (int currentKeywordInfoHash : keywordInfoHashes) {
                    // сравниваем с другими новыми фразами (пытаемся добавить минус слово в текущую)
                    IntArrayList otherNewKeywordIndexes =
                            newCutKeywordsInfoMapping.getOrDefault(currentKeywordInfoHash, new IntArrayList());
                    for (int otherNewKeywordIndex : otherNewKeywordIndexes) {
                        KeywordForUnglue otherNewKeyword = newKeywords.get(otherNewKeywordIndex);
                        if (newKeyword == otherNewKeyword) {
                            continue;
                        }
                        NormalizedWord<SingleKeyword> normalizedMinusWord = tryToAddMinusWord(newKeyword, otherNewKeyword);
                        if (normalizedMinusWord != null) {
                            addedMinusWords.add(normalizedMinusWord);
                        }
                    }

                    // сравниваем с существующими фразами (пытаемся добавить минус слово в текущую)
                    IntArrayList existingKeywordIndexes =
                            existingCutKeywordsInfoMapping.getOrDefault(currentKeywordInfoHash, new IntArrayList());
                    for (int existingKeywordIndex : existingKeywordIndexes) {
                        KeywordForUnglue existingKeyword = existingKeywords.get(existingKeywordIndex);
                        NormalizedWord<SingleKeyword> minusWord = tryToAddMinusWord(newKeyword, existingKeyword);
                        if (minusWord != null) {
                            addedMinusWords.add(minusWord);
                        }
                    }
                }

                // сравниваем с существующими фразами (пытаемся добавить минус слово в существующую)
                if (!existingKeywordsInfoMapping.isEmpty()) {
                    for (int keywordToCutIndex = 0;
                         keywordToCutIndex < newKeyword.getPlusWords().size();
                         ++keywordToCutIndex) {
                        IntSet cutKeywordInfoHashes = createCombinedInfoHashes(newKeyword, keywordToCutIndex);
                        for (int currentCutKeywordInfoHash : cutKeywordInfoHashes) {
                            IntArrayList existingKeywordIndexes = existingKeywordsInfoMapping.getOrDefault(
                                    currentCutKeywordInfoHash, new IntArrayList());
                            for (int existingKeywordIndex : existingKeywordIndexes) {
                                KeywordForUnglue existingKeyword = existingKeywords.get(existingKeywordIndex);
                                NormalizedWord<SingleKeyword> minusWord = tryToAddMinusWord(existingKeyword, newKeyword);
                                if (minusWord != null) {
                                    addedMinusWordsToExisting.putIfAbsent(existingKeyword.getIndex(), minusWord);
                                }
                            }
                        }
                    }
                }

                if (!addedMinusWords.isEmpty() || !addedMinusWordsToExisting.isEmpty()) {
                    UnglueResult result = new UnglueResult(newKeyword.index, addedMinusWords, addedMinusWordsToExisting);
                    results.add(result);
                }

            }
            return results;
        }
    }

    private void checkArguments(List<UnglueContainer> newKeywordsContainers,
                                List<UnglueContainer> existingKeywordsContainers) {
        Set<Integer> newKeywordContainersIndexes =
                listToSet(newKeywordsContainers, UnglueContainer::getIndex);
        Set<Integer> existingKeywordContainersIndexes =
                listToSet(existingKeywordsContainers, UnglueContainer::getIndex);
        checkArgument(newKeywordContainersIndexes.size() == newKeywordsContainers.size(),
                "unglue containers for new keywords contain non-unique indexes");
        checkArgument(existingKeywordContainersIndexes.size() == existingKeywordsContainers.size(),
                "unglue containers for existing keywords contain non-unique indexes");
    }

    /**
     * Формирует маппинг хэша из набора
     * ({@link KeywordForUnglue#getAdGroupIndex()}, {@link KeywordForUnglue#getOrderedKeywords()},
     * хэш комбинации лемм {@link KeywordForUnglue#getPlusWordsWithLemmas()})
     * в индексы экземпляров KeywordForUnglue подаваемого на вход массива.
     * Можно "нарезать" слова: вместо данного keyword класть в мапу инфу про него само или plusWords.size() экземпляторов,
     * отличающихся от исходного выкидыванием одного элемента из plusWords
     *
     * @param keywordList   подаваемый на вход массив {@link KeywordForUnglue}
     * @param keywordSizes  набор из количеств слов в новых и существующих фразах
     * @param toCut         boolean параметр, отвечающий за то, "нарезать" ли keyword
     * @return {@link Int2ObjectOpenHashMap&lt;IntArrayList&gt;} результирующая мапа
     */
    private Int2ObjectOpenHashMap<IntArrayList> getKeywordInfoIndexMap(
            List<KeywordForUnglue> keywordList, Set<Integer> keywordSizes, boolean toCut) {
        Int2ObjectOpenHashMap<IntArrayList> result = new Int2ObjectOpenHashMap<>();
        for (int i = 0; i < keywordList.size(); i++) {
            KeywordForUnglue curKeyword = keywordList.get(i);
            if (!curKeyword.isQuoted()) {
                if (toCut) {
                    // если мы выкинем одно слово из curKeyword, будут ли другие фразы с таким количеством слов?
                    // если нет - нет смысла и нарезать
                    if (!keywordSizes.contains(curKeyword.getPlusWords().size() - 1)) {
                        continue;
                    }
                    for (int keywordToCutIndex = 0;
                         keywordToCutIndex < curKeyword.getPlusWords().size();
                         ++keywordToCutIndex) {
                        for (int keywordInfoHash : createCombinedInfoHashes(curKeyword, keywordToCutIndex)) {
                            IntArrayList indexes = result.computeIfAbsent(keywordInfoHash, h -> new IntArrayList());
                            indexes.add(i);
                        }
                    }
                } else {
                    for (int keywordInfoHash : createCombinedInfoHashes(curKeyword)) {
                        IntArrayList indexes = result.computeIfAbsent(keywordInfoHash, h -> new IntArrayList());
                        indexes.add(i);
                    }
                }
            }
        }
        return result;
    }

    private IntSet createCombinedInfoHashes(KeywordForUnglue keyword) {
        return createCombinedInfoHashes(keyword.getAdGroupIndex(),
                keyword.getOrderedKeywords(), keyword.getPlusWordsWithLemmas());
    }

    private IntSet createCombinedInfoHashes(KeywordForUnglue keyword, int extraWordIndex) {
        var plusWordsWithLemmas = new ArrayList<>(keyword.getPlusWordsWithLemmas());
        plusWordsWithLemmas.remove(extraWordIndex);
        return createCombinedInfoHashes(keyword.getAdGroupIndex(), keyword.getOrderedKeywords(), plusWordsWithLemmas);
    }

    /**
     * Формирует набор хэшей из (adGroupIndex, orderedKeywords, lemmasHash),
     * где lemmasHash - хэш одной из комбинаций лемм словосочетания plusWordsWithLemmas.
     */
    private IntSet createCombinedInfoHashes(Integer adGroupIndex,
                                            String orderedKeywords,
                                            List<SingleKeywordWithLemmas> plusWordsWithLemmas) {
        IntSet combinedLemmasHashes = KeywordCombinator.combineAllLemmas(plusWordsWithLemmas);
        IntSet combinedInfoHashes = new IntOpenHashSet(combinedLemmasHashes.size());
        for (int lemmasHash : combinedLemmasHashes) {
            combinedInfoHashes.add(Objects.hash(adGroupIndex, orderedKeywords, lemmasHash));
        }
        return combinedInfoHashes;
    }

    /**
     * Возвращает минус слово, если:
     * - для всех слов в subphrase имеется эквивалентное в phrase
     * - количество слов в phrase на одно больше, чем в subphrase
     * - лишнее слово имеет тип SingleKeyword
     * - оно не содержится в какой либо своей форме среди минус слов subphrase
     */
    private NormalizedWord<SingleKeyword> tryToAddMinusWord(KeywordForUnglue subPhrase, KeywordForUnglue phrase) {
        if (!subPhrase.getAdGroupIndex().equals(phrase.getAdGroupIndex()) ||
                !subPhrase.getOrderedKeywords().equals(phrase.getOrderedKeywords()) ||
                subPhrase.getPlusWords().size() != phrase.getPlusWords().size() - 1) {
            // Поскольку одна из двух входных фраз подбиралась с помощью вычисления хеша в createCombinedInfoHashes(),
            // то для исключения хэш-коллизий проверяем равенство adGroupIndex, orderedKeywords и
            // количества слов во фразах. Равенство самих слов проверяется далее в getExtraWord().
            return null;
        }
        NormalizedWord<SingleKeyword> wordForAdd = getExtraWord(subPhrase, phrase);
        if (wordForAdd != null) {
            String normalizedWordForAdd =
                    normalizeMinus(stopWordService, keywordFactory, wordForAdd.getNormalizedWord());

            boolean canAddMinusWord;
            Word originalWordForAdd = wordForAdd.getOriginalWord().getWord();

            // Минус-слово добавляется с оператором "!", если оригинальное слово было с оператором "!" или "+".
            // Например, если оригинальные слова были "!новые" или "+когда", то их минус-слова
            // добавятся как "-!новые" или "-!когда".
            if (originalWordForAdd.getKind() == WordKind.FIXED || originalWordForAdd.getKind() == WordKind.PLUS) {
                // Не дублируем фиксированные минус-слова, а также не добавляем минус-слово,
                // если оно уже уже встречалось в не зафиксированной форме
                canAddMinusWord = !subPhrase.getFixedOriginalMinusWords().contains(originalWordForAdd.getText()) &&
                        !subPhrase.getRawNormalizedMinusWords().contains(normalizedWordForAdd);
                if (canAddMinusWord) {
                    subPhrase.getFixedOriginalMinusWords().add(originalWordForAdd.getText());
                    subPhrase.getFixedNormalizedMinusWords().add(normalizedWordForAdd);
                }
            } else {
                // Не добавляем минус-слово, если оно уже встречалось в зафиксированной форме, а также
                // не дублируем не фиксированные минус-слова
                canAddMinusWord = !subPhrase.getFixedNormalizedMinusWords().contains(normalizedWordForAdd) &&
                        !subPhrase.getRawNormalizedMinusWords().contains(normalizedWordForAdd);
                if (canAddMinusWord) {
                    subPhrase.getRawNormalizedMinusWords().add(normalizedWordForAdd);
                }
            }
            if (canAddMinusWord) {
                return getNormalizedMinusWord(wordForAdd);
            }
        }
        return null;
    }

    /**
     * Для корректной работы должно выполняться:
     * subphrase.getPlusWords().size() == phrase.getPlusWords().size() - 1 и слова в списках должны быть
     * отсортированы одинаково.
     *
     * @param subphrase - список слов в подфразе
     * @param phrase    - список слов в фразе
     * @return - если все слова, кроме одного, имеют эквивалентную пару в другом списке, возвращает это слово,
     * иначе null
     */
    private static NormalizedWord<SingleKeyword> getExtraWord(KeywordForUnglue subphrase,
                                                              KeywordForUnglue phrase) {
        NormalizedWord<SingleKeyword> result = null;
        for (int i = 0, j = 0; i < subphrase.getPlusWords().size(); ++i, ++j) {
            if (!isFirstIncludedInSecond(subphrase.getPlusWordsWithLemmas().get(i),
                    phrase.getPlusWordsWithLemmas().get(j))) {
                if (result != null) {
                    return null;
                }
                result = phrase.getPlusWords().get(j);
                --i;
            }
        }
        return result != null ? result : phrase.getPlusWords().get(phrase.getPlusWords().size() - 1);
    }

    /**
     * Для слов с ! или + возвращает первую лемму этого слова, для остальных она и так уже должна быть первая
     */
    private static String normalizeMinus(StopWordService stopWordService,
                                         KeywordWithLemmasFactory keywordFactory,
                                         SingleKeyword singleKeyword) {
        boolean fixed = (singleKeyword.getWord().getKind() == WordKind.PLUS
                && stopWordService.isStopWord(singleKeyword.getWord().getText()))
                || singleKeyword.getWord().getKind() == WordKind.FIXED;

        return fixed
                ? getFirstLemma(stopWordService, keywordFactory, singleKeyword.getWord().getText())
                : singleKeyword.getWord().getText();
    }

    /**
     * Возвращает первую лемму слова word
     */
    private static String getFirstLemma(StopWordService stopWordService,
                                        KeywordWithLemmasFactory keywordFactory,
                                        String word) {
        List<String> lemmas = keywordFactory.keywordFrom(word)
                .getSingleKeywords().iterator().next()
                .getLemmas();
        String firstLemma = lemmas.get(0);
        return StreamEx.of(lemmas)
                .remove(stopWordService::isStopWord) // аналогично KeywordNormalizer::getNormalizedLemma,
                // иначе в случае, когда первая лемма - стоп-слово, получаем разную нормализацию
                .findFirst()
                .orElse(firstLemma);
    }

    /**
     * Для зафиксированных оператором "+" стоп-слов заменяем оператор на "!"
     */
    private SingleKeyword getMinusWord(SingleKeyword singleKeyword) {
        if (singleKeyword.getWord().getKind() == WordKind.PLUS) {
            if (stopWordService.isStopWord(singleKeyword.getWord().getText())) {
                return new SingleKeyword(new Word(WordKind.FIXED, singleKeyword.getWord().getText()));
            } else {
                return new SingleKeyword(new Word(WordKind.RAW, singleKeyword.getWord().getText()));
            }
        } else {
            return singleKeyword;
        }
    }

    /**
     * Для зафиксированных оператором "+" стоп-слов заменяем оператор на "!"
     */
    private NormalizedWord<SingleKeyword> getNormalizedMinusWord(NormalizedWord<SingleKeyword> normalizedMinusWord) {
        return new NormalizedWord<>(getMinusWord(normalizedMinusWord.getOriginalWord()),
                getMinusWord(normalizedMinusWord.getNormalizedWord()));
    }

    private static class KeywordForUnglue {
        Integer index;
        Integer adGroupIndex;
        List<NormalizedWord<SingleKeyword>> plusWords = new ArrayList<>();
        List<SingleKeywordWithLemmas> plusWordsWithLemmas = new ArrayList<>();
        String orderedKeywords;

        boolean isQuoted;

        // для быстрого поиска добавляемого минус-слова в текущих минус-словах
        private Set<String> fixedOriginalMinusWords;
        private Set<String> fixedNormalizedMinusWords;
        private Set<String> rawNormalizedMinusWords;

        static KeywordForUnglue build(StopWordService stopWordService,
                                      KeywordWithLemmasFactory keywordFactory,
                                      SingleKeywordsCache singleKeywordsCache,
                                      UnglueContainer unglueContainer) {
            KeywordForUnglue result = new KeywordForUnglue();
            result.index = unglueContainer.getIndex();
            result.adGroupIndex = unglueContainer.getAdGroupIndex();
            List<NormalizedWord<AnyKeyword>> tmpKeywords = KeywordProcessingUtils.cleanLoneWordsFromBrackets(stopWordService,
                    unglueContainer.getNormalizedKeywordWithMinuses().getKeyword()).getAllKeywords();

            StringBuilder sb = new StringBuilder();
            for (NormalizedWord<AnyKeyword> keyword : tmpKeywords) {
                AnyKeyword normalizedWord = keyword.getNormalizedWord();
                AnyKeyword originalWord = keyword.getOriginalWord();
                if (normalizedWord instanceof SingleKeyword) {
                    result.plusWords.add(new NormalizedWord<>((SingleKeyword) originalWord,
                            (SingleKeyword) normalizedWord));
                } else if (normalizedWord instanceof OrderedKeyword) {
                    sb.append(normalizedWord.toString());
                } else {
                    throw new UnknownKeywordTypeException("Unknown keyword type " + keyword.getClass());
                }
            }
            result.orderedKeywords = sb.toString();
            List<NormalizedWord<SingleKeyword>> minusWords =
                    unglueContainer.getNormalizedKeywordWithMinuses().getMinusWords();
            result.fixedOriginalMinusWords = StreamEx.of(minusWords)
                    .map(NormalizedWord::getOriginalWord)
                    .filter(sk -> sk.getWord().getKind() == WordKind.FIXED)
                    .map(sk -> sk.getWord().getText())
                    .toSet();
            result.fixedNormalizedMinusWords = StreamEx.of(minusWords)
                    .filter(nw -> nw.getOriginalWord().getWord().getKind() == WordKind.FIXED)
                    .map(NormalizedWord::getNormalizedWord)
                    .map(sk -> normalizeMinus(stopWordService, keywordFactory, sk))
                    .toSet();
            result.rawNormalizedMinusWords = StreamEx.of(minusWords)
                    .filter(nw -> nw.getOriginalWord().getWord().getKind() == WordKind.RAW)
                    .map(NormalizedWord::getNormalizedWord)
                    .map(sk -> normalizeMinus(stopWordService, keywordFactory, sk))
                    .toSet();
            result.isQuoted = unglueContainer.getNormalizedKeywordWithMinuses().getKeyword().isQuoted();
            result.plusWordsWithLemmas = StreamEx.of(result.plusWords)
                    // берем только первый SingleKeywordWithLemmas, т.к. фразы нормализованы
                    .map(kw -> singleKeywordsCache.singleKeywordsFrom(kw.getOriginalWord().getWord()).get(0))
                    .toList();
            return result;
        }

        public Integer getIndex() {
            return index;
        }

        public Integer getAdGroupIndex() {
            return adGroupIndex;
        }

        List<NormalizedWord<SingleKeyword>> getPlusWords() {
            return plusWords;
        }

        List<SingleKeywordWithLemmas> getPlusWordsWithLemmas() {
            return plusWordsWithLemmas;
        }

        String getOrderedKeywords() {
            return orderedKeywords;
        }

        Set<String> getFixedOriginalMinusWords() {
            return fixedOriginalMinusWords;
        }

        Set<String> getFixedNormalizedMinusWords() {
            return fixedNormalizedMinusWords;
        }

        Set<String> getRawNormalizedMinusWords() {
            return rawNormalizedMinusWords;
        }

        boolean isQuoted() {
            return isQuoted;
        }
    }
}
