package ru.yandex.direct.core.entity.keyword.processing.unglue;

import java.util.List;
import java.util.Map;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.keyword.processing.NormalizedWord;
import ru.yandex.direct.libs.keywordutils.model.SingleKeyword;

import static java.util.Objects.requireNonNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Результат расклейки для новой фразы (добавляемой или обновляемой).
 */
public class UnglueResult {

    /**
     * Индекс, соответствующий переданному индексу новой фразы во входном контейнере {@link UnglueContainer}.
     */
    private final Integer index;
    /**
     * Добавленные минус-слова (если не добавилось - пустой список).
     */
    private final List<NormalizedWord<SingleKeyword>> addedMinusWords;
    /**
     * Мапа с существующими фразами, которым необходимо добавить минус-слова.
     * Ключами являются индексы существующих фраз, а значениями - добавляемые минус-слова.
     */
    private final Map<Integer, NormalizedWord<SingleKeyword>> addedMinusWordsToExistingMap;

    public UnglueResult(Integer index, List<NormalizedWord<SingleKeyword>> addedMinusWords,
                        Map<Integer, NormalizedWord<SingleKeyword>> addedMinusWordsToExistingMap) {
        this.index = requireNonNull(index, "index");
        this.addedMinusWords = ImmutableList.copyOf(requireNonNull(addedMinusWords,
                "addedMinusWords"));
        this.addedMinusWordsToExistingMap = ImmutableMap.copyOf(
                requireNonNull(addedMinusWordsToExistingMap, "addedMinusWordsToExistingMap"));
    }

    public Integer getIndex() {
        return index;
    }

    public List<NormalizedWord<SingleKeyword>> getAddedMinusWords() {
        return addedMinusWords;
    }

    public List<SingleKeyword> getNormalizedAddedMinusWords() {
        return mapList(addedMinusWords, NormalizedWord::getNormalizedWord);
    }

    public Map<Integer, SingleKeyword> getNormalizedAddedMinusWordsToExistingMap() {
        return StreamEx.of(addedMinusWordsToExistingMap.entrySet())
                .mapToEntry(Map.Entry::getKey, e -> e.getValue().getNormalizedWord())
                .toMap();
    }
}
