package ru.yandex.direct.core.entity.keyword.repository;

import java.math.BigInteger;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.types.ULong;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.keyword.repository.internal.DbAddedPhraseType;
import ru.yandex.direct.core.entity.keyword.repository.internal.DbAddedPhrasesCache;
import ru.yandex.direct.dbschema.ppc.tables.records.AddedPhrasesCacheRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.Arrays.asList;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.dbschema.ppc.Tables.ADDED_PHRASES_CACHE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.HashingUtils.getMd5HalfHashUtf8;

/**
 * Репозиторий для таблицы ppc.added_phrases_cache
 */
@Repository
@ParametersAreNonnullByDefault
public class KeywordCacheRepository {

    private JooqMapperWithSupplier<DbAddedPhrasesCache> jooqMapper = buildJooqMapper();
    private Collection<Field<?>> allFieldsToRead = jooqMapper.getFieldsToRead();

    private final DslContextProvider dslContextProvider;

    @Autowired
    public KeywordCacheRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public void addCampaignMinusKeywords(int shard, Map<Long, List<String>> campaignIdToNormalMinusKeywords) {
        LocalDateTime now = LocalDateTime.now();
        List<DbAddedPhrasesCache> keywordsForCaching = new ArrayList<>();

        for (Map.Entry<Long, List<String>> campaignIdToNormalKeywordsEntry : campaignIdToNormalMinusKeywords
                .entrySet()) {
            long campaignId = campaignIdToNormalKeywordsEntry.getKey();
            List<String> minusKeywords = campaignIdToNormalKeywordsEntry.getValue();
            List<DbAddedPhrasesCache> campaignKeywordsForCaching = StreamEx.of(minusKeywords)
                    .map(minusKeyword -> new DbAddedPhrasesCache()
                            .withCampaignId(campaignId)
                            .withAdGroupId(0L)
                            .withBidsId(0L)
                            .withType(DbAddedPhraseType.MINUS)
                            .withHash(getMd5HalfHashUtf8(minusKeyword))
                            .withAddDate(now))
                    .toList();
            keywordsForCaching.addAll(campaignKeywordsForCaching);
        }

        addKeywordsCache(shard, keywordsForCaching);
    }

    public void addAdGroupMinusKeywords(int shard, Map<AdGroup, List<String>> adGroupToNormalMinusKeywords) {
        LocalDateTime now = LocalDateTime.now();
        List<DbAddedPhrasesCache> keywordsForCaching = new ArrayList<>();

        for (Map.Entry<AdGroup, List<String>> adGroupToNormalKeywordsEntry : adGroupToNormalMinusKeywords.entrySet()) {
            long adGroupId = adGroupToNormalKeywordsEntry.getKey().getId();
            long campaignId = adGroupToNormalKeywordsEntry.getKey().getCampaignId();
            List<String> minusKeywords = adGroupToNormalKeywordsEntry.getValue();
            List<DbAddedPhrasesCache> adGroupKeywordsForCaching = StreamEx.of(minusKeywords)
                    .map(minusKeyword -> new DbAddedPhrasesCache()
                            .withCampaignId(campaignId)
                            .withAdGroupId(adGroupId)
                            .withBidsId(0L)
                            .withType(DbAddedPhraseType.MINUS)
                            .withHash(getMd5HalfHashUtf8(minusKeyword))
                            .withAddDate(now))
                    .toList();
            keywordsForCaching.addAll(adGroupKeywordsForCaching);
        }

        addKeywordsCache(shard, keywordsForCaching);
    }

    private void addKeywordsCache(int shard, List<DbAddedPhrasesCache> dbAddedPhrasesCaches) {
        LocalDateTime now = LocalDateTime.now();

        InsertHelper<AddedPhrasesCacheRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), ADDED_PHRASES_CACHE);

        for (DbAddedPhrasesCache dbAddedPhrasesCache : dbAddedPhrasesCaches) {
            insertHelper.add(jooqMapper, dbAddedPhrasesCache).newRecord();
        }

        insertHelper.onDuplicateKeyUpdate()
                .set(ADDED_PHRASES_CACHE.ADD_DATE, now);
        insertHelper.executeIfRecordsAdded();
    }

    public List<DbAddedPhrasesCache> getCachedKeywordsByCampaignIds(int shard, List<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(allFieldsToRead)
                .from(ADDED_PHRASES_CACHE)
                .where(ADDED_PHRASES_CACHE.CID.in(campaignIds))
                .fetch()
                .map(jooqMapper::fromDb);
    }

    public Map<Long, List<BigInteger>> getAdGroupCachedKeywords(int shard, DbAddedPhraseType type,
                                                                List<Long> adGroupIds) {
        Result<Record> result = dslContextProvider.ppc(shard)
                .select(asList(ADDED_PHRASES_CACHE.PID, ADDED_PHRASES_CACHE.PHRASE_HASH))
                .from(ADDED_PHRASES_CACHE)
                .where(ADDED_PHRASES_CACHE.PID.in(adGroupIds))
                .and(ADDED_PHRASES_CACHE.TYPE.eq(DbAddedPhraseType.toSource(type)))
                .fetch();
        return StreamEx.of(result)
                .collect(groupingBy(
                        rec -> rec.getValue(ADDED_PHRASES_CACHE.PID),
                        mapping(rec -> rec.getValue(ADDED_PHRASES_CACHE.PHRASE_HASH).toBigInteger(), toList())));
    }

    public Map<Long, List<BigInteger>> getCampaignCachedKeywords(int shard, DbAddedPhraseType type,
                                                                 List<Long> campaignIds) {
        Result<Record> result = dslContextProvider.ppc(shard)
                .select(asList(ADDED_PHRASES_CACHE.CID, ADDED_PHRASES_CACHE.PHRASE_HASH))
                .from(ADDED_PHRASES_CACHE)
                .where(ADDED_PHRASES_CACHE.CID.in(campaignIds))
                .and(ADDED_PHRASES_CACHE.PID.eq(0L))
                .and(ADDED_PHRASES_CACHE.TYPE.eq(DbAddedPhraseType.toSource(type)))
                .fetch();
        return StreamEx.of(result)
                .collect(groupingBy(
                        rec -> rec.getValue(ADDED_PHRASES_CACHE.CID),
                        mapping(rec -> rec.getValue(ADDED_PHRASES_CACHE.PHRASE_HASH).toBigInteger(), toList())));
    }

    public static JooqMapperWithSupplier<DbAddedPhrasesCache> buildJooqMapper() {
        return JooqMapperWithSupplierBuilder.builder(DbAddedPhrasesCache::new)
                .map(property(DbAddedPhrasesCache.CAMPAIGN_ID, ADDED_PHRASES_CACHE.CID))
                .map(property(DbAddedPhrasesCache.AD_GROUP_ID, ADDED_PHRASES_CACHE.PID))
                .map(property(DbAddedPhrasesCache.BIDS_ID, ADDED_PHRASES_CACHE.BIDS_ID))
                .map(convertibleProperty(DbAddedPhrasesCache.TYPE, ADDED_PHRASES_CACHE.TYPE,
                        DbAddedPhraseType::fromSource, DbAddedPhraseType::toSource))
                .map(convertibleProperty(DbAddedPhrasesCache.HASH, ADDED_PHRASES_CACHE.PHRASE_HASH, ULong::toBigInteger,
                        ULong::valueOf))
                .map(property(DbAddedPhrasesCache.ADD_DATE, ADDED_PHRASES_CACHE.ADD_DATE))
                .build();
    }
}
