package ru.yandex.direct.core.entity.keyword.repository;

import java.math.BigInteger;
import java.util.Collection;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.types.ULong;
import org.jooq.types.UNumber;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.keyword.model.ForecastCtr;
import ru.yandex.direct.dbschema.ppcdict.tables.records.ForecastCtrRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppcdict.tables.ForecastCtr.FORECAST_CTR;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class KeywordForecastRepository {
    private static final Logger logger = LoggerFactory.getLogger(KeywordForecastRepository.class);

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<ForecastCtr> jooqMapper;

    @Autowired
    public KeywordForecastRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        jooqMapper = buildMapper();
    }

    @Nonnull
    private static JooqMapperWithSupplier<ForecastCtr> buildMapper() {
        return JooqMapperWithSupplierBuilder.builder(ForecastCtr::new)
                .map(convertibleProperty(ForecastCtr.HALF_HASH, FORECAST_CTR.HASH, UNumber::toBigInteger, ULong::valueOf))
                .map(property(ForecastCtr.GUARANTEE_CTR, FORECAST_CTR.CTR))
                .map(property(ForecastCtr.PREMIUM_CTR, FORECAST_CTR.PCTR))
                .build();
    }

    /**
     * По заданным числовым представлениям hash'ей от нормализованных фраз получает набор {@link ForecastCtr}
     * с данными о прогнозируемых CTR'ах из {@code PPCDICT.FORECAST_CTR}.
     * Если по указанному hash-значению не найдено значение, в ответе этот hash будет отсутствовать.
     */
    public List<ForecastCtr> getForecastsByHalfHashes(Collection<BigInteger> halfHashes) {
        return dslContextProvider.ppcdict()
                .select(jooqMapper.getFieldsToRead())
                .from(FORECAST_CTR)
                .where(FORECAST_CTR.HASH.in(halfHashes))
                .fetch(jooqMapper::fromDb);
    }

    /**
     * Добавляет записи в {@link ru.yandex.direct.dbschema.ppcdict.tables.ForecastCtr#FORECAST_CTR}
     *
     * @param forecastCtrs набор {@link ForecastCtr}
     */
    public void addForecastsByHalfHash(Collection<ForecastCtr> forecastCtrs) {
        DSLContext dslContext = dslContextProvider.ppcdict();
        InsertHelper<ForecastCtrRecord> insertHelper = new InsertHelper<>(dslContext, FORECAST_CTR);
        int addedCount = insertHelper.addAll(jooqMapper, forecastCtrs).execute();
        logger.debug("Added {} rows to FORECAST_CTR", addedCount);
    }

    /**
     * Удаляет записи из {@link ru.yandex.direct.dbschema.ppcdict.tables.ForecastCtr#FORECAST_CTR}
     *
     * @param halfHashes набор полухэшей от нормализованных ключевых фраз
     */
    public void removeForecastsByHalfHash(Collection<BigInteger> halfHashes) {
        int deletedRows = dslContextProvider.ppcdict()
                .deleteFrom(FORECAST_CTR)
                .where(FORECAST_CTR.HASH.in(halfHashes))
                .execute();
        logger.debug("Deleted {} rows from FORECAST_CTR", deletedRows);
    }

}
