package ru.yandex.direct.core.entity.keyword.repository;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import com.google.common.collect.MultimapBuilder;
import com.google.common.collect.SetMultimap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Record2;
import org.jooq.RecordMapper;
import org.jooq.Result;
import org.jooq.Select;
import org.jooq.SelectConditionStep;
import org.jooq.SelectJoinStep;
import org.jooq.TableField;
import org.jooq.impl.DSL;
import org.jooq.impl.TableImpl;
import org.jooq.types.ULong;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.bshistory.History;
import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.keyword.container.CampaignIdAndKeywordIdPair;
import ru.yandex.direct.core.entity.keyword.container.KeywordDeleteInfo;
import ru.yandex.direct.core.entity.keyword.container.PhraseIdHistoryInfo;
import ru.yandex.direct.core.entity.keyword.model.FindAndReplaceKeyword;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.model.KeywordForecast;
import ru.yandex.direct.core.entity.keyword.model.KeywordModeration;
import ru.yandex.direct.core.entity.keyword.model.KeywordText;
import ru.yandex.direct.core.entity.keyword.model.Place;
import ru.yandex.direct.core.entity.keyword.model.StatusModerate;
import ru.yandex.direct.core.entity.keyword.processing.PhraseIdAndKeywordIdComparator;
import ru.yandex.direct.core.entity.statistics.container.ChangedPhraseIdInfo;
import ru.yandex.direct.dbschema.ppc.Indexes;
import ru.yandex.direct.dbschema.ppc.enums.BidsStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatusmoderate;
import ru.yandex.direct.dbschema.ppc.tables.BidsArc;
import ru.yandex.direct.dbschema.ppc.tables.records.BidsHrefParamsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.BidsPhraseidHistoryRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.BidsRecord;
import ru.yandex.direct.dbutil.DBUtils;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperUtils;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.core.entity.campaign.repository.CampaignMappings.archivedFromDb;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_ARC;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_HREF_PARAMS;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_MANUAL_PRICES;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_PHRASEID_HISTORY;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.dbutil.SqlUtils.STRAIGHT_JOIN;
import static ru.yandex.direct.jooqmapper.JooqMapperUtils.makeCaseStatement;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperties;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;
import static ru.yandex.direct.utils.FunctionalUtils.castListElementTypeOfMap;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class KeywordRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<Keyword> keywordJooqMapper;
    private final JooqMapperWithSupplier<Keyword> archivedKeywordJooqMapper;
    private final Collection<TableField<?, ?>> keywordFieldsToRead;
    private final Collection<TableField<?, ?>> archivedKeywordFieldsToRead;
    private final Set<Field> keywordArcListToReadFromKeywordMapper;
    private final Collection<TableField<?, ?>> keywordTextFieldsToRead;
    private final Collection<TableField<?, ?>> findAndReplaceKeywordFieldsToRead;
    private final Collection<TableField<?, ?>> keywordModerationFieldsToRead;
    private final ShardHelper shardHelper;

    private static final Comparator<Keyword> PHRASE_ID_KEYWORD_ID_COMPARATOR =
            new PhraseIdAndKeywordIdComparator();
    private static final int UPDATE_PHRASEID_CHUNK_SIZE = 400;

    @Autowired
    public KeywordRepository(DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;

        this.keywordJooqMapper = JooqMapperWithSupplierBuilder.builder(Keyword::new)
                .map(property(Keyword.ID, BIDS.ID))
                .map(property(Keyword.NORM_PHRASE, BIDS.NORM_PHRASE))
                .map(integerProperty(Keyword.WORDS_COUNT, BIDS.NUMWORD))
                .map(convertibleProperty(Keyword.PRICE, BIDS.PRICE, KeywordMapping::priceFromDbFormat,
                        KeywordMapping::priceToDbFormat))
                .map(convertibleProperty(Keyword.PRICE_CONTEXT, BIDS.PRICE_CONTEXT, KeywordMapping::priceFromDbFormat,
                        KeywordMapping::priceToDbFormat))
                .map(convertibleProperty(Keyword.PLACE, BIDS.PLACE, Place::convertFromDb, Place::convertToDb))
                .map(property(Keyword.CAMPAIGN_ID, BIDS.CID))
                .map(property(Keyword.MODIFICATION_TIME, BIDS.MODTIME))
                .map(convertibleProperty(Keyword.PHRASE_BS_ID, BIDS.PHRASE_ID, ULong::toBigInteger, ULong::valueOf))
                .map(convertibleProperty(Keyword.STATUS_MODERATE, BIDS.STATUS_MODERATE, StatusModerate::fromSource,
                        StatusModerate::toSource))
                .map(convertibleProperty(Keyword.NEED_CHECK_PLACE_MODIFIED, BIDS.WARN,
                        KeywordMapping::needCheckPlaceModifiedFromDbFormat,
                        KeywordMapping::needCheckPlaceModifiedToDbFormat))
                .map(convertibleProperty(Keyword.STATUS_BS_SYNCED, BIDS.STATUS_BS_SYNCED,
                        KeywordMapping::statusBsSyncedFromDbFormat, KeywordMapping::statusBsSyncedToDbFormat))
                .map(integerProperty(Keyword.AUTOBUDGET_PRIORITY, BIDS.AUTOBUDGET_PRIORITY))
                .map(property(Keyword.SHOWS_FORECAST, BIDS.SHOWS_FORECAST))
                .map(convertibleProperty(Keyword.IS_SUSPENDED, BIDS.IS_SUSPENDED, RepositoryUtils::booleanFromLong,
                        RepositoryUtils::booleanToLong))
                .map(property(Keyword.AD_GROUP_ID, BIDS.PID))
                .writeField(BIDS_PHRASEID_HISTORY.ID, fromProperty(Keyword.ID))
                .writeField(BIDS_PHRASEID_HISTORY.CID, fromProperty(Keyword.CAMPAIGN_ID))
                .map(convertibleProperty(Keyword.PHRASE_ID_HISTORY, BIDS_PHRASEID_HISTORY.PHRASE_ID_HISTORY,
                        t -> t != null ? History.parse(t) : null, History::serialize))
                .writeField(BIDS_PHRASEID_HISTORY.UPDATE_TIME, fromProperty(Keyword.MODIFICATION_TIME))
                .writeField(BIDS_HREF_PARAMS.ID, fromProperty(Keyword.ID))
                .writeField(BIDS_HREF_PARAMS.CID, fromProperty(Keyword.CAMPAIGN_ID))
                .readProperty(Keyword.IS_AUTOTARGETING,
                        fromField(BIDS.PHRASE).by(KeywordMapping::isAutotargetingFromDb))
                .readProperty(Keyword.PHRASE, fromField(BIDS.PHRASE).by(KeywordMapping::phraseFromDb))
                .writeField(BIDS.PHRASE,
                        fromProperties(Keyword.IS_AUTOTARGETING, Keyword.PHRASE).by(KeywordMapping::phraseToDbFormat))
                .map(property(Keyword.HREF_PARAM1, BIDS_HREF_PARAMS.PARAM1))
                .map(property(Keyword.HREF_PARAM2, BIDS_HREF_PARAMS.PARAM2))
                .build();

        this.archivedKeywordJooqMapper = buildArchivedKeywordJooqMapper();

        this.keywordFieldsToRead =
                mapList(keywordJooqMapper.getFieldsToRead(Keyword.allModelProperties()), DBUtils::safeMapToTableField);

        Collection<TableField<?, ?>> bidsArcFieldsToRead =
                mapList(archivedKeywordJooqMapper.getFieldsToRead(Keyword.allModelProperties()),
                        DBUtils::safeMapToTableField);

        this.archivedKeywordFieldsToRead = StreamEx.of(
                        filterList(keywordFieldsToRead,
                                tableField -> !BIDS.getName().equals(tableField.getTable().getName())),
                        bidsArcFieldsToRead
                )
                .flatMap(Collection::stream)
                .distinct()
                .toList();

        this.keywordArcListToReadFromKeywordMapper = filterAndMapToSet(
                keywordFieldsToRead,
                tableField -> !BIDS.getName().equals(tableField.getTable().getName()),
                Function.identity());

        this.keywordTextFieldsToRead =
                mapList(keywordJooqMapper.getFieldsToRead(KeywordText.allModelProperties()),
                        DBUtils::safeMapToTableField);
        this.findAndReplaceKeywordFieldsToRead =
                mapList(keywordJooqMapper.getFieldsToRead(FindAndReplaceKeyword.allModelProperties()),
                        DBUtils::safeMapToTableField);
        this.keywordModerationFieldsToRead =
                mapList(keywordJooqMapper.getFieldsToRead(KeywordModeration.allModelProperties()),
                        DBUtils::safeMapToTableField);
    }

    public static JooqMapperWithSupplier<Keyword> buildArchivedKeywordJooqMapper() {
        return JooqMapperWithSupplierBuilder.builder(Keyword::new)
                .map(property(Keyword.ID, BIDS_ARC.ID))
                .map(property(Keyword.NORM_PHRASE, BIDS_ARC.NORM_PHRASE))
                .map(integerProperty(Keyword.WORDS_COUNT, BIDS_ARC.NUMWORD))
                .map(convertibleProperty(Keyword.PRICE, BIDS_ARC.PRICE, KeywordMapping::priceFromDbFormat,
                        KeywordMapping::priceToDbFormat))
                .map(convertibleProperty(Keyword.PRICE_CONTEXT, BIDS_ARC.PRICE_CONTEXT,
                        KeywordMapping::priceFromDbFormat,
                        KeywordMapping::priceToDbFormat))
                .map(convertibleProperty(Keyword.PLACE, BIDS_ARC.PLACE, Place::convertFromDb, Place::convertToDb))
                .map(property(Keyword.CAMPAIGN_ID, BIDS_ARC.CID))
                .map(property(Keyword.MODIFICATION_TIME, BIDS_ARC.MODTIME))
                .map(convertibleProperty(Keyword.PHRASE_BS_ID, BIDS_ARC.PHRASE_ID, ULong::toBigInteger, ULong::valueOf))
                .map(convertibleProperty(Keyword.STATUS_MODERATE, BIDS_ARC.STATUS_MODERATE,
                        KeywordMapping::statusModerateFromDbFormat,
                        KeywordMapping::statusModerateToArcDbFormat))
                .map(convertibleProperty(Keyword.NEED_CHECK_PLACE_MODIFIED, BIDS_ARC.WARN,
                        KeywordMapping::needCheckPlaceModifiedFromDbFormat,
                        KeywordMapping::needCheckPlaceModifiedToArcDbFormat))
                .map(convertibleProperty(Keyword.STATUS_BS_SYNCED, BIDS_ARC.STATUS_BS_SYNCED,
                        KeywordMapping::statusBsSyncedFromDbFormat, KeywordMapping::statusBsSyncedToArcDbFormat))
                .map(integerProperty(Keyword.AUTOBUDGET_PRIORITY, BIDS_ARC.AUTOBUDGET_PRIORITY))
                .map(property(Keyword.SHOWS_FORECAST, BIDS_ARC.SHOWS_FORECAST))
                .map(convertibleProperty(Keyword.IS_SUSPENDED, BIDS_ARC.IS_SUSPENDED, RepositoryUtils::booleanFromLong,
                        RepositoryUtils::booleanToLong))
                .map(property(Keyword.AD_GROUP_ID, BIDS_ARC.PID))
                .readProperty(Keyword.IS_AUTOTARGETING,
                        fromField(BIDS_ARC.PHRASE).by(KeywordMapping::isAutotargetingFromDb))
                .readProperty(Keyword.PHRASE, fromField(BIDS_ARC.PHRASE).by(KeywordMapping::phraseFromDb))
                .writeField(BIDS_ARC.PHRASE,
                        fromProperties(Keyword.IS_AUTOTARGETING, Keyword.PHRASE).by(KeywordMapping::phraseToDbFormat))
                .build();
    }

    /**
     * Возвращает мапу удаленных ключевиков.
     * Данных метод удаляет любые дубли ключевых фраз в рамках группы (напр. не переданные в метод)
     */
    public Map<Long, Keyword> addAndUpdateWithDeduplication(Configuration configuration,
                                                            List<Keyword> keywordsToAdd,
                                                            List<AppliedChanges<Keyword>> keywordsToUpdate) {
        addKeywords(configuration, keywordsToAdd);
        update(configuration, keywordsToUpdate);

        Set<Long> affectedAdGroupIds = listToSet(keywordsToAdd, Keyword::getAdGroupId);
        return listToMap(deduplicateKeywords(configuration, affectedAdGroupIds), Keyword::getId);
    }

    /**
     * Данных метод удаляет любые дубли в рамках группы (напр. не переданные в {@code campaignIdToKeywordIdDeleteList})
     *
     * @param configuration                   конфиг
     * @param allKeywordsToUpdate             все ключевые слова для обновления
     * @param campaignIdToKeywordIdDeleteList список ключевиков на удаления, состоящие из ид кампании и ид ключевика
     * @return список удаленных ключевиков
     */
    public List<Keyword> updateAndDeleteWithDeduplication(Configuration configuration,
                                                          Collection<AppliedChanges<Keyword>> allKeywordsToUpdate,
                                                          List<CampaignIdAndKeywordIdPair> campaignIdToKeywordIdDeleteList) {
        update(configuration, allKeywordsToUpdate);

        if (!campaignIdToKeywordIdDeleteList.isEmpty()) {
            deleteKeywords(configuration, campaignIdToKeywordIdDeleteList);
        }

        Set<Long> affectedAdGroupIds = listToSet(allKeywordsToUpdate, ac -> ac.getModel().getAdGroupId());
        return deduplicateKeywords(configuration, affectedAdGroupIds);
    }

    /**
     * Добавляет ключевые фразы путем добавления в таблицы: bids, bids_phraseid_history, bids_href_params
     *
     * @return id добавленных ключевых фраз
     */
    public List<Long> addKeywords(Configuration config, List<Keyword> keywords) {
        generateKeywordsIds(keywords);
        new InsertHelper<>(DSL.using(config), BIDS)
                .addAll(keywordJooqMapper, keywords)
                .executeIfRecordsAdded();
        addToBidsPhraseidHistory(DSL.using(config), keywords);
        addToBidsHrefParams(DSL.using(config), keywords);
        return mapList(keywords, Keyword::getId);
    }

    /**
     * Архивирует фразы по списку ids, см. Common.pm::_arc_camp
     */
    public void archiveKeywords(Configuration config, List<Long> ids) {
        DSLContext dslContext = DSL.using(config);
        copyKeywordsToArchive(config, ids);

        deleteFromBidsTable(dslContext, ids);
    }

    /**
     * Создает не валидный стейт в БД! Нужен для использования в тестах.
     * Написано в этом классе, ввиду наличия тут мэпперов которые не очень красиво выставлять наружу пабликом
     */
    public void copyKeywordsToArchiveNoProduction(Configuration config, List<Long> ids) {
        copyKeywordsToArchive(config, ids);
    }

    private void copyKeywordsToArchive(Configuration config, List<Long> ids) {
        DSLContext dslContext = DSL.using(config);
        BidsArc tableCopyTo = BIDS_ARC;

        Set<Field<?>> bidsArcFieldsToWrite = archivedKeywordJooqMapper.getFieldsToRead();
        List<Field<?>> bidsFieldsToCopy = bidsArcFieldsToWrite.stream()
                .map(f -> DSL.field(DSL.name(BIDS.getName(), f.getName())))
                .collect(toList());

        SelectConditionStep<Record> bidsToInsertSelect =
                dslContext.select(bidsFieldsToCopy).from(BIDS).where(BIDS.ID.in(ids));

        Map<Field, Field> onDuplicateKeyUpdateMap = new HashMap<>();

        for (Field<?> field : bidsArcFieldsToWrite) {
            if (!tableCopyTo.ID.equals(field)) {
                Field<?> selectedBidsField = DSL.field(DSL.name(BIDS.getName(), field.getName()));
                onDuplicateKeyUpdateMap.put(field, selectedBidsField);
            }
        }

        dslContext.insertInto(tableCopyTo, bidsArcFieldsToWrite)
                .select(bidsToInsertSelect).onDuplicateKeyUpdate().set(onDuplicateKeyUpdateMap).execute();
    }

    /**
     * Разархивирует фразы по списку id кампаний, см. Common.pm::unarc_camp
     */
    public void unarchiveKeywords(Configuration config, List<Long> campaignIds,
                                  List<Long> needRecalcPriceContextCids, BigDecimal minPrice) {
        if (campaignIds.isEmpty()) {
            return;
        }
        DSLContext dslContext = DSL.using(config);

        Set<Field<?>> bidsArcFieldsToCopy = archivedKeywordJooqMapper.getFieldsToRead();

        List<Field<?>> bidsFieldsToWrite = bidsArcFieldsToCopy.stream()
                .map(f -> DSL.field(DSL.name(BIDS.getName(), f.getName())))
                .collect(toList());

        SelectConditionStep<Record> bidsArcToInsertSelect = dslContext
                .select(bidsArcFieldsToCopy).from(BIDS_ARC).where(BIDS_ARC.CID.in(campaignIds));

        Map<Field<?>, Field<?>> onDuplicateKeyUpdateMap = new HashMap<>();
        for (Field<?> field : bidsFieldsToWrite) {
            if (!BIDS.ID.equals(field)) {
                Field<?> selectedBidsField = DSL.field(DSL.name(BIDS_ARC.getName(), field.getName()));
                onDuplicateKeyUpdateMap.put(field, selectedBidsField);
            }
        }

        dslContext.insertInto(BIDS, bidsFieldsToWrite)
                .select(bidsArcToInsertSelect).onDuplicateKeyUpdate().set(onDuplicateKeyUpdateMap).execute();

        // нужно для кампаний, заархивированных до миграции DIRECT-102225
        if (!needRecalcPriceContextCids.isEmpty()) {
            dslContext.update(BIDS)
                    .set(BIDS.PRICE_CONTEXT, DSL.greatest(
                            BIDS.PRICE.mul(DSL.select(CAMPAIGNS.CONTEXT_PRICE_COEF.div(100.0))
                                    .from(CAMPAIGNS).where(CAMPAIGNS.CID.eq(BIDS.CID)).asField()),
                            DSL.val(minPrice)))
                    .where(BIDS.CID.in(needRecalcPriceContextCids))
                    .execute();
        }

        dslContext.deleteFrom(BIDS_ARC)
                .where(BIDS_ARC.CID.in(campaignIds))
                .execute();
    }

    private void generateKeywordsIds(Collection<Keyword> keywords) {
        Iterator<Long> keywordIdsIterator = shardHelper.generatePhraseIds(keywords.size()).iterator();
        keywords.forEach(kw -> kw.setId(keywordIdsIterator.next()));
    }

    private void addToBidsPhraseidHistory(DSLContext context, Collection<Keyword> keywords) {
        List<Keyword> keywordsForAdd = filterList(keywords, keyword -> keyword.getPhraseIdHistory() != null);
        new InsertHelper<>(context, BIDS_PHRASEID_HISTORY)
                .addAll(keywordJooqMapper, keywordsForAdd)
                .executeIfRecordsAdded();
    }

    private void addToBidsHrefParams(DSLContext context, Collection<Keyword> keywords) {
        List<Keyword> keywordsForAdd = filterList(keywords,
                keyword -> keyword.getHrefParam1() != null || keyword.getHrefParam2() != null);
        new InsertHelper<>(context, BIDS_HREF_PARAMS)
                .addAll(keywordJooqMapper, keywordsForAdd)
                .executeIfRecordsAdded();
    }

    @SuppressWarnings("WeakerAccess")
    public List<Keyword> getKeywordsByCampaignId(int shard, long campaignId) {
        Map<Long, List<Keyword>> keywordsByCampaignId =
                getKeywordsByCampaignIds(shard, null, singletonList(campaignId));
        return keywordsByCampaignId.getOrDefault(campaignId, emptyList());
    }

    public Map<Long, List<Keyword>> getKeywordsByCampaignIds(int shard, @Nullable ClientId clientId,
                                                             Collection<Long> campaignIds) {
        return getPartialKeywordsByCampaignIds(shard, clientId, campaignIds, keywordFieldsToRead);
    }

    public Map<Long, List<KeywordText>> getKeywordTextsByCampaignIds(int shard, @Nullable ClientId clientId,
                                                                     Collection<Long> campaignIds) {
        return getPartialKeywordsByCampaignIds(shard, clientId, campaignIds, keywordTextFieldsToRead);
    }

    private <T> Map<Long, List<T>> getPartialKeywordsByCampaignIds(int shard, @Nullable ClientId clientId,
                                                                   Collection<Long> campaignIds,
                                                                   Collection<TableField<?, ?>> fieldsToRead) {
        List<Keyword> keywords = getKeywordsByIds(dslContextProvider.ppc(shard), clientId, emptyList(), emptyList(),
                campaignIds, fieldsToRead, false);

        Map<Long, List<Keyword>> keywordsByCampaignId = keywords
                .stream()
                .collect(groupingBy(Keyword::getCampaignId));
        return castListElementTypeOfMap(keywordsByCampaignId);
    }

    @SuppressWarnings("WeakerAccess")
    public List<Keyword> getKeywordsByAdGroupId(int shard, long adGroupId) {
        Map<Long, List<Keyword>> keywordsByAdGroupId =
                getKeywordsByAdGroupIds(shard, null, singletonList(adGroupId));
        return keywordsByAdGroupId.getOrDefault(adGroupId, emptyList());
    }

    public Map<Long, List<Keyword>> getKeywordsByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        return getKeywordsByAdGroupIds(shard, null, adGroupIds);
    }

    public Map<Long, List<Keyword>> getKeywordsByAdGroupIds(int shard, @Nullable ClientId clientId,
                                                            Collection<Long> adGroupIds) {
        return getPartialKeywordsByAdGroupIds(dslContextProvider.ppc(shard), clientId, adGroupIds, keywordFieldsToRead);
    }

    public SetMultimap<Long, Long> getKeywordIdsByAdGroupIds(int shard, @Nullable ClientId clientId,
                                                             Collection<Long> adGroupIds) {
        Map<Long, List<Keyword>> keywordsByAdGroupIds =
                getPartialKeywordsByAdGroupIds(dslContextProvider.ppc(shard), clientId,
                        adGroupIds, asList(BIDS.ID, BIDS.PID));

        SetMultimap<Long, Long> multimap = MultimapBuilder.hashKeys().hashSetValues().build();
        EntryStream.of(keywordsByAdGroupIds)
                .mapValues(keywords -> mapList(keywords, Keyword::getId))
                .forKeyValue(multimap::putAll);
        return multimap;
    }

    public Map<Long, List<KeywordText>> getKeywordTextsByAdGroupIds(int shard, @Nullable ClientId clientId,
                                                                    Collection<Long> adGroupIds) {
        return getKeywordTextsByAdGroupIds(dslContextProvider.ppc(shard).configuration(), clientId, adGroupIds);
    }

    public Map<Long, List<KeywordText>> getKeywordTextsByAdGroupIds(Configuration configuration,
                                                                    @Nullable ClientId clientId,
                                                                    Collection<Long> adGroupIds) {
        return getPartialKeywordsByAdGroupIds(DSL.using(configuration), clientId, adGroupIds, keywordTextFieldsToRead);
    }

    public Map<Long, List<KeywordModeration>> getKeywordModerationsByAdGroupIds(int shard, @Nullable ClientId clientId,
                                                                                Collection<Long> adGroupIds) {
        return getPartialKeywordsByAdGroupIds(dslContextProvider.ppc(shard), clientId, adGroupIds, keywordModerationFieldsToRead);
    }

    public Map<Long, List<KeywordModeration>> getKeywordModerationsByAdGroupIds(Configuration configuration,
                                                                                @Nullable ClientId clientId,
                                                                                Collection<Long> adGroupIds) {
        return getPartialKeywordsByAdGroupIds(DSL.using(configuration), clientId, adGroupIds, keywordModerationFieldsToRead);
    }

    private <T> Map<Long, List<T>> getPartialKeywordsByAdGroupIds(DSLContext dslContext, @Nullable ClientId clientId,
                                                                  Collection<Long> adGroupIds, Collection<TableField<
            ?, ?>> fieldsToRead) {
        List<Keyword> keywords = getKeywordsByIds(dslContext, clientId, emptyList(), adGroupIds,
                emptyList(), fieldsToRead, false);

        Map<Long, List<Keyword>> keywordsByAdGroupId = keywords
                .stream()
                .collect(groupingBy(Keyword::getAdGroupId));
        return castListElementTypeOfMap(keywordsByAdGroupId);
    }

    public List<Keyword> getKeywordsByIds(int shard, ClientId clientId, Collection<Long> keywordIds) {
        return getKeywordsByIds(dslContextProvider.ppc(shard), clientId, keywordIds, emptyList(), emptyList(),
                keywordFieldsToRead, true);
    }

    public Map<Long, List<Keyword>> getArchivedKeywordsByAdGroupIds(int shard,
                                                                    @Nullable ClientId clientId,
                                                                    Collection<Long> campaignIds,
                                                                    Collection<Long> adGroupIds) {
        List<Keyword> keywords = getArchivedKeywordsByIds(shard, clientId, campaignIds, adGroupIds,
                archivedKeywordFieldsToRead);

        return keywords.stream().collect(groupingBy(Keyword::getAdGroupId));
    }

    private List<Keyword> getArchivedKeywordsByIds(int shard,
                                                   @Nullable ClientId clientId,
                                                   Collection<Long> campaignIds,
                                                   Collection<Long> adGroupIds,
                                                   Collection<TableField<?, ?>> fieldsToRead) {
        if (campaignIds.isEmpty()) { //в bids_arc только один индекс: по (cid, pid, id)
            return emptyList();
        }
        Condition condition = BIDS_ARC.CID.in(campaignIds);
        if (!adGroupIds.isEmpty()) {
            condition = condition.and(BIDS_ARC.PID.in(adGroupIds));
        }
        return getKeywordsByIdsBase(dslContextProvider.ppc(shard), clientId, BIDS_ARC, BIDS_ARC.ID, BIDS_ARC.CID,
                condition, fieldsToRead, this::archivedKeywordFromDb, false);
    }

    private Keyword archivedKeywordFromDb(Record record) {
        Keyword result = new Keyword();
        keywordJooqMapper.fromDb(record, result, keywordArcListToReadFromKeywordMapper);
        archivedKeywordJooqMapper.fromDb(record, result);
        return result;
    }

    private List<Keyword> getKeywordsByIds(DSLContext dslContext, @Nullable ClientId clientId,
                                           Collection<Long> keywordIds,
                                           Collection<Long> adGroupIds, Collection<Long> campaignIds,
                                           Collection<TableField<?, ?>> fieldsToRead,
                                           boolean needOrderById) {
        if (keywordIds.isEmpty() && adGroupIds.isEmpty() && campaignIds.isEmpty()) {
            return emptyList();
        }
        Condition condition;
        if (!keywordIds.isEmpty()) {
            condition = BIDS.ID.in(keywordIds);
        } else if (!adGroupIds.isEmpty()) {
            condition = BIDS.PID.in(adGroupIds);
        } else {
            condition = BIDS.CID.in(campaignIds);
        }
        return getKeywordsByIdsBase(dslContext, clientId, BIDS, BIDS.ID, BIDS.CID,
                condition, fieldsToRead, keywordJooqMapper::fromDb, needOrderById);
    }

    private <T extends Record> List<Keyword> getKeywordsByIdsBase(
            DSLContext dslContext,
            @Nullable ClientId clientId,
            TableImpl<T> baseTable,
            TableField<T, Long> idField,
            TableField<T, Long> cidField,
            Condition condition,
            Collection<TableField<?, ?>> fieldsToRead,
            RecordMapper<Record, Keyword> mapper,
            boolean needOrderById) {

        SelectJoinStep<Record> joinStep = dslContext
                .select(fieldsToRead)
                .hint(STRAIGHT_JOIN)
                .from(baseTable);

        if (fieldsToRead.stream().anyMatch(f -> f.getTable().equals(BIDS_PHRASEID_HISTORY))) {
            joinStep = joinStep.leftJoin(BIDS_PHRASEID_HISTORY)
                    .on(BIDS_PHRASEID_HISTORY.CID.eq(cidField)
                            .and(BIDS_PHRASEID_HISTORY.ID.eq(idField)));
        }

        if (fieldsToRead.stream().anyMatch(f -> f.getTable().equals(BIDS_HREF_PARAMS))) {
            joinStep = joinStep.leftJoin(BIDS_HREF_PARAMS)
                    .on(BIDS_HREF_PARAMS.CID.eq(cidField)
                            .and(BIDS_HREF_PARAMS.ID.eq(idField)));
        }

        if (clientId != null) {
            joinStep = joinStep.join(CAMPAIGNS)
                    .on(CAMPAIGNS.CID.eq(cidField)
                            .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                            .and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())));
        }

        SelectConditionStep<Record> conditionStep = joinStep.where(condition);

        Select<Record> select = conditionStep;
        if (needOrderById) {
            select = conditionStep.orderBy(idField);
        }

        return select.fetch(mapper);
    }

    // этот метод не то чем кажется и заполняет только 4 поля из всей модели
    public List<Keyword> getKeywordsByIds(int shard, Collection<Long> keywordIds) {
        return dslContextProvider.ppc(shard)
                .select(keywordTextFieldsToRead)
                .from(BIDS)
                .where(BIDS.ID.in(keywordIds))
                .orderBy(BIDS.ID)
                .fetch(keywordJooqMapper::fromDb);
    }

    /**
     * Получить количество ключевых фраз на группах
     *
     * @param shard       — шард
     * @param campaignIds — id кампаний (нужны для запроса в bids_arc)
     * @param adGroupIds  — id групп
     * @return мапа из id групп в количество ключевых фраз. Ключи будут только по группам с фразами.
     */
    public Map<Long, Integer> getKeywordQuantitiesByAdGroupIds(int shard, Collection<Long> campaignIds,
                                                               Collection<Long> adGroupIds) {
        return dslContextProvider.ppc(shard)
                .select(BIDS.PID, DSL.count())
                .from(BIDS)
                .where(BIDS.PID.in(adGroupIds))
                .groupBy(BIDS.PID)
                .unionAll(DSL.select(BIDS_ARC.PID, DSL.count())
                        .from(BIDS_ARC)
                        .where(BIDS_ARC.CID.in(campaignIds).and(BIDS_ARC.PID.in(adGroupIds)))
                        .groupBy(BIDS_ARC.CID, BIDS_ARC.PID))
                .fetchMap(Record2::value1, Record2::value2);
    }

    /**
     * Возвращает id записей, если кампания удалена
     */
    public List<CampaignIdAndKeywordIdPair> getIdsOfBidsPhraseIdHistoryForDeletedCampaigns(int shard) {
        return dslContextProvider.ppc(shard)
                .select(BIDS_PHRASEID_HISTORY.CID, BIDS_PHRASEID_HISTORY.ID)
                .hint(STRAIGHT_JOIN)
                .from(BIDS_PHRASEID_HISTORY)
                .leftJoin(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BIDS_PHRASEID_HISTORY.CID))
                .where(CAMPAIGNS.CID.isNull())
                .fetch(this::getIdsOfBidsPhraseIdHistoryFromRecord);
    }

    /**
     * Возвращает id записей архивных кампаний, которые не обновлялись с {@param expireDateTime}
     */
    public List<CampaignIdAndKeywordIdPair> getIdsOfBidsPhraseIdHistoryForArchivedCampaigns(int shard,
                                                                                            LocalDateTime expireDateTime) {
        return dslContextProvider.ppc(shard)
                .select(BIDS_PHRASEID_HISTORY.CID, BIDS_PHRASEID_HISTORY.ID)
                .hint(STRAIGHT_JOIN)
                .from(BIDS_PHRASEID_HISTORY)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BIDS_PHRASEID_HISTORY.CID))
                .where(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.Yes)
                        .and(BIDS_PHRASEID_HISTORY.UPDATE_TIME.lessThan(expireDateTime)))
                .fetch(this::getIdsOfBidsPhraseIdHistoryFromRecord);
    }

    /**
     * Возвращает id записей не архивных кампаний, которые не обновлялись с {@param expireDateTime}
     */
    public List<CampaignIdAndKeywordIdPair> getIdsOfBidsPhraseIdHistoryForNotArchivedCampaigns(int shard,
                                                                                               LocalDateTime expireDateTime) {
        return dslContextProvider.ppc(shard)
                .select(BIDS_PHRASEID_HISTORY.CID, BIDS_PHRASEID_HISTORY.ID)
                .hint(STRAIGHT_JOIN)
                .from(BIDS_PHRASEID_HISTORY)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BIDS_PHRASEID_HISTORY.CID))
                .leftJoin(BIDS).on(BIDS.ID.eq(BIDS_PHRASEID_HISTORY.ID))
                .where(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No)
                        .and(BIDS.ID.isNull()
                                .or(BIDS.PHRASE_ID.notEqual(ULong.valueOf(SqlUtils.ID_NOT_SET))
                                        .and(BIDS_PHRASEID_HISTORY.UPDATE_TIME.lessThan(expireDateTime)))))
                .fetch(this::getIdsOfBidsPhraseIdHistoryFromRecord);
    }

    private CampaignIdAndKeywordIdPair getIdsOfBidsPhraseIdHistoryFromRecord(Record2<Long, Long> record) {
        return new CampaignIdAndKeywordIdPair(record.get(BIDS_PHRASEID_HISTORY.CID),
                record.get(BIDS_PHRASEID_HISTORY.ID));
    }

    public void update(int shard, Collection<AppliedChanges<Keyword>> appliedChanges) {
        update(dslContextProvider.ppc(shard).configuration(), appliedChanges);
    }

    /**
     * Обновить набор таблиц с информацией по ключевой фразе
     * <p>
     * Обновляются только те поля, которые помечены изменёнными в {@link AppliedChanges}
     *
     * @param conf           контекст
     * @param appliedChanges коллекция применённых изменений
     */
    public void update(Configuration conf, Collection<AppliedChanges<Keyword>> appliedChanges) {

        List<Keyword> keywordsToAddOrUpdateHrefParams = appliedChanges.stream()
                .filter(change -> change.changed(Keyword.HREF_PARAM1) || change.changed(Keyword.HREF_PARAM2))
                .map(AppliedChanges::getModel)
                .collect(toList());

        List<Keyword> keywordsToAddOrUpdateBidsHistory = appliedChanges.stream()
                .filter(change -> change.changed(Keyword.PHRASE_ID_HISTORY)
                        && change.getNewValue(Keyword.PHRASE_ID_HISTORY) != null)
                .map(AppliedChanges::getModel)
                .collect(toList());
        List<CampaignIdAndKeywordIdPair> keywordsToDeleteBidsHistory = appliedChanges.stream()
                .filter(change -> change.changed(Keyword.PHRASE_ID_HISTORY)
                        && change.getNewValue(Keyword.PHRASE_ID_HISTORY) == null)
                .map(AppliedChanges::getModel)
                .map(model -> new CampaignIdAndKeywordIdPair(model.getCampaignId(), model.getId()))
                .collect(toList());

        updateBids(DSL.using(conf), appliedChanges);

        addOrUpdateHrefParams(DSL.using(conf), keywordsToAddOrUpdateHrefParams);

        addOrUpdateBidsPhraseHistory(DSL.using(conf), keywordsToAddOrUpdateBidsHistory);
        deleteFromBidsPhraseIdHistoryTable(DSL.using(conf), keywordsToDeleteBidsHistory);

    }

    private void updateBids(DSLContext context, Collection<AppliedChanges<Keyword>> appliedChanges) {
        JooqUpdateBuilder<BidsRecord, Keyword> bidsUpdateBuilder =
                new JooqUpdateBuilder<>(BIDS.ID, appliedChanges);

        bidsUpdateBuilder.processProperty(Keyword.PHRASE, BIDS.PHRASE);
        bidsUpdateBuilder.processProperty(Keyword.NORM_PHRASE, BIDS.NORM_PHRASE);
        bidsUpdateBuilder.processProperty(Keyword.WORDS_COUNT, BIDS.NUMWORD, Integer::longValue);
        bidsUpdateBuilder.processProperty(Keyword.PRICE, BIDS.PRICE, KeywordMapping::priceToDbFormat);
        bidsUpdateBuilder.processProperty(Keyword.PRICE_CONTEXT, BIDS.PRICE_CONTEXT, KeywordMapping::priceToDbFormat);
        bidsUpdateBuilder
                .processProperty(Keyword.AUTOBUDGET_PRIORITY, BIDS.AUTOBUDGET_PRIORITY, RepositoryUtils::intToLong);
        bidsUpdateBuilder.processProperty(Keyword.IS_SUSPENDED, BIDS.IS_SUSPENDED, RepositoryUtils::booleanToLong);

        bidsUpdateBuilder.processProperty(Keyword.STATUS_MODERATE, BIDS.STATUS_MODERATE, StatusModerate::toSource);
        bidsUpdateBuilder.processProperty(Keyword.STATUS_BS_SYNCED, BIDS.STATUS_BS_SYNCED,
                status -> BidsStatusbssynced.valueOf(status.toDbFormat()));
        bidsUpdateBuilder.processProperty(Keyword.MODIFICATION_TIME, BIDS.MODTIME);

        bidsUpdateBuilder.processProperty(Keyword.PHRASE_BS_ID, BIDS.PHRASE_ID, RepositoryUtils::bigIntegerToULong);
        bidsUpdateBuilder.processProperty(Keyword.NEED_CHECK_PLACE_MODIFIED, BIDS.WARN,
                KeywordMapping::needCheckPlaceModifiedToDbFormat);
        bidsUpdateBuilder.processProperty(Keyword.PLACE, BIDS.PLACE, Place::convertToDb);
        bidsUpdateBuilder.processProperty(Keyword.SHOWS_FORECAST, BIDS.SHOWS_FORECAST);

        context.update(BIDS)
                .set(bidsUpdateBuilder.getValues())
                .where(BIDS.ID.in(bidsUpdateBuilder.getChangedIds()))
                .execute();
    }

    private void addOrUpdateHrefParams(DSLContext context, Collection<Keyword> keywords) {
        InsertHelper<BidsHrefParamsRecord> insertHelper =
                new InsertHelper<>(context, BIDS_HREF_PARAMS)
                        .addAll(keywordJooqMapper, keywords);
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(BIDS_HREF_PARAMS.PARAM1, MySQLDSL.values(BIDS_HREF_PARAMS.PARAM1))
                    .set(BIDS_HREF_PARAMS.PARAM2, MySQLDSL.values(BIDS_HREF_PARAMS.PARAM2));
        }
        insertHelper.executeIfRecordsAdded();
    }

    private void addOrUpdateBidsPhraseHistory(DSLContext context, Collection<Keyword> keywords) {
        InsertHelper<BidsPhraseidHistoryRecord> insertHelper =
                new InsertHelper<>(context, BIDS_PHRASEID_HISTORY)
                        .addAll(keywordJooqMapper, keywords);
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(BIDS_PHRASEID_HISTORY.PHRASE_ID_HISTORY,
                            MySQLDSL.values(BIDS_PHRASEID_HISTORY.PHRASE_ID_HISTORY))
                    .set(BIDS_PHRASEID_HISTORY.UPDATE_TIME, MySQLDSL.values(BIDS_PHRASEID_HISTORY.UPDATE_TIME));
        }
        insertHelper.executeIfRecordsAdded();
    }

    public void deleteKeywords(int shard, List<CampaignIdAndKeywordIdPair> keys) {
        deleteKeywords(dslContextProvider.ppc(shard).configuration(), keys);
    }

    /**
     * Удаляет ключевые слова
     *
     * @param conf конфиг
     * @param keys список пар (campaignId, bid_id)
     */
    public void deleteKeywords(Configuration conf, List<CampaignIdAndKeywordIdPair> keys) {
        //в вспомогательных таблицах ключами является связка campaignId+bid_id
        deleteFromBidsHrefParamsTable(DSL.using(conf), keys);
        deleteFromBidsManualPricesTable(DSL.using(conf), keys);
        deleteFromBidsPhraseIdHistoryTable(DSL.using(conf), keys);

        List<Long> bidIds = mapList(keys, CampaignIdAndKeywordIdPair::getKeywordId);
        deleteFromBidsTable(DSL.using(conf), bidIds);
    }

    private void deleteFromBidsHrefParamsTable(DSLContext context, List<CampaignIdAndKeywordIdPair> keys) {
        Condition deleteCondition = StreamEx.of(keys)
                .map(key -> BIDS_HREF_PARAMS.CID.eq(key.getCampaignId())
                        .and(BIDS_HREF_PARAMS.ID.eq(key.getKeywordId())))
                .reduce(Condition::or)
                .orElse(DSL.falseCondition());
        context.deleteFrom(BIDS_HREF_PARAMS)
                .where(deleteCondition)
                .execute();
    }

    private void deleteFromBidsManualPricesTable(DSLContext context, List<CampaignIdAndKeywordIdPair> keys) {
        Condition deleteCondition = StreamEx.of(keys)
                .map(key -> BIDS_MANUAL_PRICES.CID.eq(key.getCampaignId())
                        .and(BIDS_MANUAL_PRICES.ID.eq(key.getKeywordId())))
                .reduce(Condition::or)
                .orElse(DSL.falseCondition());
        context.deleteFrom(BIDS_MANUAL_PRICES)
                .where(deleteCondition)
                .execute();
    }

    /**
     * Удаляет записи из {@link ru.yandex.direct.dbschema.ppc.tables.BidsPhraseidHistory}
     */
    public int deleteFromBidsPhraseIdHistoryTable(int shard, List<CampaignIdAndKeywordIdPair> keys) {
        return deleteFromBidsPhraseIdHistoryTable(dslContextProvider.ppc(shard), keys);
    }

    private int deleteFromBidsPhraseIdHistoryTable(DSLContext context, List<CampaignIdAndKeywordIdPair> keys) {
        if (keys.isEmpty()) {
            return 0;
        }
        Condition deleteCondition = StreamEx.of(keys)
                .map(key -> BIDS_PHRASEID_HISTORY.CID.eq(key.getCampaignId())
                        .and(BIDS_PHRASEID_HISTORY.ID.eq(key.getKeywordId())))
                .reduce(Condition::or)
                .orElse(DSL.falseCondition());
        return context.deleteFrom(BIDS_PHRASEID_HISTORY)
                .where(deleteCondition)
                .execute();
    }

    private void deleteFromBidsTable(DSLContext context, List<Long> ids) {
        context.deleteFrom(BIDS)
                .where(BIDS.ID.in(ids))
                .execute();
    }

    /**
     * Обновить значения прогноза показов для ключевых фраз из списка. Прогноз обновляется только при совпадении
     * в строке id фразы и ее текста.
     * Если новый прогноз показов отличается от текущего, будет сброшен статус синхронизации фразы с БК
     *
     * @param forecastData список из фраз, их id и прогноза показов
     * @param shard        шард
     * @return количество обновленных строк.
     */
    public int updateShowsForecast(int shard, List<KeywordForecast> forecastData) {
        Field<Long> forecastCase = JooqMapperUtils.makeCaseStatement(BIDS.ID, BIDS.SHOWS_FORECAST,
                StreamEx.of(forecastData).toMap(KeywordForecast::getId, KeywordForecast::getShowsForecast));
        Field<String> keywordCase = JooqMapperUtils.makeCaseStatement(BIDS.ID, BIDS.PHRASE,
                StreamEx.of(forecastData).toMap(KeywordForecast::getId, KeywordForecast::getKeyword));
        return dslContextProvider.ppc(shard)
                .update(BIDS)
                .set(BIDS.SHOWS_FORECAST, forecastCase)
                .set(BIDS.STATUS_BS_SYNCED, DSL.decode()
                        .value(BIDS.SHOWS_FORECAST)
                        .when(forecastCase, BIDS.STATUS_BS_SYNCED)
                        .otherwise(BidsStatusbssynced.No))
                .set(BIDS.MODTIME, BIDS.MODTIME)
                .where(BIDS.ID.in(mapList(forecastData, KeywordForecast::getId)).and(BIDS.PHRASE.eq(keywordCase)))
                .execute();
    }

    public Map<Long, KeywordDeleteInfo> getKeywordDeleteInfo(int shard, List<Long> ids) {
        return dslContextProvider.ppc(shard)
                .select(BIDS.ID, BIDS.CID, BIDS.PID, CAMPAIGNS.ARCHIVED, CAMPAIGNS.UID, PHRASES.GROUP_NAME,
                        PHRASES.STATUS_MODERATE)
                .from(BIDS)
                .join(PHRASES).on(BIDS.PID.eq(PHRASES.PID))
                .join(CAMPAIGNS).on(BIDS.CID.eq(CAMPAIGNS.CID))
                .where(BIDS.ID.in(ids))
                .fetchMap(BIDS.ID, KeywordRepository::createKeywordDeleteInfo);
    }

    private static KeywordDeleteInfo createKeywordDeleteInfo(Record record) {
        return new KeywordDeleteInfo()
                .withCampaignId(record.get(BIDS.CID))
                .withAdGroupId(record.get(BIDS.PID))
                .withAdGroupName(record.get(PHRASES.GROUP_NAME))
                .withAdGroupDraft(record.get(PHRASES.STATUS_MODERATE).equals(PhrasesStatusmoderate.New))
                .withCampaignArchived(archivedFromDb(record.get(CAMPAIGNS.ARCHIVED)))
                .withOwnerUid(record.get(CAMPAIGNS.UID));
    }

    /**
     * Ищет одинаковые фразы (в рамках одной группы и с одинаковой фразой),
     * удаляет дублирующиеся (оставляет фразу c наименьшим  bids.id и у которой  bids.PhraseID > 0 )
     * и возвращает список ID фраз которые были удалены
     *
     * @param conf       конфигурация
     * @param adGroupIds список id групп, в рамках которых ищет одинаковые фразы
     * @return список удаленных ключевиков.
     */
    public List<Keyword> deduplicateKeywords(Configuration conf, Collection<Long> adGroupIds) {
        Map<Long, List<Keyword>> keywordsByAdGroupIds = getKeywordsByAdGroupIds(DSL.using(conf), adGroupIds);
        List<Keyword> keywordsToDelete = findDuplicated(keywordsByAdGroupIds);
        List<CampaignIdAndKeywordIdPair> pairsCampWithKw = StreamEx.of(keywordsToDelete)
                .map(kw -> new CampaignIdAndKeywordIdPair(kw.getCampaignId(), kw.getId()))
                .toList();
        deleteKeywords(conf, pairsCampWithKw);
        return keywordsToDelete;
    }

    private Map<Long, List<Keyword>> getKeywordsByAdGroupIds(DSLContext context, Collection<Long> adGroupIds) {
        Result<Record> result = context.select(asList(BIDS.ID, BIDS.PID, BIDS.CID, BIDS.PHRASE, BIDS.PHRASE_ID))
                .from(BIDS)
                .where(BIDS.PID.in(adGroupIds)).fetch();
        return StreamEx.of(result)
                .map(keywordJooqMapper::fromDb)
                .groupingBy(Keyword::getAdGroupId);
    }

    /**
     * Поиск одинаковых фраз в рамках одной группы
     */
    private List<Keyword> findDuplicated(Map<Long, List<Keyword>> keywordsByAdGroupIds) {
        List<Keyword> keywordsToDelete = new ArrayList<>();
        keywordsByAdGroupIds.forEach((adGroupId, keywords) -> {
            Map<String, List<Keyword>> keywordMaps = StreamEx.of(keywords).groupingBy(Keyword::getPhrase);
            keywordMaps.forEach((key, duplicateKeywords) -> {
                //если нашлись дубли по фразе, выбираем какие удалять
                if (duplicateKeywords.size() > 1) {
                    duplicateKeywords.sort(PHRASE_ID_KEYWORD_ID_COMPARATOR);
                    Optional<Keyword> firstDuplicateWithAutotargeting = StreamEx.of(duplicateKeywords)
                            .filter(Keyword::getIsAutotargeting)
                            .findFirst();
                    Optional<Keyword> firstDuplicateWithoutAutotargeting = StreamEx.of(duplicateKeywords)
                            .filter(keyword -> !keyword.getIsAutotargeting())
                            .findFirst();
                    firstDuplicateWithAutotargeting.ifPresent(duplicateKeywords::remove);
                    firstDuplicateWithoutAutotargeting.ifPresent(duplicateKeywords::remove);
                    keywordsToDelete.addAll(duplicateKeywords);
                }
            });
        });
        return keywordsToDelete;
    }

    public Set<Long> getAdGroupIdsByKeywordsIds(int shard, ClientId clientId, Collection<Long> keywordIds) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(BIDS.PID)
                .from(BIDS)
                .join(CAMPAIGNS)
                .on(CAMPAIGNS.CID.eq(BIDS.CID))
                .where(BIDS.ID.in(keywordIds))
                .and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .fetchSet(BIDS.PID);
    }

    public Map<Long, PhraseIdHistoryInfo> getInfoForPhraseIdHistory(int shard, ClientId clientId,
                                                                    List<Long> keywordIds) {
        Map<Long, PhraseIdHistoryInfo> result = new HashMap<>();
        dslContextProvider.ppc(shard)
                .select(BIDS.ID, BIDS.PHRASE_ID, BIDS_PHRASEID_HISTORY.PHRASE_ID_HISTORY, CAMPAIGNS.ORDER_ID,
                        PHRASES.PID, BANNERS.BID, BANNERS.BANNER_ID, BANNER_IMAGES.BID, BANNER_IMAGES.BANNER_ID)
                .from(BIDS)
                .join(PHRASES).on(PHRASES.PID.eq(BIDS.PID))
                .leftJoin(BANNERS).on(BANNERS.PID.eq(PHRASES.PID))
                .leftJoin(BANNER_IMAGES).on(BANNER_IMAGES.BID.eq(BANNERS.BID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .leftJoin(BIDS_PHRASEID_HISTORY)
                .on(BIDS_PHRASEID_HISTORY.ID.eq(BIDS.ID).and(BIDS_PHRASEID_HISTORY.CID.eq(CAMPAIGNS.CID)))
                .where(BIDS.ID.in(keywordIds).and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())))
                .fetch()
                .forEach(record -> addRecordToHistory(record, result));
        return result;
    }

    private void addRecordToHistory(Record record, Map<Long, PhraseIdHistoryInfo> historyMap) {
        PhraseIdHistoryInfo phraseIdHistoryInfo = historyMap.computeIfAbsent(record.get(BIDS.ID), keywordId -> {
                    PhraseIdHistoryInfo historyInfo = new PhraseIdHistoryInfo()
                            .withPhraseId(record.get(BIDS.PHRASE_ID).toBigInteger())
                            .withOrderId(record.get(CAMPAIGNS.ORDER_ID))
                            .withAdGroupId(record.get(PHRASES.PID));
                    String oldHistory = record.get(BIDS_PHRASEID_HISTORY.PHRASE_ID_HISTORY);
                    if (oldHistory != null) {
                        historyInfo.withOldHistory(History.parse(oldHistory));
                    }
                    return historyInfo;
                }
        );
        Long bannerId = record.get(BANNERS.BID);
        if (bannerId != null) {
            phraseIdHistoryInfo.getAdGroupBanners().put(bannerId, record.get(BANNERS.BANNER_ID));
            Long bannerImageId = record.get(BANNER_IMAGES.BID);
            if (bannerImageId != null) {
                phraseIdHistoryInfo.getAdGroupImageBanners().put(bannerImageId, record.get(BANNER_IMAGES.BANNER_ID));
            }
        }
    }

    public List<FindAndReplaceKeyword> getFindAndReplaceKeywordByIds(int shard, ClientId clientId,
                                                                     Collection<Long> keywordIds) {
        List<Keyword> keywordsByIds =
                getKeywordsByIds(dslContextProvider.ppc(shard), clientId, keywordIds, emptyList(), emptyList(),
                        findAndReplaceKeywordFieldsToRead, false);

        return mapList(keywordsByIds, FindAndReplaceKeyword.class::cast);
    }

    public Map<String, Long> getKeywordIdByPhrase(List<String> keywordTexts, Long campaignId, ClientId clientId) {
        var shard = shardHelper.getShardByClientId(clientId);

        return dslContextProvider.ppc(shard)
                .select(BIDS.ID, BIDS.PHRASE)
                .from(BIDS.forceIndex(Indexes.BIDS_CID.getName()))
                .leftJoin(CAMPAIGNS)
                .on(CAMPAIGNS.CID.eq(BIDS.CID)
                        .and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())))
                .where(BIDS.PHRASE.in(keywordTexts)
                        .and(BIDS.CID.eq(campaignId)))
                .fetchMap(BIDS.PHRASE, BIDS.ID);
    }

    /**
     * Обновить PhraseID в таблице.
     */
    public void updatePhraseId(int shard, Collection<ChangedPhraseIdInfo> updateInfo) {
        Iterables.partition(updateInfo, UPDATE_PHRASEID_CHUNK_SIZE).forEach(chunk -> updatePhraseIdChunk(shard, chunk));
    }

    private void updatePhraseIdChunk(int shard, Iterable<ChangedPhraseIdInfo> updateInfo) {
        var changes = StreamEx.of(updateInfo.iterator())
                .mapToEntry(ChangedPhraseIdInfo::getId, item -> ULong.valueOf(item.getNewPhraseId()))
                .toMap();

        Field<ULong> phraseIdCase = makeCaseStatement(BIDS.ID, BIDS.PHRASE_ID, changes);
        dslContextProvider.ppc(shard)
                .update(BIDS)
                .set(BIDS.MODTIME, BIDS.MODTIME)
                .set(BIDS.PHRASE_ID, phraseIdCase)
                .where(BIDS.ID.in(changes.keySet()))
                .execute();
    }
}
