package ru.yandex.direct.core.entity.keyword.service;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.advq.query.ast.Word;
import ru.yandex.advq.query.ast.WordKind;
import ru.yandex.direct.core.entity.keyword.container.FixStopwordsResult;
import ru.yandex.direct.core.entity.keyword.container.StopwordsFixation;
import ru.yandex.direct.core.entity.keyword.exceptions.WrongKeywordType;
import ru.yandex.direct.core.entity.keyword.repository.FixationPhraseRepository;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.libs.keywordutils.helper.ParseKeywordCache;
import ru.yandex.direct.libs.keywordutils.model.AnyKeyword;
import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;
import ru.yandex.direct.libs.keywordutils.model.SingleKeyword;

import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class FixStopwordsService {

    private final StopWordService stopWordService;

    private final FixationPhraseRepository fixationPhraseRepository;

    private final ParseKeywordCache parseKeywordCache;

    private volatile List<FixationInfo> fixationsInfo = null;

    @Autowired
    public FixStopwordsService(StopWordService stopWordService,
                               FixationPhraseRepository fixationPhraseRepository,
                               ParseKeywordCache parseKeywordCache) {
        this.stopWordService = stopWordService;
        this.fixationPhraseRepository = fixationPhraseRepository;
        this.parseKeywordCache = parseKeywordCache;
    }

    /**
     * Фиксирует стоп-слова, удовлетворяющие одному из условий:
     * - стоп-слова в ключевой фразе, состоящей только из стоп-слов и чисел
     * - стоп-слово в "фиксированной фразе" - фразе из словаря: если ключевая фраза содержит в себе подфразу из словаря
     * (порядок слов и формы слов важны), то все стоп-слова внутри этой подфразы фиксируются оператором +, если они еще
     * не зафиксированы.
     * Например: фраза из словаря="мастер на час", ключевая фраза="вызвать мастер на час" -> "вызвать мастер +на час"
     */
    public FixStopwordsResult fixStopwords(KeywordWithMinuses keywordWithMinuses) {
        FixResultBuilder resultBuilder = new FixResultBuilder();
        resultBuilder.sourceKeyword = keywordWithMinuses;
        if (keywordWithMinuses.getKeyword().isQuoted()) {
            return resultBuilder.build();
        }
        if (isStopwordsAndNumbers(keywordWithMinuses)) {
            processStopwordsAndNumbers(keywordWithMinuses.getKeyword(), resultBuilder);
        } else {
            processDictionaryFixation(keywordWithMinuses.getKeyword(), resultBuilder);
        }
        fixMinusWords(keywordWithMinuses, resultBuilder);
        return resultBuilder.build();
    }

    /**
     * Проверяет, что все плюс слова являются либо стоп-словами, либо числами
     */
    private boolean isStopwordsAndNumbers(KeywordWithMinuses keyword) {
        List<AnyKeyword> keywords = keyword.getKeyword().getAllKeywords();
        return allWordsAreSingle(keywords) && allWordsAreStopwordsOrNumbers(keywords);
    }

    private boolean allWordsAreSingle(List<AnyKeyword> keywords) {
        return keywords.stream()
                .allMatch(kw -> kw instanceof SingleKeyword);
    }

    private boolean allWordsAreStopwordsOrNumbers(List<AnyKeyword> keywords) {
        return keywords.stream()
                .map(kw -> (SingleKeyword) kw)
                .allMatch(kw -> stopWordService.isStopWord(kw.getWord().getText()) || isNumber(kw));
    }

    /**
     * Обрабатывает случай, когда ключевая фраза состоит только из стоп-слов и чисел
     *
     * @param keyword - ключевая фраза, состоящая из стоп слов и чисел
     */
    private void processStopwordsAndNumbers(Keyword keyword, FixResultBuilder builder) {
        List<SingleKeyword> words = mapList(keyword.getAllKeywords(), ak -> (SingleKeyword) ak);
        boolean wasFixed = false;
        for (int i = 0; i < words.size(); ++i) {
            SingleKeyword word = words.get(i);
            if (!isNumber(word) && word.getWord().getKind() == WordKind.RAW) {
                wasFixed = true;
                words.set(i, new SingleKeyword(new Word(WordKind.PLUS, word.getWord().getText())));
            }
        }
        if (wasFixed) {
            KeywordWithMinuses destKeyword = new KeywordWithMinuses(new Keyword(false,
                    mapList(words, Function.identity())));
            builder.setKeywords(destKeyword.getKeyword().getAllKeywords());
            builder.addFixation(keyword.toString(), destKeyword.toString());
        }
    }

    /**
     * Общий случай: фиксируем стоп-слова, которые являются частью особых фраз из словаря.
     * Например, в словаре есть фраза "мастер на час", где на является стоп-словом. Тогда в ключевой фразе
     * "заказать мастер на час" зафиксируется стоп-слово "на", то есть на выходе получаем "заказать мастер +на час".
     * При поиске фразы в ключевой фразе учитывается порядок слов и словоформы, то есть в фразе "мастера на час"
     * стоп-слово "на" зафиксировано не будет.
     */
    private void processDictionaryFixation(Keyword keyword, FixResultBuilder builder) {
        getFixationsInfo();
        List<SingleKeyword> currentPhrase = mapList(keyword.getAllKeywords(),
                kw -> kw instanceof SingleKeyword ? (SingleKeyword) kw : null);
        for (FixationInfo fixation : fixationsInfo) {
            processByFixation(currentPhrase, fixation, builder);
        }
        if (builder.fixationsEmpty()) {
            builder.setKeywords(keyword.getAllKeywords());
            return;
        }
        List<AnyKeyword> restoredPhrase = mapList(currentPhrase, Function.identity());
        List<AnyKeyword> sourcePhrase = keyword.getAllKeywords();
        for (int i = 0; i < restoredPhrase.size(); ++i) {
            if (restoredPhrase.get(i) == null) {
                restoredPhrase.set(i, sourcePhrase.get(i));
            }
        }
        builder.setKeywords(restoredPhrase);
    }

    private void processByFixation(List<SingleKeyword> phrase, FixationInfo fixation, FixResultBuilder builder) {
        List<SingleKeyword> fixedPhrase = fixation.phrase;
        List<Integer> stopwordsPositions = fixation.stopwordsPositions;
        List<SingleKeyword> lowerCasePhraseWithoutOperators = mapList(phrase,
                sk -> sk == null ?
                        null :
                        new SingleKeyword(new Word(WordKind.RAW, sk.getWord().getText().toLowerCase())));

        for (int i = 0; i < phrase.size() - fixedPhrase.size() + 1; ++i) {

            List<SingleKeyword> currentSubphrase = phrase.subList(i, i + fixedPhrase.size());
            List<SingleKeyword> currentSubphraseWithoutOperators =
                    lowerCasePhraseWithoutOperators.subList(i, i + fixedPhrase.size());
            if (!fixedPhrase.equals(currentSubphraseWithoutOperators)) {
                continue;
            }

            String before = getStringPhrase(currentSubphrase);
            boolean wasChanged = false;
            for (Integer index : stopwordsPositions) {
                Word currentStopWord = currentSubphrase.get(index).getWord();
                if (currentStopWord.getKind() == WordKind.RAW) {
                    currentSubphrase.set(index, new SingleKeyword(new Word(WordKind.PLUS, currentStopWord.getText())));
                    wasChanged = true;
                }
            }
            if (wasChanged) {
                builder.addFixation(before, getStringPhrase(currentSubphrase));
            }
        }
    }

    private String getStringPhrase(List<SingleKeyword> singleKeywords) {
        return new Keyword(false, mapList(singleKeywords, Function.identity())).toString();
    }

    private boolean isNumber(SingleKeyword singleKeyword) {
        return singleKeyword.getWord().getText().matches("[\\d]+(\\.[\\d]+)?\\.?|[\\d]+(e-?[\\d]+)?\\.?");
    }

    /**
     * Фиксирует незафиксированные минус стоп-слова оператором "!"
     */
    private void fixMinusWords(KeywordWithMinuses keyword, FixResultBuilder builder) {
        List<SingleKeyword> minusWords = mapList(keyword.getMinusKeywords(),
                kw -> (SingleKeyword) kw.getAllKeywords().get(0));
        for (int i = 0; i < minusWords.size(); ++i) {
            Word currentWord = minusWords.get(i).getWord();
            if (currentWord.getKind() == WordKind.RAW && stopWordService.isStopWord(currentWord.getText())) {
                SingleKeyword newWord = new SingleKeyword(new Word(WordKind.FIXED, currentWord.getText()));
                minusWords.set(i, newWord);
                builder.addFixation("-" + currentWord.toString(), "-" + newWord.toString());
            }
        }
        builder.setMinusWords(minusWords);
    }

    private static class FixationInfo {
        List<SingleKeyword> phrase;
        List<Integer> stopwordsPositions;
    }

    private synchronized void getFixationsInfo() {
        if (fixationsInfo != null) {
            return;
        }
        List<FixationInfo> result = new ArrayList<>();
        List<Keyword> fixedPhrases = mapList(fixationPhraseRepository.getPhrases(),
                sf -> parseKeywordCache.parse(sf.getPhrase()));

        for (Keyword fixedPhrase : fixedPhrases) {
            FixationInfo fixationInfo = new FixationInfo();
            if (!fixedPhrase.getAllKeywords().stream().allMatch(ak -> ak instanceof SingleKeyword)) {
                throw new WrongKeywordType("Some word in fixed phrase " + fixedPhrase.toString()
                        + " is not a SingleKeyword");
            }
            List<SingleKeyword> castedPhrase = mapList(fixedPhrase.getAllKeywords(), ak -> ((SingleKeyword) ak));
            fixationInfo.phrase = castedPhrase;
            fixationInfo.stopwordsPositions = IntStream.range(0, castedPhrase.size())
                    .filter(i -> stopWordService.isStopWord(castedPhrase.get(i).getWord().getText()))
                    .boxed().collect(toList());
            if (!fixationInfo.stopwordsPositions.isEmpty()) {
                result.add(fixationInfo);
            }
        }
        fixationsInfo = result;
    }

    private static class FixResultBuilder {
        KeywordWithMinuses sourceKeyword;
        List<AnyKeyword> keywords = new ArrayList<>();
        List<SingleKeyword> minusWords = new ArrayList<>();
        List<StopwordsFixation> fixations = new ArrayList<>();

        public boolean fixationsEmpty() {
            return fixations.isEmpty();
        }

        public void addFixation(String before, String after) {
            fixations.add(new StopwordsFixation(before, after));
        }

        public void setKeywords(List<AnyKeyword> keywords) {
            this.keywords = keywords;
        }

        public void setMinusWords(List<SingleKeyword> minusWords) {
            this.minusWords = minusWords;
        }

        public FixStopwordsResult build() {
            KeywordWithMinuses destKeyword = fixationsEmpty() ?
                    null :
                    new KeywordWithMinuses(new Keyword(false, keywords), wrapInKeywords(minusWords));
            return new FixStopwordsResult(sourceKeyword, destKeyword, fixations);
        }

        private List<Keyword> wrapInKeywords(List<SingleKeyword> words) {
            return mapList(words, sk -> new Keyword(false, singletonList(sk)));
        }
    }
}
