package ru.yandex.direct.core.entity.keyword.service;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.auction.container.bs.KeywordBidBsAuctionData;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.keyword.container.InternalKeyword;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.processing.KeywordNormalizer;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;

import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.safeParseWithMinuses;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Класс для вычисления автоматической ставки ключевой фразы.
 * Подробности вычисления в комментариях к {@link #calcSearchAutoPrice(InternalKeyword, CampaignType, KeywordBidBsAuctionData)}
 * и {@link #calcContextAutoPrice(InternalKeyword)}
 */
@ParametersAreNonnullByDefault
public class KeywordAutoPricesCalculator {

    public static final double AUTOBROKER_MULTIPLIER = 1.3;

    private Currency clientCurrency;
    private KeywordNormalizer keywordNormalizer;
    private final Map<Long, Map<String, Keyword>> adGroupOldKeywordsByNormPhrase;
    private Map<Long, Money> adGroupCommonSearchPrice;
    private Map<Long, Money> adGroupCommonContextPrice;

    /**
     * @param keywordsBeforeOperation плоский список всех ключевых фраз из всех групп, в которые будут входить те фразы,
     *                                для которых нужно расчитать ставку.
     *                                Они нужны, чтобы можно было вычислить общую ставку среди всех фраз в группе, или
     *                                восстановить ставку удаленной фразы по ее нормальной форме.
     */
    public KeywordAutoPricesCalculator(Currency clientCurrency, Collection<Keyword> keywordsBeforeOperation,
                                       KeywordNormalizer keywordNormalizer) {
        this.clientCurrency = clientCurrency;
        this.keywordNormalizer = keywordNormalizer;

        Map<Long, List<Keyword>> keywordsByAdGroup = StreamEx.of(keywordsBeforeOperation)
                .groupingBy(Keyword::getAdGroupId);
        adGroupOldKeywordsByNormPhrase = getOldKeywordsByNormPhrase(keywordsByAdGroup);
        adGroupCommonSearchPrice = getAdGroupCommonPrices(keywordsByAdGroup, Keyword::getPrice);
        adGroupCommonContextPrice = getAdGroupCommonPrices(keywordsByAdGroup, Keyword::getPriceContext);
    }

    /**
     * Группировка старых ключевых фраз по группам объявлений и по нормальным
     * формам фраз внутри групп.
     *
     * @return мапа adGroupId -> normPhrase -> oldKeyword
     */
    private Map<Long, Map<String, Keyword>> getOldKeywordsByNormPhrase(Map<Long, List<Keyword>> keywordsByAdGroup) {
        return EntryStream.of(keywordsByAdGroup)
                .mapValues(this::mapKeywordsByRenormalizedPhrases)
                .toMap();
    }

    /**
     * Пересчитывает нормальную форму фраз и делает мапу
     * "нормальная форма" -> фраза
     * <p>
     * Пересчитывать нормальную форму важно, т.к. нормальная форма прочитанная
     * из базы может быть уже не актуальна, т.к. например поменялся нормализатор.
     * <p>
     * В результирующей мапе могут быть не все КФ в случае, если не удастся
     * распарсить фразу.
     */
    private Map<String, Keyword> mapKeywordsByRenormalizedPhrases(List<Keyword> keywords) {
        HashMap<String, Keyword> result = new HashMap<>(keywords.size());
        for (Keyword kw : keywords) {
            KeywordWithMinuses rawKeyword = safeParseWithMinuses(kw.getPhrase());
            if (rawKeyword == null) {
                continue;
            }
            KeywordWithMinuses normalizedKeyword = keywordNormalizer.normalizeKeywordWithMinuses(rawKeyword);
            result.put(normalizedKeyword.toString(), kw);
        }
        return result;
    }

    /**
     * Группировка старых ключевых фраз по группам объявлений с вычислением
     * в каждой группе единой ставки, если такая есть.
     * В результирующей мапе могут быть, таким образом, не все группы.
     *
     * @param priceExtractor функция для получения ставки из ключевой фразы
     * @return мапа adGroupId -> единая ставка
     */
    private Map<Long, Money> getAdGroupCommonPrices(Map<Long, List<Keyword>> keywordsByAdGroup,
                                                    Function<Keyword, BigDecimal> priceExtractor) {
        return EntryStream.of(keywordsByAdGroup)
                .mapValues(kws -> mapList(kws, priceExtractor))
                .mapValues(this::getCommonPrice)
                .filterValues(Objects::nonNull)
                .toMap();
    }

    /**
     * @return если все цены в коллекции {@code prices} одинаковые, возвращает ее.
     * иначе возвращает {@code null}
     */
    @Nullable
    private Money getCommonPrice(Collection<BigDecimal> prices) {
        List<Money> distinctMoneys = prices.stream()
                .filter(Objects::nonNull)
                .map(price -> Money.valueOf(price, clientCurrency.getCode()))
                .distinct()
                .collect(Collectors.toList());
        if (distinctMoneys.size() == 1) {
            return distinctMoneys.get(0);
        }
        return null;
    }

    /**
     * Вычисление автоматической поисковой ставки и приведение ее к допустимым
     * границам для валюты клиента и типа кампании.
     * <ol>
     * <li>Если среди старых фраз в группе есть фраза с такой же нормальной формой, берется ее ставка</li>
     * <li>Если у всех старых фраз одинаковая поисковая ставка, берется эта</li>
     * <li>Если есть данные аукциона для фразы ({@code auctionData != null}), берется ставка первого места
     * гарантии + 30%</li>
     * <li>Иначе, берется ставка по умолчанию для клиентской валюты</li>
     * </ol>
     */
    Money calcSearchAutoPrice(InternalKeyword keyword,
                              CampaignType campaignType,
                              @Nullable KeywordBidBsAuctionData auctionData) {
        Keyword oldKeyword = getOldKeywordByNormPhrase(keyword);
        if (oldKeyword != null && oldKeyword.getPrice() != null) {
            return Money.valueOf(oldKeyword.getPrice(), clientCurrency.getCode());
        }

        Long adGroupId = keyword.getAdGroupId();
        if (adGroupCommonSearchPrice.containsKey(adGroupId)) {
            return adGroupCommonSearchPrice.get(adGroupId);
        }

        if (auctionData != null) {
            Money autoPrice = auctionData.getGuarantee().first().getBidPrice()
                    .multiply(AUTOBROKER_MULTIPLIER).roundToAuctionStepUp();
            if (campaignType == CampaignType.CPM_DEALS || campaignType == CampaignType.CPM_BANNER) {
                return autoPrice.adjustToCPMRange();
            } else {
                return autoPrice.adjustToAuctionBidRange();
            }
        }

        return Money.valueOf(clientCurrency.getDefaultPrice(), clientCurrency.getCode());
    }

    /**
     * Вычисление автоматической ставки в сети.
     *
     * <ol>
     * <li>Если среди старых фраз в группе есть фраза с такой же нормальной формой, берется ее ставка</li>
     * <li>Если у всех старых фраз одинаковая ставка в сети, берется эта</li>
     * <li>Иначе, берется ставка по умолчанию для клиентской валюты</li>
     * </ol>
     */
    Money calcContextAutoPrice(InternalKeyword keyword) {
        Keyword oldKeyword = getOldKeywordByNormPhrase(keyword);
        if (oldKeyword != null && oldKeyword.getPriceContext() != null) {
            return Money.valueOf(oldKeyword.getPriceContext(), clientCurrency.getCode());
        }

        Long adGroupId = keyword.getAdGroupId();
        if (adGroupCommonContextPrice.containsKey(adGroupId)) {
            return adGroupCommonContextPrice.get(adGroupId);
        }

        return Money.valueOf(clientCurrency.getDefaultPrice(), clientCurrency.getCode());
    }

    /**
     * Ищет в группе старую фразу с такой же нормальной формой, что и у
     * {@code keyword}. Возвращает {@code null}, если не найдена.
     */
    @Nullable
    private Keyword getOldKeywordByNormPhrase(InternalKeyword keyword) {
        long adGroupId = keyword.getAdGroupId();
        if (adGroupOldKeywordsByNormPhrase.containsKey(adGroupId)) {
            Map<String, Keyword> oldPhrases = adGroupOldKeywordsByNormPhrase.get(adGroupId);
            String normPhrase = keyword.getParsedNormalKeyword().toString();
            if (oldPhrases.containsKey(normPhrase)) {
                return oldPhrases.get(normPhrase);
            }
        }

        return null;
    }
}
