package ru.yandex.direct.core.entity.keyword.service;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.adgroup.service.MinusKeywordPreparingTool;
import ru.yandex.direct.core.entity.auction.container.AdGroupForAuction;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordBidBsAuctionData;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordTrafaretData;
import ru.yandex.direct.core.entity.auction.exception.BsAuctionUnavailableException;
import ru.yandex.direct.core.entity.auction.service.BsAuctionService;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.bids.utils.autoprice.PlaceSearch;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.domain.model.Domain;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.model.Place;
import ru.yandex.direct.core.entity.minuskeywordspack.model.MinusKeywordsPack;
import ru.yandex.direct.core.entity.minuskeywordspack.repository.MinusKeywordsPackRepository;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.service.VcardService;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;

import static com.google.common.base.Preconditions.checkState;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.auction.utils.BsAuctionConverter.convertToPositionsAuctionData;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.nullSafetyFlatMap;
import static ru.yandex.direct.utils.FunctionalUtils.selectList;

@Service
public class KeywordBsAuctionService {

    private static final Logger logger = LoggerFactory.getLogger(KeywordBsAuctionService.class);

    private final ShardHelper shardHelper;
    private final AdGroupService adGroupService;
    private final BannerService bannerService;
    private final VcardService vcardService;
    private final CampaignService campaignService;
    private final BsAuctionService bsAuctionService;
    private final MinusKeywordsPackRepository minusKeywordsPackRepository;
    private final MinusKeywordPreparingTool minusKeywordPreparingTool;

    @Autowired
    public KeywordBsAuctionService(ShardHelper shardHelper,
                                   AdGroupService adGroupService,
                                   BannerService bannerService, VcardService vcardService, CampaignService campaignService,
                                   BsAuctionService bsAuctionService,
                                   MinusKeywordsPackRepository minusKeywordsPackRepository,
                                   MinusKeywordPreparingTool minusKeywordPreparingTool) {
        this.shardHelper = shardHelper;
        this.adGroupService = adGroupService;
        this.bannerService = bannerService;
        this.vcardService = vcardService;
        this.campaignService = campaignService;
        this.bsAuctionService = bsAuctionService;
        this.minusKeywordsPackRepository = minusKeywordsPackRepository;
        this.minusKeywordPreparingTool = minusKeywordPreparingTool;
    }

    /**
     * Для ставок из переданного списка {@code keywords} ходит в Торги за актуальными данными.
     * Возвращает список из {@link KeywordTrafaretData}, где содержится информация из Торгов
     */
    public List<KeywordTrafaretData> getTrafaretAuction(ClientId clientId, List<Keyword> keywords,
                                                        Map<Long, Campaign> campaignByIds)
            throws BsAuctionUnavailableException {
        List<AdGroupForAuction> bsAdGroups = buildAdGroupForAuctions(
                new IndexedContainer(clientId, keywords, campaignByIds));
        return bsAuctionService.getBsTrafaretResults(clientId, bsAdGroups);
    }

    /**
     * Для ставок из переданного списка {@code keywords} ходит в Торги за актуальными данными.
     * Возвращает отображение {@link Keyword} -> {@link KeywordTrafaretData}, возможно, не для всех фраз.
     * Возвращает то же самое, что {@link KeywordBsAuctionService#getTrafaretAuction(ClientId, List)}, только в виде мапы.
     * <p>
     * Обертка над методом {@link #getTrafaretAuctionMap}, которая в случае возникновения исключения
     * {@link BsAuctionUnavailableException} возвращает пустую мапу.
     */
    IdentityHashMap<Keyword, KeywordTrafaretData> getTrafaretAuctionMapSafe(ClientId clientId, List<Keyword> keywords,
                                                                            Map<Long, Campaign> campaignByIds) {
        try {
            return getTrafaretAuctionMap(clientId, keywords, campaignByIds);
        } catch (BsAuctionUnavailableException e) {
            logger.error("can't get bs auction data for all phrases", e);
            return new IdentityHashMap<>();
        }
    }

    private IdentityHashMap<Keyword, KeywordTrafaretData> getTrafaretAuctionMap(ClientId clientId,
                                                                                List<Keyword> keywords,
                                                                                Map<Long, Campaign> campaignByIds)
            throws BsAuctionUnavailableException {
        List<KeywordTrafaretData> trafaretData = getTrafaretAuction(clientId, keywords, campaignByIds);
        return StreamEx.of(trafaretData)
                .mapToEntry(KeywordTrafaretData::getKeyword, identity())
                .toCustomMap(IdentityHashMap::new);
    }

    /**
     * Преобразование ответа торгов в мапу "ключевая фраза - место"
     * исключая фразы, у которых нет цены на поиске
     */
    static IdentityHashMap<Keyword, Place> getKeywordPlaces(Collection<KeywordTrafaretData> keywordTrafaretData,
                                                            Currency currency) {
        return StreamEx.of(keywordTrafaretData)
                .filter(trafaretData -> trafaretData.getKeyword().getPrice() != null)
                .mapToEntry(KeywordTrafaretData::getKeyword, identity())
                .mapValues(trafaretData -> {
                    KeywordBidBsAuctionData bsAuctionData = convertToPositionsAuctionData(trafaretData, currency);
                    BigDecimal price = trafaretData.getKeyword().getPrice();
                    return new PlaceSearch(bsAuctionData).findPlaceByPrice(price);
                })
                .toCustomMap(IdentityHashMap::new);
    }

    /**
     * Создаёт список {@link AdGroupForAuction} для отправки клиенту Торгов на основе переданных наборов core-объектов
     * (кампаний, баннеров, групп, фраз)
     */
    private List<AdGroupForAuction> buildAdGroupForAuctions(IndexedContainer idx) {
        return EntryStream.of(idx.keywordsByAdGroupId)
                .mapKeyValue((adGroupId, keywords) ->
                        {
                            AdGroup adGroup = idx.adGroups.get(adGroupId);
                            checkState(adGroup != null, "группа с id: " + adGroupId + " должна существовать");
                            List<List<String>> adGroupLibraryMinusKeywords =
                                    mapList(adGroup.getLibraryMinusKeywordsIds(), idx.libraryMinusKeywords::get);
                            adGroup.withMinusKeywords(minusKeywordPreparingTool
                                    .mergePrivateAndLibrary(adGroup.getMinusKeywords(), adGroupLibraryMinusKeywords));

                            Campaign campaign = idx.campaignsByCampaignIds.get(adGroup.getCampaignId());
                            return AdGroupForAuction.builder()
                                    .campaign(campaign)
                                    .adGroup(adGroup)
                                    .mainBanner(idx.mainBannersByAdGroupId.get(adGroupId))
                                    .mainBannerVcard(idx.vcardsByAdGroupIds.get(adGroupId))
                                    .publisherDomain(idx.mobileContentPublisherDomainsByAdGroupIds.get(adGroupId))
                                    .storeAppId(idx.mobileContentBundleIdsByAdGroupIds.get(adGroupId))
                                    .keywords(keywords)
                                    .currency(Currencies.getCurrencies().get(campaign.getCurrency().name()))
                                    .bannerQuantity(idx.bannerQuantitiesByAdGroupIds.get(adGroupId))
                                    .build();
                        }
                )
                .toList();
    }

    /**
     * Получить для списка фраз {@code keywords} результат торгов в позиционном формате.
     *
     * @return мапа с ответами торгов, где в качестве ключа - инстансы {@code keywords}. Ответ может быть не на все КФ
     */
    public IdentityHashMap<Keyword, KeywordBidBsAuctionData> getBsAuctionData(ClientId clientId,
                                                                              List<Keyword> keywords, Currency clientCurrency,
                                                                              Map<Long, Campaign> campaignByIds) {
        List<KeywordTrafaretData> trafaretAuction = getTrafaretAuction(clientId, keywords, campaignByIds);
        List<KeywordBidBsAuctionData> auctionDataList = convertToPositionsAuctionData(trafaretAuction, clientCurrency);
        return StreamEx.of(auctionDataList)
                .mapToEntry(KeywordBidBsAuctionData::getKeyword, identity())
                .toCustomMap(IdentityHashMap::new);
    }

    /**
     * Самособирающийся контейнер информации, нужной для формирования списка групп для отправки в Торги,
     * т.е. списка экземпляров {@link AdGroupForAuction}.
     * <p>
     * Нужен, чтобы метод {@link #buildAdGroupForAuctions} принимал один аргумент, а не десяток мап.
     */
    private class IndexedContainer {
        private final Map<Long, AdGroup> adGroups;
        private final Map<Long, List<Keyword>> keywordsByAdGroupId;
        private final Map<Long, BannerWithSystemFields> mainBannersByAdGroupId;
        private final Map<Long, Integer> bannerQuantitiesByAdGroupIds;
        private final Map<Long, Domain> mobileContentPublisherDomainsByAdGroupIds;
        private final Map<Long, String> mobileContentBundleIdsByAdGroupIds;
        private final Map<Long, Campaign> campaignsByCampaignIds;
        private final Map<Long, Vcard> vcardsByAdGroupIds;
        private final Map<Long, List<String>> libraryMinusKeywords;

        IndexedContainer(ClientId clientId, List<Keyword> keywords, Map<Long, Campaign> campaignByIds) {
            keywordsByAdGroupId = StreamEx.of(keywords)
                    .mapToEntry(Keyword::getAdGroupId, identity())
                    .grouping();
            Set<Long> adGroupIds = keywordsByAdGroupId.keySet();

            adGroups = listToMap(adGroupService.getAdGroups(clientId, adGroupIds), AdGroup::getId);

            mainBannersByAdGroupId = bannerService.getMainBannerByAdGroupIds(clientId, adGroupIds);

            List<TextBanner> mainTextBanners = selectList(mainBannersByAdGroupId.values(), TextBanner.class);
            Set<Long> vcardIds = listToSet(mainTextBanners, TextBanner::getVcardId);
            Map<Long, Vcard> vcardsByIds = vcardService.getVcardsById(clientId, vcardIds);
            vcardsByAdGroupIds = StreamEx.of(mainTextBanners)
                    .mapToEntry(TextBanner::getAdGroupId, b -> vcardsByIds.get(b.getVcardId()))
                    .nonNullKeys().nonNullValues()
                    .toMap();

            bannerQuantitiesByAdGroupIds =
                    bannerService.getBannerQuantitiesByAdGroupIds(clientId, adGroupIds);

            List<Long> mobileContentAdGroupIds = StreamEx.ofValues(adGroups)
                    .filter(adGroup -> adGroup.getType() == AdGroupType.MOBILE_CONTENT)
                    .map(AdGroup::getId)
                    .toList();
            mobileContentPublisherDomainsByAdGroupIds =
                    adGroupService.getMobileContentPublisherDomains(clientId, mobileContentAdGroupIds);
            mobileContentBundleIdsByAdGroupIds =
                    adGroupService.getMobileContentAppIds(clientId, mobileContentAdGroupIds);

            campaignsByCampaignIds = campaignByIds;

            Set<Long> libraryMinusKeywordsPacksIds =
                    nullSafetyFlatMap(adGroups.values(), AdGroup::getLibraryMinusKeywordsIds, toSet());

            int shard = shardHelper.getShardByClientId(clientId);
            List<MinusKeywordsPack> minusKeywordsPacks =
                    minusKeywordsPackRepository.get(shard, clientId, libraryMinusKeywordsPacksIds);
            libraryMinusKeywords =
                    listToMap(minusKeywordsPacks, MinusKeywordsPack::getId, MinusKeywordsPack::getMinusKeywords);
        }
    }
}

