package ru.yandex.direct.core.entity.keyword.service;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import org.jooq.Configuration;

import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.keyword.container.CampaignIdAndKeywordIdPair;
import ru.yandex.direct.core.entity.keyword.container.KeywordDeleteInfo;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.keyword.service.validation.DeleteKeywordValidationService;
import ru.yandex.direct.core.entity.mailnotification.model.GenericEvent;
import ru.yandex.direct.core.entity.mailnotification.model.KeywordEvent;
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.operationwithid.AbstractOperationWithId;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

public class KeywordDeleteOperation extends AbstractOperationWithId {

    private final KeywordRepository keywordRepository;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final MailNotificationEventService mailNotificationEventService;
    private final DeleteKeywordValidationService deleteKeywordValidationService;
    private final LogPriceService logPriceService;
    private final DslContextProvider dslContextProvider;
    private final List<Keyword> existingKeywords;
    private final long operatorUid;
    private final ClientId clientId;
    private final int shard;

    private List<Keyword> existingKeywordsForDelete;
    private List<Keyword> validKeywords;
    private Map<Long, KeywordDeleteInfo> keywordDeleteInfo;
    private Set<Long> validIds;

    /**
     * Список идентификаторов групп объявлений которые необходимо обновить после удаления
     * связанных ключевых фраз
     */
    private Set<Long> needUpdateAdGroupIds;
    private Set<Long> campaignIdsForUpdate;
    private List<LogPriceData> priceDataList;
    private List<GenericEvent> eventsToSend;

    public KeywordDeleteOperation(Applicability applicability, List<Long> modelIds, KeywordRepository keywordRepository,
                                  AdGroupRepository adGroupRepository,
                                  CampaignRepository campaignRepository,
                                  MailNotificationEventService mailNotificationEventService,
                                  DeleteKeywordValidationService deleteKeywordValidationService,
                                  LogPriceService logPriceService,
                                  DslContextProvider dslContextProvider,
                                  long operatorUid,
                                  ClientId clientId,
                                  int shard) {
        this(applicability, modelIds, keywordRepository, adGroupRepository, campaignRepository,
                mailNotificationEventService,
                deleteKeywordValidationService, logPriceService, dslContextProvider, operatorUid, clientId, shard,
                null);
    }

    /**
     * Использовать, если необходимо параметризовать операцию списком существующих в базе фраз
     * и использовать их для валидации, а не делать запрос в базу
     */
    public KeywordDeleteOperation(Applicability applicability, List<Long> modelIds,
                                  KeywordRepository keywordRepository,
                                  AdGroupRepository adGroupRepository,
                                  CampaignRepository campaignRepository,
                                  MailNotificationEventService mailNotificationEventService,
                                  DeleteKeywordValidationService deleteKeywordValidationService,
                                  LogPriceService logPriceService,
                                  DslContextProvider dslContextProvider,
                                  long operatorUid,
                                  ClientId clientId,
                                  int shard,
                                  List<Keyword> existingKeywords) {
        super(applicability, modelIds);

        this.keywordRepository = keywordRepository;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.mailNotificationEventService = mailNotificationEventService;
        this.deleteKeywordValidationService = deleteKeywordValidationService;
        this.logPriceService = logPriceService;
        this.dslContextProvider = dslContextProvider;
        this.operatorUid = operatorUid;
        this.clientId = clientId;
        this.shard = shard;
        this.existingKeywords = existingKeywords;
    }

    @Override
    protected ValidationResult<List<Long>, Defect> validate(List<Long> ids) {
        prepareKeywordsForDelete(ids);
        keywordDeleteInfo = getKeywordDeleteInfo();

        ValidationResult<List<Long>, Defect> vr =
                deleteKeywordValidationService.validate(ids, keywordDeleteInfo, operatorUid, clientId);
        if (!vr.hasErrors()) {
            validIds = new HashSet<>(getValidItems(vr));
        }
        return vr;
    }

    /**
     * Если операция не параметризована списком существующих в базе фраз (existingKeywords == null),
     * загружает ключевые фразы по id из базы данных, иначе фильтрует заданный список фраз, чтобы они содержали только
     * удаляемые ключевые фразы.
     *
     * @param ids id удаляемых ключевых фраз
     */
    private void prepareKeywordsForDelete(List<Long> ids) {
        Set<Long> idsSet = new HashSet<>(ids);
        if (existingKeywords == null) {
            // filter by client (права через кампанию)
            existingKeywordsForDelete = keywordRepository.getKeywordsByIds(shard, clientId, ids);
        } else {
            existingKeywordsForDelete = filterList(existingKeywords, k -> idsSet.contains(k.getId()));
        }
    }

    /**
     * Загружает из базы информацию, необходимую для удаления ключевых фраз.
     */
    private Map<Long, KeywordDeleteInfo> getKeywordDeleteInfo() {
        List<Long> existingIdsForDelete = mapList(existingKeywordsForDelete, Keyword::getId);
        return keywordRepository.getKeywordDeleteInfo(shard, existingIdsForDelete);
    }

    public Set<Long> getPotentiallyDeletedKeywordIds() {
        checkState(isPrepared(), "operation must be prepared before calling this method");
        checkState(!isExecuted(), "must not be called after execution");
        checkState(!getResult().isPresent(), "must not be called on invalid input data");
        return validIds;
    }

    @Override
    protected void beforeExecution(List<Long> ids) {
        Set<Long> idsSet = new HashSet<>(ids);
        validKeywords = filterList(existingKeywordsForDelete, k -> idsSet.contains(k.getId()));
        keywordDeleteInfo = EntryStream.of(keywordDeleteInfo)
                .filterKeys(idsSet::contains)
                .toMap();
        campaignIdsForUpdate = listToSet(keywordDeleteInfo.values(), KeywordDeleteInfo::getCampaignId);
        needUpdateAdGroupIds = listToSet(keywordDeleteInfo.values(), KeywordDeleteInfo::getAdGroupId);
        prepareLogs();
        prepareMailNotifications();
    }

    private void prepareLogs() {
        priceDataList = mapList(validKeywords, this::keywordToLog);
    }

    private LogPriceData keywordToLog(Keyword keyword) {
        return new LogPriceData(
                keyword.getCampaignId(),
                keyword.getAdGroupId(),
                keyword.getId(),
                null,
                null,
                null,
                LogPriceData.OperationType.DELETE_1
        );
    }

    /**
     * Подготавливает информацию для рассылки писем по группам, в которых удалили фразы (запись в таблицу ppc.events)
     */
    private void prepareMailNotifications() {
        eventsToSend = new ArrayList<>();
        for (Keyword keyword : validKeywords) {
            KeywordDeleteInfo deleteInfo = keywordDeleteInfo.get(keyword.getId());
            KeywordEvent<String> deletedKeywordEvent = KeywordEvent
                    .deletedKeywordEvent(operatorUid, deleteInfo.getAdGroupId(), deleteInfo, keyword.getPhrase());
            eventsToSend.add(deletedKeywordEvent);
        }
    }

    @Override
    protected void execute(List<Long> ids) {
        List<CampaignIdAndKeywordIdPair> idsForDelete = mapList(validKeywords,
                k -> new CampaignIdAndKeywordIdPair(k.getCampaignId(), k.getId()));
        dslContextProvider.ppcTransaction(shard, ctx -> {
            keywordRepository.deleteKeywords(ctx, idsForDelete);
            updateAdGroups(ctx);
            updateCampaigns(ctx);
        });
    }

    /**
     * Для групп, не являющихся черновиками:
     * - пересинхронизируем с БК
     * - пересчитать прогноз бюджета для группы и кампании
     * - если группа стала пустой, то ставим ей статусы модерации, позволяющие отправить остановку в БК
     */
    private void updateAdGroups(Configuration conf) {
        adGroupRepository.updateAfterKeywordsDeleted(conf, needUpdateAdGroupIds);
        Set<Long> nonEmptyGroupIds = adGroupRepository.getAdGroupIdsWithConditions(conf.dsl(), needUpdateAdGroupIds);
        Set<Long> emptyGroupIds = Sets.difference(needUpdateAdGroupIds, nonEmptyGroupIds);
        adGroupRepository.updateModerationStatusesAfterConditionsAreGone(conf, emptyGroupIds);
    }

    private void updateCampaigns(Configuration conf) {
        campaignRepository.setAutobudgetForecastDate(conf, campaignIdsForUpdate, null);
    }

    @Override
    protected void afterExecution(List<Long> ids) {
        logPriceService.logPrice(priceDataList, operatorUid);
        mailNotificationEventService.queueEvents(operatorUid, clientId, eventsToSend);
    }
}
