package ru.yandex.direct.core.entity.keyword.service;

import java.math.BigInteger;
import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.Map;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.keyword.model.ForecastCtr;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.repository.KeywordForecastRepository;
import ru.yandex.direct.utils.HashingUtils;

import static java.util.Collections.singletonList;

@Service
@ParametersAreNonnullByDefault
public class KeywordForecastService {
    private final KeywordForecastRepository keywordForecastRepository;

    @Autowired
    public KeywordForecastService(KeywordForecastRepository keywordForecastRepository) {
        this.keywordForecastRepository = keywordForecastRepository;
    }

    /**
     * Для переданных ключевых слов получает информацию о прогнозируемом CTR из таблицы {@code PPCDICT.FORECAST_CTR}.
     * Если по нормализованному тексту найти данные о прогнозе не удаётся, то эта ключевая фраза
     * будет отсутствовать в ответе.
     *
     * @return {@link Map} где ключи &ndash; переданный объект {@link Keyword}, а значения &ndash; данные о среднем CTR по фразе
     */
    public IdentityHashMap<Keyword, ForecastCtr> getForecast(Collection<Keyword> keywords) {
        IdentityHashMap<Keyword, BigInteger> keywordIdsWithHalfHashes = StreamEx.of(keywords)
                .mapToEntry(Keyword::getNormPhrase)
                .mapValues(HashingUtils::getMd5HalfHashUtf8)
                .toCustomMap(IdentityHashMap::new);

        return getForecastByKey(keywordIdsWithHalfHashes);
    }

    public Map<String, ForecastCtr> getForecastByPhrase(Collection<String> phrases) {
        IdentityHashMap<String, BigInteger> normPhrasesWithHalfHashes = StreamEx.of(phrases)
                // TODO(dimitrovsd): нормализовать фразы
                .mapToEntry(HashingUtils::getMd5HalfHashUtf8)
                .toCustomMap(IdentityHashMap::new);

        return getForecastByKey(normPhrasesWithHalfHashes);
    }

    private <Key> IdentityHashMap<Key, ForecastCtr> getForecastByKey(Map<Key, BigInteger> halfHashesByKey) {
        // halfHash является первичным ключом для forecast_ctr, поэтому тут безопасно можем использовать его как ключ в Map'е
        Map<BigInteger, ForecastCtr> forecastByHash =
                StreamEx.of(keywordForecastRepository.getForecastsByHalfHashes(halfHashesByKey.values()))
                        .mapToEntry(ForecastCtr::getHalfHash)
                        .invert()
                        .toMap();

        return EntryStream.of(halfHashesByKey)
                .mapValues(forecastByHash::get)
                .filterValues(Objects::nonNull)
                .toCustomMap(IdentityHashMap::new); // TODO(dimitrovsd): можно ли использовать IdentityHashMap для фраз?
    }

    /**
     * Добавляет запись в таблицу {@code PPCDICT.FORECAST_CTR} для заданной нормализованной фразы.
     * <p>
     * <b>Внимание:</b> этот метод используется только для тестов.
     */
    @SuppressWarnings("SameParameterValue")
    void addForecast(String normPhrase, Double guaranteeCtr, Double premiumCtr) {
        BigInteger halfHash = HashingUtils.getMd5HalfHashUtf8(normPhrase);
        ForecastCtr forecastCtr = new ForecastCtr()
                .withHalfHash(halfHash)
                .withGuaranteeCtr(guaranteeCtr)
                .withPremiumCtr(premiumCtr);
        keywordForecastRepository.addForecastsByHalfHash(singletonList(forecastCtr));
    }

    /**
     * Удаляет записи из таблицы {@code PPCDICT.FORECAST_CTR} для заданной нормализованной фразы.
     * <p>
     * <b>Внимание:</b> этот метод используется только для тестов.
     */
    void removeForecast(String normPhrase) {
        BigInteger halfHash = HashingUtils.getMd5HalfHashUtf8(normPhrase);
        keywordForecastRepository.removeForecastsByHalfHash(singletonList(halfHash));
    }
}
