package ru.yandex.direct.core.entity.keyword.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.model.KeywordText;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.libs.keywordutils.StopWordMatcher;
import ru.yandex.direct.libs.keywordutils.inclusion.KeywordInclusionUtils;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordForInclusion;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class KeywordInclusionService {

    private static final Logger logger = LoggerFactory.getLogger(KeywordInclusionService.class);

    private static final Pattern MINUS_KEYWORD_DELIMITER = Pattern.compile("\\s+-");

    private final ShardHelper shardHelper;
    private final KeywordRepository keywordRepository;
    private final KeywordWithLemmasFactory keywordFactory;
    private final StopWordMatcher stopWordMatcher;

    @Autowired
    public KeywordInclusionService(ShardHelper shardHelper,
                                   KeywordRepository keywordRepository,
                                   KeywordWithLemmasFactory keywordFactory,
                                   StopWordService stopWordService) {
        this.shardHelper = shardHelper;
        this.keywordRepository = keywordRepository;
        this.keywordFactory = keywordFactory;
        stopWordMatcher = stopWordService::isStopWord;
    }

    /**
     * @param keywordsByAdGroupId Набор минус-фраз, сгруппированных по {@code adGroupId}
     * @return Минус-фразы ({@link String}) сгруппированные по {@code adGroupId}, которые полностью вычитают
     * ту или иную ключевую фразу на соответствующей группе
     */
    public Map<Long, Collection<String>> getMinusKeywordsIncludedInPlusKeywordsForAdGroup(
            Map<Long, ? extends Collection<String>> keywordsByAdGroupId, ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Function<Collection<Long>, Map<Long, ? extends Collection<KeywordText>>> getKeywordsByAdGroupIdsFunc =
                ids -> keywordRepository.getKeywordTextsByAdGroupIds(shard, clientId, ids);
        return getIncludedMinusKeywords(keywordsByAdGroupId, getKeywordsByAdGroupIdsFunc);
    }

    /**
     * @param keywordsByCampaignId Набор минус-фраз, сгруппированных по {@code campaignId}
     * @return Минус-фразы ({@link String}) сгруппированные по {@code campaignId}, которые полностью вычитают
     * ту или иную ключевую фразу на соответствующей кампании
     */
    public Map<Long, Collection<String>> getMinusKeywordsIncludedInPlusKeywordsForCampaign(
            Map<Long, ? extends Collection<String>> keywordsByCampaignId, ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Function<Collection<Long>, Map<Long, ? extends Collection<KeywordText>>> getKeywordsByCampaignIds =
                ids -> keywordRepository.getKeywordTextsByCampaignIds(shard, clientId, ids);
        return getIncludedMinusKeywords(keywordsByCampaignId, getKeywordsByCampaignIds);
    }

    /**
     * @param keywords Набор фраз
     * @param minusKeywords Набор минус-фраз
     * @return Минус-фразы из {@code minusKeywords}, которые не вычитают ни одной из плюс-фраз из {@code keywords}
     */
    public List<String> getMinusKeywordsNotIncludedInPlusKeywords(Collection<String> keywords,
                                                                  Collection<String> minusKeywords) {
        List<String> plusKeywords = mapList(keywords, this::extractPlusKeyword);
        Set<String> includedMinusKeywords = KeywordInclusionUtils
                .getIncludedMinusKeywords(keywordFactory, stopWordMatcher, plusKeywords, minusKeywords);
        return StreamEx.of(minusKeywords)
                .remove(includedMinusKeywords::contains)
                .toList();
    }

    @Nonnull
    private Map<Long, Collection<String>> getIncludedMinusKeywords(
            Map<Long, ? extends Collection<String>> minusKeywordsByParentId,
            Function<Collection<Long>, Map<Long, ? extends Collection<KeywordText>>> getKeywordsObjectsByParentIdsFunc) {
        Set<Long> parentIds = minusKeywordsByParentId.keySet();
        Map<Long, ? extends Collection<KeywordText>> plusKeywordsObjectsByParentIds =
                getKeywordsObjectsByParentIdsFunc.apply(parentIds);

        // В поле Keyword.phrase (BIDS.PHRASE) записана ключевая фраза с минус-фразами.
        Function<KeywordText, String> keywordObjToStringMapper = kw -> extractPlusKeyword(kw.getPhrase());

        Map<Long, Collection<String>> plusKeywordsByParentId = EntryStream.of(plusKeywordsObjectsByParentIds)
                .mapValues(keywords -> (Collection<String>) mapList(keywords, keywordObjToStringMapper))
                .toMap();

        return getIncludedMinusKeywords(minusKeywordsByParentId, plusKeywordsByParentId).asMap();
    }

    @Nonnull
    private Multimap<Long, String> getIncludedMinusKeywords(
            Map<Long, ? extends Collection<String>> minusKeywordsByParentId,
            Map<Long, ? extends Collection<String>> plusKeywordsByParentId) {
        Multimap<Long, String> wrongMinusKeyword = ArrayListMultimap.create();

        for (Map.Entry<Long, ? extends Collection<String>> entry : minusKeywordsByParentId.entrySet()) {
            Long parentId = entry.getKey();
            Collection<String> minusKeywords = entry.getValue();
            Collection<String> plusKeywords = plusKeywordsByParentId.get(parentId);
            if (plusKeywords == null) {
                plusKeywords = new ArrayList<>();
            }

            Collection<String> wrongMinusKeywords = KeywordInclusionUtils
                    .getIncludedMinusKeywords(keywordFactory, stopWordMatcher, plusKeywords, minusKeywords);

            wrongMinusKeyword.putAll(parentId, wrongMinusKeywords);
        }
        return wrongMinusKeyword;
    }

    /**
     * @param keyword ключевая фраза с минус-фразами как она сохранена в {@link Keyword#PHRASE}
     * @return ключевая фраза отделённая от минус-фраз из {@code keyword}
     */
    private String extractPlusKeyword(String keyword) {
        return MINUS_KEYWORD_DELIMITER.split(keyword, 2)[0];
    }

    /**
     * Возвращает результат разбора переданных фраз.
     * Фразы с ошибками парсинга игнорируются.
     */
    public Collection<KeywordForInclusion> safeParseKeywords(Collection<String> keywords) {
        return StreamEx.of(keywords)
                .map(keywordFactory::safeKeywordFrom)
                .filter(Optional::isPresent)
                .map(Optional::get)
                .toList();
    }

    /**
     * @param keyword ключевая фраза с минус-фразами как она сохранена в {@link Keyword#PHRASE}
     */
    public Optional<KeywordForInclusion> safeParsePlusKeyword(String keyword) {
        return keywordFactory.safeKeywordFrom(extractPlusKeyword(keyword));
    }
}
