package ru.yandex.direct.core.entity.keyword.service;

import java.time.Duration;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;

import static ru.yandex.direct.common.db.PpcPropertyNames.MODERATE_EVERY_KEYWORD_CHANGE;

@Service
public class KeywordModerationService {

    private static final Duration PROPERTY_EXPIRE = Duration.ofMinutes(3);
    private final PpcProperty<Boolean> isModerateEveryKeywordChange;

    @Autowired
    public KeywordModerationService(PpcPropertiesSupport ppcPropertiesSupport) {
        isModerateEveryKeywordChange =
                ppcPropertiesSupport.get(MODERATE_EVERY_KEYWORD_CHANGE, PROPERTY_EXPIRE);
    }

    /**
     * Проверяет необходимость модерации {@link Keyword} после изменений
     * Отправлять на модерацию, если нормальные формы плюс фразы отличаются
     * Если включенна проперти MODERATE_EVERY_KEYWORD_CHANGE, то отправлять на модерацию при любом различии фраз
     *
     * @param oldNormalKeyword нормальная форма старой фразы
     * @param newNormalKeyword нормальная форма новой фразы
     * @param oldKeyword       старая фраза
     * @param newKeyword       новая фраза
     * @return нужна ли перемодерация. true - нужна, false - нет
     */
    public boolean checkModerateKeyword(KeywordWithMinuses oldNormalKeyword, KeywordWithMinuses newNormalKeyword,
                                        KeywordWithMinuses oldKeyword, KeywordWithMinuses newKeyword) {
        if (isModerateEveryKeywordChange.getOrDefault(false)) {
            return !oldKeyword.getKeyword().toString().equals(newKeyword.getKeyword().toString());
        } else {
            return !oldNormalKeyword.getKeyword().toString().equals(newNormalKeyword.getKeyword().toString());
        }
    }
}
