package ru.yandex.direct.core.entity.keyword.service;

import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.autobudget.service.AutobudgetAlertService;
import ru.yandex.direct.core.entity.autoprice.service.AutoPriceCampQueueService;
import ru.yandex.direct.core.entity.banner.repository.BannerRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.service.ClientLimitsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.keyword.container.KeywordsAddOperationParams;
import ru.yandex.direct.core.entity.keyword.container.KeywordsUpdateOperationParams;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.processing.KeywordNormalizer;
import ru.yandex.direct.core.entity.keyword.processing.unglue.KeywordUngluer;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.keyword.service.internal.InternalKeywordFactory;
import ru.yandex.direct.core.entity.keyword.service.validation.DeleteKeywordValidationService;
import ru.yandex.direct.core.entity.keyword.service.validation.KeywordsAddValidationService;
import ru.yandex.direct.core.entity.keyword.service.validation.UpdateKeywordValidationService;
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService;
import ru.yandex.direct.core.entity.moderation.service.ModerationService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.libs.keywordutils.helper.SingleKeywordsCache;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;

@Component
@ParametersAreNonnullByDefault
public class KeywordOperationFactory {

    /**
     * Типы групп, для которых торги не поддерживаются.
     */
    static final Set<AdGroupType> ADGROUP_TYPES_NO_AUCTION = ImmutableSet.of(AdGroupType.CPM_BANNER,
            AdGroupType.CPM_VIDEO, AdGroupType.CONTENT_PROMOTION_VIDEO, AdGroupType.CONTENT_PROMOTION);

    private final FixStopwordsService fixStopwordsService;
    private final KeywordNormalizer keywordNormalizer;
    private final KeywordUngluer keywordUngluer;
    private final KeywordsAddValidationService addValidationService;
    private final UpdateKeywordValidationService updateValidationService;
    private final DeleteKeywordValidationService deleteValidationService;
    private final ClientService clientService;
    private final ClientLimitsService clientLimitsService;
    private final ModerationService moderationService;
    private final AutoPriceCampQueueService autoPriceCampQueueService;
    private final MailNotificationEventService mailNotificationEventService;
    private final LogPriceService logPriceService;
    private final KeywordBsAuctionService keywordBsAuctionService;
    private final AutobudgetAlertService autobudgetAlertService;
    private final SingleKeywordsCache singleKeywordsCache;
    private final KeywordModerationService keywordModerationService;
    private final InternalKeywordFactory internalKeywordFactory;
    private final KeywordShowsForecastService keywordShowsForecastService;
    private final ShardHelper shardHelper;

    private final DslContextProvider dslContextProvider;
    private final KeywordRepository keywordRepository;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final AggregatedStatusesRepository aggregatedStatusesRepository;
    private final BannerRepository bannerRepository;

    public KeywordOperationFactory(
            FixStopwordsService fixStopwordsService,
            KeywordNormalizer keywordNormalizer,
            KeywordUngluer keywordUngluer,
            KeywordsAddValidationService addValidationService,
            UpdateKeywordValidationService updateValidationService,
            DeleteKeywordValidationService deleteValidationService,
            ClientService clientService,
            ClientLimitsService clientLimitsService,
            ModerationService moderationService,
            AutoPriceCampQueueService autoPriceCampQueueService,
            MailNotificationEventService mailNotificationEventService,
            LogPriceService logPriceService,
            KeywordBsAuctionService keywordBsAuctionService,
            AutobudgetAlertService autobudgetAlertService,
            SingleKeywordsCache singleKeywordsCache,
            KeywordModerationService keywordModerationService,
            InternalKeywordFactory internalKeywordFactory,
            KeywordShowsForecastService keywordShowsForecastService,
            ShardHelper shardHelper,
            DslContextProvider dslContextProvider,
            KeywordRepository keywordRepository,
            AdGroupRepository adGroupRepository,
            CampaignRepository campaignRepository,
            AggregatedStatusesRepository aggregatedStatusesRepository,
            BannerRepository bannerRepository) {
        this.fixStopwordsService = fixStopwordsService;
        this.keywordNormalizer = keywordNormalizer;
        this.keywordUngluer = keywordUngluer;
        this.addValidationService = addValidationService;
        this.updateValidationService = updateValidationService;
        this.deleteValidationService = deleteValidationService;
        this.clientService = clientService;
        this.clientLimitsService = clientLimitsService;
        this.moderationService = moderationService;
        this.autoPriceCampQueueService = autoPriceCampQueueService;
        this.mailNotificationEventService = mailNotificationEventService;
        this.logPriceService = logPriceService;
        this.keywordBsAuctionService = keywordBsAuctionService;
        this.autobudgetAlertService = autobudgetAlertService;
        this.singleKeywordsCache = singleKeywordsCache;
        this.keywordModerationService = keywordModerationService;
        this.internalKeywordFactory = internalKeywordFactory;
        this.keywordShowsForecastService = keywordShowsForecastService;
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
        this.keywordRepository = keywordRepository;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;
        this.bannerRepository = bannerRepository;
    }

    /**
     * Создание операции добавления КФ с параметрами, указанными
     * в {@code operationParams}
     *
     * @param fixedAutoPrices контейнер с фиксированными ставками, которые нужно выставить у фразы, если ставка явно
     *                        не указана, но нужна в текущей стратегии. Могут быть ставки не для всех фраз.
     *                        Должен быть не {@code null}, если в параметрах операции включен режим {@code autoPrices}
     *                        см. коммент к классу {@link KeywordsAddOperation}
     */
    public KeywordsAddOperation createKeywordsAddOperation(Applicability applicability,
                                                           KeywordsAddOperationParams operationParams,
                                                           List<Keyword> keywords,
                                                           @Nullable List<Keyword> existingKeywords,
                                                           @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
                                                           long operatorUid, ClientId clientId, long clientUid) {
        return createKeywordsAddOperation(applicability, operationParams, keywords, existingKeywords,
                fixedAutoPrices, null,
                operatorUid, clientId, clientUid);
    }

    /**
     * Создание операции добавления КФ с параметрами по умолчанию:
     * <ul>
     * <li>КФ создается в уже существующей группе</li>
     * <li>Не производится создание фраз копированием группы</li>
     * <li>Нет расклейки фраз</li>
     * <li>Переполнение группы не игнорируется</li>
     * </ul>
     *
     * @param fixedAutoPrices  контейнер с фиксированными ставками, которые нужно выставить у фразы, если ставка явно
     *                         не указана, но нужна в текущей стратегии. Если он {@code null}, то автоматический расчет
     *                         недостающих ставок производиться не будет
     *                         (см. коммент к классу {@link KeywordsAddOperation})
     * @param weakenValidation Ослабленная валидация, пропускает проверки, которые могут не проходить для
     *                         существующих фраз, используется при копировании фраз
     */
    public KeywordsAddOperation createKeywordsAddOperation(Applicability applicability,
                                                           List<Keyword> keywords,
                                                           @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
                                                           boolean weakenValidation,
                                                           long operatorUid, ClientId clientId, long clientUid) {
        KeywordsAddOperationParams operationParams = KeywordsAddOperationParams.builder()
                .withAdGroupsNonexistentOnPrepare(false)
                .withUnglueEnabled(false)
                .withIgnoreOversize(false)
                .withAutoPrices(fixedAutoPrices != null)
                .withWeakenValidation(weakenValidation)
                .build();

        return createKeywordsAddOperation(applicability, operationParams, keywords, null,
                fixedAutoPrices, null,
                operatorUid, clientId, clientUid);
    }

    /**
     * Создание операции добавления КФ с параметрами по умолчанию:
     * <ul>
     * <li>КФ создается в уже существующей группе</li>
     * <li>Не производится создание фраз копированием группы</li>
     * <li>Нет расклейки фраз</li>
     * <li>Переполнение группы не игнорируется</li>
     * <li>Не производится автоматический расчет недостающих ставок</li>
     * <li>Валидация фраз не ослаблена</li></li>
     * </ul>
     */
    public KeywordsAddOperation createKeywordsAddOperation(
            Applicability applicability,
            List<Keyword> keywords, long operatorUid, ClientId clientId, long clientUid) {
        return createKeywordsAddOperation(applicability, keywords, null, false, operatorUid, clientId, clientUid);
    }

    /**
     * Создание операции добавления КФ с параметрами, указанными
     * в {@code operationParams} и возможностью передать готовый объект для
     * вычисления автоматических ставок.
     * <p>
     * Это нужно для создания операции добавления КФ из операции изменения КФ,
     * т.к. у операции изменения больше данных, которые нужны для вычисления
     * автоматических недостающих ставок.
     *
     * @param fixedAutoPrices      контейнер с фиксированными ставками, которые нужно выставить у фразы, если ставка
     *                             явно не указана, но нужна в текущей стратегии. Могут быть ставки не для всех фраз.
     *                             Должен быть не {@code null}, если в параметрах операции включен
     *                             режим {@code autoPrices}. Cм. коммент к классу {@link KeywordsAddOperation}
     * @param autoPricesCalculator объект для вычисления автоматических недостающих ставок. Должен быть не {@code null},
     *                             если в параметрах включен режим {@code autoPrices}
     */
    KeywordsAddOperation createKeywordsAddOperation(Applicability applicability,
                                                    KeywordsAddOperationParams operationParams,
                                                    List<Keyword> keywords,
                                                    @Nullable List<Keyword> existingKeywords,
                                                    @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
                                                    @Nullable KeywordAutoPricesCalculator autoPricesCalculator,
                                                    long operatorUid, ClientId clientId, long clientUid) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        return new KeywordsAddOperation(applicability, operationParams, keywords,
                fixStopwordsService, keywordNormalizer, keywordUngluer, addValidationService, clientService,
                moderationService, autoPriceCampQueueService, mailNotificationEventService, logPriceService,
                keywordBsAuctionService, clientLimitsService, autobudgetAlertService, singleKeywordsCache,
                internalKeywordFactory, keywordShowsForecastService,
                dslContextProvider, keywordRepository, bannerRepository, adGroupRepository, campaignRepository,
                operatorUid, clientId, clientUid, shard, existingKeywords, fixedAutoPrices,
                autoPricesCalculator);
    }

    /**
     * Создание операции обновления КФ без автоматического пересчитывания
     * ставок и приоритетов автобюджета при изменении текста фразы.
     */
    public KeywordsUpdateOperation createKeywordsUpdateOperation(Applicability applicability,
                                                                 List<ModelChanges<Keyword>> modelChangesList,
                                                                 long operatorUid, ClientId clientId, long clientUid,
                                                                 boolean unglueEnabled,
                                                                 boolean disablePhraseModification) {
        KeywordsUpdateOperationParams operationParams = KeywordsUpdateOperationParams.builder()
                .withUnglueEnabled(unglueEnabled)
                .withAutoPrices(false)
                .withModificationDisabled(disablePhraseModification)
                .build();
        return createKeywordsUpdateOperation(applicability, operationParams, modelChangesList,
                null, null, null,
                operatorUid, clientId, clientUid);
    }

    /**
     * Создание операции обновления КФ с параметрами, указанными
     * в {@code operationParams} и возможностью передать готовый объект для
     * вычисления автоматических ставок при изменении текста фразы.
     * <p>
     * Это нужно для создания операции обновления КФ из операции изменения КФ,
     * т.к. у операции изменения больше данных, которые нужны для вычисления
     * автоматических недостающих ставок.
     *
     * @param fixedAutoPrices      контейнер с фиксированными ставками, которые нужно выставить у фразы, если ее текст
     *                             поменялся, новая ставка явно не указана и нужна в текущей стратегии.
     *                             Могут быть ставки не для всех фраз.
     *                             Должен быть не {@code null}, если в параметрах операции включен
     *                             режим {@code autoPrices}. Cм. коммент к классу {@link KeywordsUpdateOperation}
     * @param autoPricesCalculator объект для вычисления автоматических недостающих ставок. Должен быть не {@code null},
     *                             если в параметрах включен режим {@code autoPrices}
     */
    KeywordsUpdateOperation createKeywordsUpdateOperation(Applicability applicability,
                                                          KeywordsUpdateOperationParams operationParams,
                                                          List<ModelChanges<Keyword>> modelChangesList,
                                                          @Nullable List<Keyword> existingKeywords,
                                                          @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
                                                          @Nullable KeywordAutoPricesCalculator autoPricesCalculator,
                                                          long operatorUid, ClientId clientId, long clientUid) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return new KeywordsUpdateOperation(applicability, operationParams, modelChangesList, fixStopwordsService,
                keywordNormalizer, keywordUngluer, updateValidationService, clientService, moderationService,
                autoPriceCampQueueService, autobudgetAlertService, mailNotificationEventService, logPriceService,
                keywordBsAuctionService, keywordModerationService, singleKeywordsCache, internalKeywordFactory,
                keywordShowsForecastService,
                dslContextProvider, keywordRepository, bannerRepository,
                adGroupRepository, campaignRepository, aggregatedStatusesRepository,
                operatorUid, clientId, clientUid, shard, existingKeywords,
                fixedAutoPrices, autoPricesCalculator);
    }

    public KeywordDeleteOperation createKeywordsDeleteOperation(Applicability applicability, List<Long> idsToDelete,
                                                                long operatorUid, ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return new KeywordDeleteOperation(applicability, idsToDelete, keywordRepository, adGroupRepository,
                campaignRepository, mailNotificationEventService, deleteValidationService, logPriceService,
                dslContextProvider, operatorUid, clientId, shard);
    }

    public KeywordDeleteOperation createKeywordsDeleteOperation(Applicability applicability, List<Long> idsToDelete,
                                                                List<Keyword> existingKeywords, long operatorUid,
                                                                ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return new KeywordDeleteOperation(applicability, idsToDelete, keywordRepository, adGroupRepository,
                campaignRepository, mailNotificationEventService, deleteValidationService, logPriceService,
                dslContextProvider, operatorUid, clientId, shard, existingKeywords);
    }
}
