package ru.yandex.direct.core.entity.keyword.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.keyword.container.AddedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.model.FindAndReplaceKeyword;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.ConvertResultOperation;
import ru.yandex.direct.result.MassResult;

import static ru.yandex.direct.result.ResultConverters.massResultValueConverter;
import static ru.yandex.direct.utils.converter.Converters.nullSafeConverter;

@ParametersAreNonnullByDefault
@Service
public class KeywordService implements EntityService<Keyword, Long> {

    private final ShardHelper shardHelper;
    private final KeywordRepository keywordRepository;
    private final KeywordOperationFactory keywordOperationFactory;
    private final ClientService clientService;

    public KeywordService(ShardHelper shardHelper,
                          KeywordRepository keywordRepository,
                          KeywordOperationFactory keywordOperationFactory,
                          ClientService clientService) {
        this.shardHelper = shardHelper;
        this.keywordRepository = keywordRepository;
        this.keywordOperationFactory = keywordOperationFactory;
        this.clientService = clientService;
    }

    /**
     * Получает список {@link Keyword} по набору ID.
     */
    public List<Keyword> getKeywords(ClientId clientId, List<Long> keywordIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        return keywordRepository.getKeywordsByIds(shard, clientId, keywordIds);
    }

    /**
     * Получает списки {@link Keyword}, сгруппированные по {@code adGroupId} для указанных ID групп
     */
    public Map<Long, List<Keyword>> getKeywordsByAdGroupIds(ClientId clientId, Collection<Long> adGroupIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        return keywordRepository.getKeywordsByAdGroupIds(shard, clientId, adGroupIds);
    }

    /**
     * Получить количество ключевых фраз на группах
     *
     * @param clientId    — id клиента
     * @param campaignIds — id кампаний (нужны для запроса в bids_arc)
     * @param adGroupIds  — id групп
     * @return мапа из id групп в количество ключевых фраз.
     */
    public Map<Long, Integer> getKeywordQuantitiesByAdGroupIds(ClientId clientId,
                                                               Collection<Long> campaignIds,
                                                               Collection<Long> adGroupIds) {
        var shard = shardHelper.getShardByClientIdStrictly(clientId);
        var result = keywordRepository.getKeywordQuantitiesByAdGroupIds(shard, campaignIds,
                adGroupIds);
        // заполняем значения для групп, которые не удалось заполнить из БД
        adGroupIds.forEach(id -> result.putIfAbsent(id, 0));
        return result;
    }

    /**
     * Получает список {@link FindAndReplaceKeyword} по набору ID.
     */
    public List<FindAndReplaceKeyword> getFindAndReplaceKeyword(ClientId clientId, List<Long> keywordIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        return keywordRepository.getFindAndReplaceKeywordByIds(shard, clientId, keywordIds);
    }

    /**
     * Создает операцию добавления ключевых фраз
     *
     * @param clientId    id клиента
     * @param operatorUid id оператора, из-под которого производится добавление
     * @param clientUid   id главного пользователя клиента, ему будут отправлены письма о добавлении кдючевых фраз
     * @param keywords    добавляемые ключевые фразы
     * @return операция добавления ключевых фраз
     */
    public KeywordsAddOperation createKeywordAddOperation(ClientId clientId, Long operatorUid, Long clientUid,
                                                          List<Keyword> keywords) {
        return createKeywordAddOperation(clientId, operatorUid, clientUid, keywords, null, Applicability.PARTIAL);
    }

    /**
     * Создает операцию добавления ключевых фраз
     *
     * @param clientId        id клиента
     * @param operatorUid     id оператора, из-под которого производится добавление
     * @param clientUid       id главного пользователя клиента, ему будут отправлены письма о добавлении кдючевых фраз
     * @param keywords        добавляемые ключевые фразы
     * @param fixedAutoPrices контейнер с фиксированными ставками, которые нужно выставить у фразы, если ставка явно
     *                        не указана, но нужна в текущей стратегии.
     * @return операция добавления ключевых фраз
     */
    public KeywordsAddOperation createKeywordAddOperation(ClientId clientId, Long operatorUid, Long clientUid,
                                                          List<Keyword> keywords,
                                                          @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
                                                          Applicability applicability) {
        return keywordOperationFactory.createKeywordsAddOperation(
                applicability, keywords, fixedAutoPrices, false, operatorUid, clientId, clientUid);
    }

    @Override
    public List<Keyword> get(ClientId clientId, Long operatorUid, Collection<Long> ids) {
        return getKeywords(clientId, List.copyOf(ids));
    }

    @Override
    public MassResult<Long> add(ClientId clientId, Long operatorUid, List<Keyword> entities,
                                Applicability applicability) {
        var clientUid = clientService.getClient(clientId).getChiefUid();
        var operation = new ConvertResultOperation<>(
                createKeywordAddOperation(clientId, operatorUid, clientUid, entities, null, applicability),
                massResultValueConverter(nullSafeConverter(AddedKeywordInfo::getId)));
        return operation.prepareAndApply();
    }

    @Override
    public MassResult<Long> copy(
            CopyOperationContainer copyContainer, List<Keyword> entities, Applicability applicability) {
        var operation = new ConvertResultOperation<>(
                keywordOperationFactory.createKeywordsAddOperation(
                        applicability,
                        entities,
                        copyContainer.getShowConditionFixedAutoPrices(),
                        true,
                        copyContainer.getOperatorUid(),
                        copyContainer.getClientIdTo(),
                        copyContainer.getClientTo().getChiefUid()),
                massResultValueConverter(nullSafeConverter(AddedKeywordInfo::getId)));
        return operation.prepareAndApply();
    }
}
