package ru.yandex.direct.core.entity.keyword.service;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.advq.AdvqClient;
import ru.yandex.direct.advq.AdvqSearchOptions;
import ru.yandex.direct.advq.SearchKeywordResult;
import ru.yandex.direct.advq.SearchRequest;
import ru.yandex.direct.advq.exception.AdvqClientException;
import ru.yandex.direct.advq.search.AdvqRequestKeyword;
import ru.yandex.direct.advq.search.ParserType;
import ru.yandex.direct.advq.search.SearchItem;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.MinusKeywordPreparingTool;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.model.KeywordWithMinuses;
import ru.yandex.direct.core.entity.minuskeywordspack.model.MinusKeywordsPack;
import ru.yandex.direct.core.entity.minuskeywordspack.repository.MinusKeywordsPackRepository;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.libs.keywordutils.StopWordMatcher;
import ru.yandex.direct.libs.keywordutils.inclusion.KeywordInclusionUtils;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordForInclusion;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.nullSafetyFlatMap;

/**
 * Сервис для получения прогнозов показов по ключевым фразам
 */
@Service
@ParametersAreNonnullByDefault
public class KeywordShowsForecastService {
    public static final Duration DEFAULT_ADVQ_CALL_TIMEOUT = Duration.ofSeconds(30L);
    private static final Logger logger = LoggerFactory.getLogger(KeywordShowsForecastService.class);
    public static final Duration SKIP_CAMP_MINUS_WORDS_PROPERTY_EXPIRE = Duration.ofMinutes(1);

    private final AdvqClient client;
    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final MinusKeywordsPackRepository minusKeywordsPackRepository;
    private final MinusKeywordPreparingTool minusKeywordPreparingTool;
    private final KeywordInclusionService keywordInclusionService;
    private final StopWordMatcher stopWordMatcher;
    private final PpcProperty<Boolean> skipCampMinusWordsProp;

    public KeywordShowsForecastService(AdvqClient client,
                                       ShardHelper shardHelper,
                                       CampaignRepository campaignRepository,
                                       AdGroupRepository adGroupRepository,
                                       MinusKeywordsPackRepository minusKeywordsPackRepository,
                                       MinusKeywordPreparingTool minusKeywordPreparingTool,
                                       KeywordInclusionService keywordInclusionService,
                                       StopWordService stopWordService,
                                       PpcPropertiesSupport ppcPropertiesSupport) {
        this.client = client;
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.minusKeywordsPackRepository = minusKeywordsPackRepository;
        this.minusKeywordPreparingTool = minusKeywordPreparingTool;
        this.keywordInclusionService = keywordInclusionService;
        this.stopWordMatcher = stopWordService::isStopWord;

        skipCampMinusWordsProp =
                ppcPropertiesSupport.get(
                        PpcPropertyNames.ADVQ_SKIP_CAMP_MINUS_WORDS,
                        SKIP_CAMP_MINUS_WORDS_PROPERTY_EXPIRE
                );
    }

    /**
     * Упрощенный интерфейс для получения прогнозов по фразам.
     * Отичается от {@link #getPhrasesShows(List, Collection, Duration, ClientId)} тем, что
     * не кидает ADVQ исключение, возвращает только успешные прогнозы,
     * а если при получении прогнозов были какие-то ошибки, пишет об этом
     * предупреждения в логи.
     *
     * @param keywords    ключевые фразы, по которым нужно получить прогноз показов
     * @param advqTimeout таймаут на выполнение запроса к ADVQ. Если {@code null}, то таймаута нет
     * @return отображение keyword -> прогноз только для тех фраз, по которым смогли получить прогноз
     */
    public IdentityHashMap<Keyword, SearchItem> getPhrasesShowsSafe(
            Collection<Keyword> keywords,
            @Nullable Duration advqTimeout,
            ClientId clientId,
            Map<Long, Campaign> campaignByIds) {
        try {
            IdentityHashMap<Keyword, SearchKeywordResult> showForecasts =
                    getPhrasesShows(keywords, advqTimeout, clientId, campaignByIds);
            IdentityHashMap<Keyword, SearchItem> successfulForecasts = new IdentityHashMap<>();

            showForecasts.forEach((keyword, result) -> {
                if (result.hasErrors()) {
                    logger.warn("Errors while getting showsForecast for {}: {}", keyword.getId(),
                            result.getErrors());
                }
                if (result.getResult() != null) {
                    successfulForecasts.put(keyword, result.getResult());
                }
            });
            return successfulForecasts;
        } catch (AdvqClientException e) {
            // ошибки вызова ADVQ пробрасываются в runtime AdvqClientException-обёртке
            logger.error("Error when requesting ADVQ/search", e);
            return new IdentityHashMap<>();
        }
    }

    /**
     * @param keywords    список ключевых фраз, по которым нужно получить прогноз показов
     * @param advqTimeout таймаут на выполнение запроса к ADVQ. Если {@code null}, то таймаута нет
     * @return Отображение Keyword -> количество показов или список ошибок
     * @throws AdvqClientException кидается, если произошли ошибки вызова ADVQ
     *                             (не путать с ошибками в ответе ADVQ по фразам)
     */
    public IdentityHashMap<Keyword, SearchKeywordResult> getPhrasesShows(
            Collection<Keyword> keywords,
            @Nullable Duration advqTimeout, ClientId clientId, Map<Long, Campaign> campaignByIds)
            throws AdvqClientException {
        return getPhrasesShows(null, keywords, advqTimeout, clientId, campaignByIds);
    }

    /**
     * Возвращает прогнозы показов для ключевых фраз групп объявлений
     *
     * @param adGroups    список групп объявлений, содержащих фразы из {@code keywords}.
     *                    Может быть null, тогда группы будут взяты из базы. Группы после работы этого метода могут
     *                    быть изменены: библиотечные минус фразы будут присоединены к обычным (в {@link AdGroup#MINUS_KEYWORDS})
     * @param keywords    список ключевых фраз, по которым нужно получить прогноз показов
     * @param advqTimeout таймаут на выполнение запроса к ADVQ. Если {@code null}, то таймаута нет
     * @return Отображение Keyword -> количество показов или список ошибок
     */
    public IdentityHashMap<Keyword, SearchKeywordResult> getPhrasesShows(
            @Nullable List<AdGroup> adGroups, Collection<Keyword> keywords,
            @Nullable Duration advqTimeout, ClientId clientId, Map<Long, Campaign> campaignByIds)
            throws AdvqClientException {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Map<Long, AdGroup> adGroupByIds;
        if (adGroups == null) {
            Set<Long> adGroupIds = listToSet(keywords, Keyword::getAdGroupId);
            adGroups = adGroupRepository.getAdGroups(shard, adGroupIds);
            adGroupByIds = listToMap(adGroups, AdGroup::getId);
        } else {
            // проверим входные данные
            adGroupByIds = listToMap(adGroups, AdGroup::getId);
            List<Keyword> invalidKeywords =
                    filterList(keywords, kw -> !adGroupByIds.keySet().contains(kw.getAdGroupId()));
            checkArgument(invalidKeywords.isEmpty(),
                    "List of AdGroups must contain all of AdGroups of Keywords in list of Keywords");
        }
        mergeLibraryMinusKeywordsToAdGroupMinusKeywords(adGroups, clientId, shard);

        Multimap<Long, Keyword> adGroupKeywords = Multimaps.index(keywords, Keyword::getAdGroupId);

        Map<AdGroup, SearchRequest> requests = new IdentityHashMap<>();
        Map<Keyword, String> effectiveKeywords = new IdentityHashMap<>();

        adGroupKeywords.asMap().forEach((adGroupId, keywordsCollection) -> {
            AdGroup adGroup = adGroupByIds.get(adGroupId);
            IdentityHashMap<Keyword, String> effectiveSearchPhrases =
                    composeSearchPhrases(keywordsCollection, adGroupByIds, campaignByIds);
            effectiveKeywords.putAll(effectiveSearchPhrases);
            requests.put(adGroup,
                    SearchRequest.fromPhrases(new ArrayList<>(effectiveSearchPhrases.values()), adGroup.getGeo()));
        });

        // можем получить RuntimeException, если вызов ADVQ завершится с ошибкой
        // Используем парсер ADVQ, поскольку парсер DIRECT долго парсится на запросах с большим количеством минус-фраз (https://st.yandex-team.ru/DIRECT-70427#1505466812000)
        AdvqSearchOptions options = new AdvqSearchOptions().withParserType(ParserType.ADVQ);
        Map<SearchRequest, Map<AdvqRequestKeyword, SearchKeywordResult>> advqResults =
                client.search(requests.values(), options, advqTimeout);

        IdentityHashMap<Keyword, SearchKeywordResult> results = new IdentityHashMap<>();
        adGroupKeywords.asMap().forEach((adGroupId, keywordCollection) -> {
            AdGroup adGroup = adGroupByIds.get(adGroupId);

            SearchRequest request = requests.get(adGroup);
            Map<AdvqRequestKeyword, SearchKeywordResult> adGroupResults = advqResults.get(request);

            for (Keyword keyword : keywordCollection) {
                SearchKeywordResult searchKeywordResult = adGroupResults.get(new AdvqRequestKeyword(effectiveKeywords.get(keyword)));
                if (searchKeywordResult != null) {
                    results.put(keyword, searchKeywordResult);
                }
            }
        });
        return results;
    }

    /**
     * Для каждой группы в поле {@link AdGroup#MINUS_KEYWORDS} объединяет то, что там было (приватный набор минус слов),
     * с библиотечными наборами (извлекаются по {@link AdGroup#LIBRARY_MINUS_KEYWORDS_IDS})
     */
    private void mergeLibraryMinusKeywordsToAdGroupMinusKeywords(List<AdGroup> adGroups, ClientId clientId, int shard) {
        Set<Long> allPackIds = nullSafetyFlatMap(adGroups, AdGroup::getLibraryMinusKeywordsIds, toSet());
        Map<Long, List<String>> minusKeywordsPacks =
                EntryStream.of(minusKeywordsPackRepository.getMinusKeywordsPacks(shard, clientId, allPackIds))
                        .mapValues(MinusKeywordsPack::getMinusKeywords)
                        .toMap();

        adGroups.forEach(adGroup -> {
            List<List<String>> adGroupLibraryMinusKeywords =
                    mapList(adGroup.getLibraryMinusKeywordsIds(), minusKeywordsPacks::get);
            adGroup.withMinusKeywords(minusKeywordPreparingTool
                    .mergePrivateAndLibrary(adGroup.getMinusKeywords(), adGroupLibraryMinusKeywords));
        });
    }

    /**
     * Для переданных ключевых слов выполняет поиск допустимых минус-фраз на группе и кампании и присоединяет их
     * к ключевой фразе, получая эффективную фразу для отправки в ADVQ.
     * <p>
     * Полученные эффективные фразы имеют формат ADVQ. Это обходи проблемы с медленным разбором фраз DIRECT-парсером на стороне ADVQ
     * Подробнее: https://st.yandex-team.ru/DIRECT-70427#1505466812000
     */
    private IdentityHashMap<Keyword, String> composeSearchPhrases(Collection<Keyword> keywordsCollection,
                                                                  Map<Long, AdGroup> adGroupsById,
                                                                  Map<Long, Campaign> campaignsById) {
        Map<Long, List<Keyword>> keywordsByAdGroupIds = StreamEx.of(keywordsCollection)
                .mapToEntry(Keyword::getAdGroupId)
                .invert()
                .grouping();

        Map<Long, List<AdGroup>> adGroupsByCampaignIds = StreamEx.ofValues(adGroupsById)
                .mapToEntry(AdGroup::getCampaignId)
                .invert()
                .grouping();

        Map<Keyword, KeywordWithMinuses> modelToKeywordWithMinus = new HashMap<>();

        for (CampaignSimple campaign : campaignsById.values()) {
            Long campaignId = campaign.getId();
            List<AdGroup> adGroups = adGroupsByCampaignIds.get(campaignId);
            if (adGroups == null) {
                continue;
            }

            // соберём и распарсим минус-фразы с кампании
            List<String> campaignMinusKeywords = nvl(campaign.getMinusKeywords(), emptyList());
            Collection<KeywordForInclusion> parsedCampaignKeywords =
                    keywordInclusionService.safeParseKeywords(campaignMinusKeywords);

            for (AdGroup adGroup : adGroups) {
                if (!keywordsByAdGroupIds.containsKey(adGroup.getId())) {
                    continue;
                }

                List<String> adGroupMinusKeywords = nvl(adGroup.getMinusKeywords(), emptyList());
                // распарсим минус-фразы группы
                Collection<KeywordForInclusion> parsedAdGroupKeywords =
                        keywordInclusionService.safeParseKeywords(adGroupMinusKeywords);
                ArrayList<KeywordForInclusion> allApplicableMinusKeywords = new ArrayList<>(parsedAdGroupKeywords);
                // добавим минус-фразы кампании, если это не выключено пропертёй
                boolean allowCampaignMinusWords = !skipCampMinusWordsProp.find().orElse(false);
                if (allowCampaignMinusWords) {
                    allApplicableMinusKeywords.addAll(parsedCampaignKeywords);
                }

                List<Keyword> keywords = keywordsByAdGroupIds.get(adGroup.getId());
                Map<KeywordForInclusion, List<Keyword>> keywordsByParsedPlusPhrase = StreamEx.of(keywords)
                        .mapToEntry(Keyword::getPhrase)
                        .mapValues(keywordInclusionService::safeParsePlusKeyword)
                        .filterValues(Optional::isPresent)
                        .mapValues(Optional::get)
                        .invert()
                        .grouping();

                Map<Keyword, KeywordWithMinuses> keywordsWithMinusesByModel =
                        findKeywordsWithNotApplicableMinuses(allApplicableMinusKeywords, keywordsByParsedPlusPhrase,
                                keywords);

                modelToKeywordWithMinus.putAll(keywordsWithMinusesByModel);
            }
        }

        return EntryStream.of(modelToKeywordWithMinus)
                .mapValues(this::keywordToAdvqFormat)
                .toCustomMap(IdentityHashMap::new);
    }

    /**
     * Для переданного набора ключевых фраз и применяемых минус-фраз возвращает {@link Map} где моделям {@link Keyword}
     * соответствуют экземпляры {@link KeywordWithMinuses}.
     */
    private Map<Keyword, KeywordWithMinuses> findKeywordsWithNotApplicableMinuses(
            Collection<KeywordForInclusion> allApplicableMinusKeywords,
            Map<KeywordForInclusion, List<Keyword>> keywordsByParsedPlusPhrase,
            Collection<Keyword> allKeywords) {
        Map<Keyword, KeywordWithMinuses> keywordWithMinusesByModel = new HashMap<>();
        Collection<KeywordForInclusion> parsedKeywords = keywordsByParsedPlusPhrase.keySet();

        Map<KeywordForInclusion, List<KeywordForInclusion>> keywordsWithBadMinuses = KeywordInclusionUtils
                .getPlusKeywordsGivenEmptyResult(stopWordMatcher, parsedKeywords, allApplicableMinusKeywords);

        for (Map.Entry<KeywordForInclusion, List<KeywordForInclusion>> entry : keywordsWithBadMinuses
                .entrySet()) {
            KeywordForInclusion plusPhraseKeyword = entry.getKey();
            List<KeywordForInclusion> notApplicableMinusPhrases = entry.getValue();
            List<Keyword> originalKeywords = keywordsByParsedPlusPhrase.get(plusPhraseKeyword);
            for (Keyword originalKeyword : originalKeywords) {
                KeywordWithMinuses keywordWithMinuses =
                        KeywordWithMinuses.fromPhrase(originalKeyword.getPhrase());
                Set<KeywordForInclusion> badAdGroupMinuses = new HashSet<>(notApplicableMinusPhrases);

                for (KeywordForInclusion parsedAdGroupKeyword : allApplicableMinusKeywords) {
                    if (!badAdGroupMinuses.contains(parsedAdGroupKeyword)) {
                        // originString() даёт строковое представление результата парсинга, а не оригинальную фразу
                        // но это не должно быть критично
                        keywordWithMinuses.addMinusKeyword(parsedAdGroupKeyword.originString());
                    }
                }
                keywordWithMinusesByModel.put(originalKeyword, keywordWithMinuses);
            }
        }

        List<String> allMinusKeywordsText = mapList(allApplicableMinusKeywords, KeywordForInclusion::originString);
        // сейчас в мапе только фразы, у которых нашлись минус-фразы, их вычитающие
        // добавим те фразы, к которым применимы все минус-фразы, или при обработке которых произошли ошибки
        for (Keyword keyword : allKeywords) {
            if (!keywordWithMinusesByModel.containsKey(keyword)) {
                KeywordWithMinuses keywordWithMinuses = KeywordWithMinuses.fromPhrase(keyword.getPhrase());
                keywordWithMinuses.addMinusKeywords(allMinusKeywordsText);
                keywordWithMinusesByModel.put(keyword, keywordWithMinuses);
            }
        }
        return keywordWithMinusesByModel;
    }

    /**
     * Преобразует набор фраз в DIRECT-формате в одну строчку с запросом в ADVQ-формате.
     * Подробнее про то, зачем пытаемся использовать ADVQ-формат, тут: https://st.yandex-team.ru/DIRECT-70427#1505466812000
     */
    String keywordToAdvqFormat(KeywordWithMinuses keyword) {
        StringBuilder sb = new StringBuilder();
        sb.append(keyword.getPlusKeyword());
        for (String minusKeyword : keyword.getMinusKeywords()) {
            sb.append(" -(").append(minusKeyword).append(")");
        }
        return sb.toString();
    }
}
