package ru.yandex.direct.core.entity.keyword.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;

import ru.yandex.advq.query.IllegalQueryException;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.libs.keywordutils.model.AnyKeyword;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;
import ru.yandex.direct.model.ModelProperty;

import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.libs.keywordutils.parser.KeywordParser.parseWithMinuses;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class KeywordUtils {

    public static final Set<ModelProperty<?, ?>> SENSITIVE_PROPERTIES =
            ImmutableSet.of(Keyword.STATUS_BS_SYNCED, Keyword.STATUS_MODERATE);

    public static final String KEYWORD_WITH_PARENTHESIS_REGEX = ".*[()|].*";

    public static final String AUTOTARGETING_KEYWORD = "---autotargeting";

    public static final String AUTOTARGETING_PREFIX = AUTOTARGETING_KEYWORD + " ";

    private static final Pattern AUTOTARGETING_PREFIX_PATTERN = Pattern.compile("^\\s*" + AUTOTARGETING_KEYWORD + "\\s+");

    private KeywordUtils() {
    }

    /**
     * Проверяет значительное изменение охвата (охват либо только расширен, либо "кардинально" изменен)
     *
     * @param oldNormalKeyword нормальная форма старой фразы
     * @param newNormalKeyword нормальная форма новой фразы
     * @return значительно ли измененился охват
     */
    public static boolean isPhraseCoverageSignificantlyChanged(KeywordWithMinuses oldNormalKeyword,
                                                               KeywordWithMinuses newNormalKeyword) {
        ru.yandex.direct.libs.keywordutils.model.Keyword oldPlusKeyword = oldNormalKeyword.getKeyword();
        ru.yandex.direct.libs.keywordutils.model.Keyword newPlusKeyword = newNormalKeyword.getKeyword();
        Set<String> oldKeywords = listToSet(oldPlusKeyword.getAllKeywords(), AnyKeyword::toString);
        Set<String> newKeywords = listToSet(newPlusKeyword.getAllKeywords(), AnyKeyword::toString);
        if (oldKeywords.size() > newKeywords.size()) {
            return true;
        }

        if (!newKeywords.containsAll(oldKeywords)) {
            return true;
        }

        return oldPlusKeyword.isQuoted() && !newPlusKeyword.isQuoted();

    }

    public static Keyword cloneKeyword(Keyword keyword) {
        return keyword.copy();
    }


    public static List<Keyword> mergeNewKeywordsWithExisting(List<Keyword> newKeywordItems,
                                                             @Nullable List<Keyword> adGroupKeywords) {
        // Ищем совпадения с сохраненными фразами учитывая ---autotargeting
        Map<String, Long> existingKeywordIdsByPhrase = (adGroupKeywords != null)
                ? getKeywordIdsByPhrase(adGroupKeywords)
                : emptyMap();

        return StreamEx.of(newKeywordItems)
                .map(kw -> KeywordUtils.cloneKeyword(kw).withId(existingKeywordIdsByPhrase
                        .getOrDefault(getPhraseWithAutotargeting(kw), kw.getId())))
                .toList();
    }

    private static Map<String, Long> getKeywordIdsByPhrase(List<Keyword> adGroupKeywords) {
        return StreamEx.of(adGroupKeywords)
                .mapToEntry(KeywordUtils::getPhraseWithAutotargeting, Keyword::getId)
                .distinctKeys()
                .toMap();
    }

    public static List<Keyword> cloneKeywords(List<Keyword> keywords) {
        return mapList(keywords, KeywordUtils::cloneKeyword);
    }

    public static boolean hasAutotargetingPrefix(String keyword) {
        if (keyword == null) {
            return false;
        }
        keyword = keyword.trim();
        return AUTOTARGETING_PREFIX_PATTERN.matcher(keyword).find();
    }

    public static boolean hasNoAutotargetingPrefix(Boolean isAutotargeting) {
        if (isAutotargeting == null) {
            return true;
        }
        return !isAutotargeting;
    }

    public static String phraseWithoutAutotargetingPrefix(String keyword) {
        if (keyword == null) {
            return null;
        }
        keyword = keyword.trim();
        Matcher matcher = AUTOTARGETING_PREFIX_PATTERN.matcher(keyword);
        boolean find = matcher.find();
        return find ? keyword.substring(matcher.end()) : keyword;
    }

    public static String getPhraseWithAutotargeting(Keyword keyword) {
        if (hasNoAutotargetingPrefix(keyword.getIsAutotargeting())) {
            return keyword.getPhrase();
        }
        return AUTOTARGETING_PREFIX + keyword.getPhrase();
    }

    public static KeywordWithMinuses safeParseWithMinuses(String phrase) {
        try {
            return parseWithMinuses(phrase);
        } catch (IllegalQueryException e) {
            return null;
        }
    }

    /**
     * Разделяет фразу, содержащую скобки, на несколько фраз без скобок
     */
    @Nullable
    public static List<String> splitKeywordWithParenthesis(String phrase) {
        List<List<String>> parsedPhrase = parsePhraseWithParenthesis(phrase);
        if (parsedPhrase == null) {
            return null;
        }
        return buildPhrases(parsedPhrase);
    }

    /**
     * Добавить минус-фразы к фразе.
     */
    public static String mergePhraseWithMinusPhrases(String phrase, @Nonnull List<String> minusPhrases) {
        ru.yandex.direct.core.entity.keyword.model.KeywordWithMinuses keywordWithMinuses =
                ru.yandex.direct.core.entity.keyword.model.KeywordWithMinuses.fromPhrase(phrase);
        keywordWithMinuses.addMinusKeywords(minusPhrases);
        return keywordWithMinuses.toString();
    }

    /**
     * Разбирает фразу со скобками в структуру, удобную для дальнейшего формирования новых, разделенных фраз.
     * Структура - список списков. Один элемент во внутреннем списке значит подфразу без скобок,
     * несколько элементов - фразы внутри оператора ().
     * Например, фраза "A (B|C|D) E" превратится в {{"A "},{"B","C","D"},{" E"}}.
     * Если при разборе фраз произошла ошибка - вернется null.
     *
     * @param phrase фраза со скобками или без
     * @return структура, описывающая фразу со скобками
     */
    private static List<List<String>> parsePhraseWithParenthesis(String phrase) {
        List<List<String>> result = new ArrayList<>();
        Matcher operatorSymbolsMatcher = Pattern.compile("[()|]").matcher(phrase);
        boolean insideOperator = false;
        boolean firstAlternative = false;
        int lastIndex = 0;
        while (operatorSymbolsMatcher.find()) {
            int operatorSymbolIndex = operatorSymbolsMatcher.start();
            char operatorSymbol = phrase.charAt(operatorSymbolIndex);
            if (operatorSymbol == '(') {
                //вложенные скобки не поддерживаются
                if (insideOperator) {
                    return null;
                }
                insideOperator = true;
                firstAlternative = true;
                if (lastIndex != operatorSymbolIndex) {
                    result.add(singletonList(phrase.substring(lastIndex, operatorSymbolIndex)));
                }
                result.add(new ArrayList<>());
            } else if (operatorSymbol == '|') {
                //пайп не разрешен вне скобок, элемент оператора не может быть пустым
                if (!insideOperator) {
                    return null;
                }
                String alternativePhrase = phrase.substring(lastIndex, operatorSymbolIndex).trim();
                //не может быть пустой фразы
                if (alternativePhrase.equals("")) {
                    return null;
                }
                result.get(result.size() - 1).add(alternativePhrase);
                firstAlternative = false;
            } else if (operatorSymbol == ')') {
                //в скобках должно быть как минимум два варианта
                if (!insideOperator || firstAlternative) {
                    return null;
                }
                String alternativePhrase = phrase.substring(lastIndex, operatorSymbolIndex).trim();
                //не может быть пустой фразы
                if (alternativePhrase.equals("")) {
                    return null;
                }
                result.get(result.size() - 1).add(alternativePhrase);
                insideOperator = false;
            } else {
                throw new IllegalStateException("Unknown operator () symbol: '" + operatorSymbol + "'");
            }
            lastIndex = operatorSymbolIndex + 1;
        }
        //непарные скобки
        if (insideOperator) {
            return null;
        }
        if (lastIndex != phrase.length()) {
            result.add(singletonList(phrase.substring(lastIndex)));
        }
        return result;
    }

    /**
     * Строит из структуры, описывающей фразу со скобками тексты фраз.
     * Например, {{"A "},{"B","C","D"},{" E"}} -> {"A B E","A C E","A D E"}
     */
    private static List<String> buildPhrases(List<List<String>> parsedPhrase) {
        List<StringBuilder> result = new ArrayList<>();
        result.add(new StringBuilder());
        for (List<String> phrasePart : parsedPhrase) {
            if (phrasePart.size() == 1) {
                result.forEach(sb -> sb.append(phrasePart.get(0)));
            } else {
                int resultInitialSize = result.size();
                for (int i = 0; i < resultInitialSize; i++) {
                    String currentPhrase = result.get(i).toString();
                    result.get(i).append(" ").append(phrasePart.get(0)).append(" ");
                    for (int alternativeIndex = 1; alternativeIndex < phrasePart.size(); alternativeIndex++) {
                        String alternative = phrasePart.get(alternativeIndex);
                        result.add(new StringBuilder(currentPhrase).append(" ").append(alternative).append(" "));
                    }
                }
            }
        }
        return mapList(result, StringBuilder::toString);
    }
}
