package ru.yandex.direct.core.entity.keyword.service;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BinaryOperator;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.SetMultimap;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.TransactionalRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.advq.search.SearchItem;
import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupName;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordBidBsAuctionData;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordTrafaretData;
import ru.yandex.direct.core.entity.auction.container.bs.TrafaretBidItem;
import ru.yandex.direct.core.entity.autobudget.service.AutobudgetAlertService;
import ru.yandex.direct.core.entity.autoprice.service.AutoPriceCampQueueService;
import ru.yandex.direct.core.entity.banner.repository.BannerRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.service.ClientLimitsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.keyword.container.AdGroupInfoForKeywordAdd;
import ru.yandex.direct.core.entity.keyword.container.AddedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.container.AffectedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.container.FixStopwordsResult;
import ru.yandex.direct.core.entity.keyword.container.InternalKeyword;
import ru.yandex.direct.core.entity.keyword.container.KeywordsAddOperationParams;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.model.Place;
import ru.yandex.direct.core.entity.keyword.model.StatusModerate;
import ru.yandex.direct.core.entity.keyword.processing.KeywordNormalizer;
import ru.yandex.direct.core.entity.keyword.processing.KeywordProcessingUtils;
import ru.yandex.direct.core.entity.keyword.processing.deduplication.DuplicatingContainer;
import ru.yandex.direct.core.entity.keyword.processing.unglue.KeywordUngluer;
import ru.yandex.direct.core.entity.keyword.processing.unglue.UnglueContainer;
import ru.yandex.direct.core.entity.keyword.processing.unglue.UnglueResult;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.keyword.service.internal.InternalKeywordFactory;
import ru.yandex.direct.core.entity.keyword.service.validation.KeywordsAddValidationService;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseConstraints;
import ru.yandex.direct.core.entity.mailnotification.model.KeywordEvent;
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService;
import ru.yandex.direct.core.entity.moderation.service.ModerationService;
import ru.yandex.direct.core.entity.showcondition.Constants;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.libs.keywordutils.helper.SingleKeywordsCache;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;
import ru.yandex.direct.libs.keywordutils.model.SingleKeyword;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.AddedModelId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.add.AbstractAddOperation;
import ru.yandex.direct.operation.add.ModelsPreValidatedStep;
import ru.yandex.direct.operation.add.ModelsValidatedStep;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.auction.utils.BsAuctionConverter.convertToPositionsAuctionData;
import static ru.yandex.direct.core.entity.keyword.processing.MinusKeywordsDeduplicator.removeDuplicates;
import static ru.yandex.direct.core.entity.keyword.processing.deduplication.AdGroupKeywordDeduplicationUtils.computeDuplicates;
import static ru.yandex.direct.core.entity.keyword.service.KeywordBsAuctionService.getKeywordPlaces;
import static ru.yandex.direct.core.entity.keyword.service.KeywordOperationFactory.ADGROUP_TYPES_NO_AUCTION;
import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordDefects.duplicatedWithExisting;
import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordDefects.duplicatedWithNew;
import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordPredicates.KEYWORD_MISSING_AUTOBUDGET_PRIORITY;
import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordPredicates.KEYWORD_MISSING_CONTEXT_PRICE;
import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordPredicates.KEYWORD_MISSING_SEARCH_PRICE;
import static ru.yandex.direct.core.entity.mailnotification.model.KeywordEvent.addedKeywordEvent;
import static ru.yandex.direct.libs.keywordutils.parser.KeywordParser.parseWithMinuses;
import static ru.yandex.direct.utils.CommonUtils.compressNumbers;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.intRange;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItemsWithIndex;

/**
 * Операция добавления ключевых фраз.
 * <p>
 * В качестве результатов возвращает объекты {@link AddedKeywordInfo},
 * которые помимо id добавленного объекта так же содержат информацию
 * о преобразованиях, которые были проведены над фразами. Мапа с результатами
 * создается в начале выполнения операции и заполняется по мере выполнения действий.
 * <p>
 * <p>
 * Параметр {@code autoPrices} включает режим автоматического выставления ставок,
 * которые отсутствуют в запросе, но нужны в текущей стратегии. При этом параметр
 * {@code fixedAutoPrices} конструктора должен быть не {@code null}.
 * <p>
 * В режиме {@code autoPrices} сначала пытаемся взять фиксированную ставку из
 * контейнера {@code fixedAutoPrices}, но если там ее не нашлось, считаем
 * ставки хитрым образом на основе старых фраз в группе и торгов, с помощью
 * калькулятора {@link KeywordAutoPricesCalculator}. Калькулятор можно передать
 * снаружи через конструктор.
 * <p>
 * Также у фраз будет выставлен приоритет автобюджета по умолчанию, если
 * его нет в запросе, но он нужен в текущей стратегии.
 * <p>
 * <p>
 * Если режим {@code autoPrices} выключен, наличие нужных ставок/приоритета
 * автобюджета будут проверяться валидацией.
 */
@ParametersAreNonnullByDefault
public class KeywordsAddOperation extends AbstractAddOperation<Keyword, AddedKeywordInfo> {

    private static final Logger logger = LoggerFactory.getLogger(KeywordsAddOperation.class);

    /**
     * Таймаут на получение прогнозов из ADVQ по умолчанию.
     * Может не хватать, если много минус фраз.
     */
    private static final Duration DEFAULT_FORECAST_TIMEOUT = Duration.ofSeconds(5);

    /**
     * Таймаут на получение прогнозов в случае, если включено вычисление
     * автоставки для новых фраз. При этом ставка, возможно, будет считаться
     * на основе ответа от торгов, а для торгов важны прогнозы.
     * Если в торги передать нулевой прогноз, ставки будут завышены.
     */
    private static final Duration AUTO_PRICE_FORECAST_TIMEOUT = Duration.ofSeconds(15);

    private final FixStopwordsService fixStopwordsService;
    private final KeywordNormalizer keywordNormalizer;
    private final KeywordUngluer keywordUngluer;
    private final KeywordsAddValidationService validationService;
    private final ClientService clientService;
    private final ModerationService moderationService;
    private final AutoPriceCampQueueService autoPriceCampQueueService;
    private final MailNotificationEventService mailNotificationEventService;
    private final LogPriceService logPriceService;
    private final KeywordBsAuctionService keywordBsAuctionService;
    private final ClientLimitsService clientLimitsService;
    private final AutobudgetAlertService autobudgetAlertService;
    private final SingleKeywordsCache singleKeywordsCache;
    private final InternalKeywordFactory internalKeywordFactory;
    private final KeywordShowsForecastService keywordShowsForecastService;

    private final DslContextProvider dslContextProvider;
    private final KeywordRepository keywordRepository;
    private final BannerRepository bannerRepository;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;

    private final KeywordsAddOperationParams operationParams;

    private final long operatorUid;
    private final ClientId clientId;
    private final long clientUid;
    private final int shard;
    /**
     * Фиксированные ставки для новых фраз. Должен быть не {@code null}, если
     * в параметрах включен режим {@code autoPrices}.
     * Контейнер может содержать ставки не для всех фраз, тогда ставки будут
     * вычислены автоматически, см. коммент к классу
     */
    @Nullable
    private final ShowConditionFixedAutoPrices showConditionFixedAutoPrices;

    private Currency clientCurrency;
    private Long keywordsLimit;

    /**
     * Мапа результатов для каждой входной ключевой фразы, которые заполняются данными
     * в процессе выполнения операции. Ключами являются индексы соответствующих
     * ключевых фраз во входном списке.
     * <p>
     * На начальном этапе результаты создаются для всех успешно предвалидированных
     * элементов, но после выполнения основной валидации невалидные элементы
     * удаляются и в мапе остаются результаты только для валидных.
     * <p>
     * Важно, что к моменту вызова apply() мапа содержит результаты -
     * хоть и не до конца заполненные, - для всех валидных элементов запроса.
     * Таким образом, во время выполнения операции результаты дополняются недостающими
     * данными и отдаются в качестве возвращаемого значения метода {@link #execute(Map)}.
     */
    private Map<Integer, AddedKeywordInfo> resultMap;

    /**
     * Мапа контейнеров для каждой входной ключевой фразы, которые заполняются
     * вспомогательными данными в процессе выполнения операции. Ключами являются
     * индексы соответствующих ключевых фраз во входном списке.
     * <p>
     * На начальном этапе контейнеры создаются для всех успешно предвалидированных
     * элементов. Затем из нее удаляются элементы, не прошедшие основной этап валидации.
     * Затем из нее удаляются дубликаты существующих фраз, а для каждого набора дубликатов
     * среди новых фраз остается только один, который должен быть добавлен в репозиторий.
     * <p>
     * Ключевым моментом является то, что к моменту вызова {@link #apply()} в ней остаются
     * контейнеры только для тех фраз, которые должны быть отправлены для создания в репозиторий.
     */
    private Map<Integer, InternalKeyword> newKeywordsMap;

    /**
     * Мапа контейнеров для существующих фраз в тех группах объявлений,
     * которые затронуты валидными элементами запроса. Создается и заполняется
     * в процессе выполнения операции. Нужна для расклейки, поиска дубликатов,
     * и, возможно, чего-то еще.
     */
    private ImmutableMap<Integer, InternalKeyword> existingKeywordsMap;

    /**
     * Мапа дубликатов добавляемых фраз с существующими.
     * <p>
     * Ключами являются индексы фраз из входного списка, а значениями -
     * индексы из мапы существующих ({@link #existingKeywordsMap}).
     * <p>
     * Нужна, чтобы понимать, что определенная добавляемая фраза дублирует существующую
     * и вернуть в ответе для неё id соответствующей существующей фразы и предупреждение.
     */
    private ImmutableMap<Integer, Integer> duplicatesWithExisting;

    /**
     * Мапа дубликатов добавляемых фраз между собой.
     * <p>
     * Ключами и значениями являются индексы фраз из входного списка.
     * <p>
     * Нужна, чтобы добавить только одну фразу из набора таких фраз-дубликатов,
     * а для всех остальных фраз вернуть в ответе id фактически добавленной и предупреждение.
     */
    private ImmutableMap<Integer, Collection<Integer>> newDuplicates;

    /**
     * Когда во входном списке присутствуют дублирующиеся между собой фразы,
     * одна из дубликатов добавляется, а для остальных дубликатов
     * возвращаются ее id и предупреждение. Эта мапа в качестве ключей содержит
     * индексы проигнорированных дубликатов, а в качестве значений - индексы
     * добавленных. Она нужна, чтобы для прогнорированных дубликатов вернуть
     * id их дубликатов, которые были фактически добавлены.
     */
    private ImmutableMap<Integer, Integer> newDuplicatesLinkingMap;

    /**
     * Список изменений, которые необходимо применить к существующим фразам.
     * На данный момент изменения в существующих фразах могут быть только
     * результатом расклейки.
     */
    private ImmutableList<AppliedChanges<Keyword>> existingKeywordsToUpdate;

    /**
     * Список объектов, описывающих, какие существующие у клиента ключевые фразы
     * претерпели изменения в результате добавления новых ключевых фраз,
     * и в чем состояли эти изменения. На данный момент содержит только информацию
     * о расклейке.
     */
    private ImmutableList<AffectedKeywordInfo> affectedKeywordInfoList;

    /**
     * Дополнительные действия, которые необходимо выполнить в одной транзакции
     * с добавлением фраз для добавляемых.
     */
    private TransactionalRunnable transactionalAdditionalAddTask;

    /**
     * Дополнительные действия, которые необходимо выполнить в одной транзакции
     * с добавлением фраз для удаляемых.
     */
    private TransactionalRunnable transactionalAdditionalDeleteTask;

    /**
     * Информация о группах по индексам ключевых фраз. Используется для линковки фраз и групп на этапе
     * {@link #prepare()}. Если выставлен флаг {@link KeywordsAddOperationParams#isAdGroupsNonexistentOnPrepare()},
     * эта мапа должна быть установлена извне через {@link #setKeywordsAdGroupInfo(Map)} перед {@link #prepare()},
     * если нет, будет выведена из id групп (см. {@link #initAdGroupInfo()})
     */
    private Map<Integer, AdGroupInfoForKeywordAdd> keywordsAdGroupInfo;

    /**
     * Соответствие индексов групп из {@link #keywordsAdGroupInfo} их id в случае невыставленного
     * {@link KeywordsAddOperationParams#isAdGroupsNonexistentOnPrepare()}. Используется для вычисления индексов групп
     * для существующих фраз.
     */
    private Map<Long, Integer> adGroupIndexById;


    /**
     * Информация о торгах по индексам ключевых фраз, содержится только по индексам валидных новых КФ.
     * <p>
     * Результат не добавляется напрямую в ответ, но сохраняется на этапе выполнения операции,
     * чтобы в случае необходимости не запрашивать повторно.
     * <p>
     * Может не содержать данных для некоторых фраз, для которых не удалось получить данные торгов.
     */
    private ImmutableMap<Integer, KeywordTrafaretData> trafaretDataByIndex;

    /**
     * Калькулятор недостающих ставок. Может быть передан снаружи через
     * конструктор, тогда он будет не {@code null} до вызова
     * {@code calcAutoPrices}.
     * Если калькулятор не передали снаружи, и в параметрах операции включен
     * режим {@code autoPrices}, метод {@code calcAutoPrices} сам создаст
     * калькулятор и положит в это поле.
     */
    @Nullable
    private KeywordAutoPricesCalculator keywordAutoPricesCalculator;

    /**
     * Дополнительные действия, которые выполняются вне транзакции.
     */
    private Runnable additionalAddTask;
    private Runnable additionalDeleteTask;

    /**
     * Этот конструктор, который принимает коллекцию существующих фраз,
     * необходимо вызывать в том случае, когда заранее известно,
     * что между подготовкой/валидацией операции и ее применением существующие в базе фразы
     * изменятся. Такое может произойти, если за один пользовательский запрос
     * необходимо одновременно выполнить несколько операций, затрагивающих ключевые фразы.
     * Например, удаление, обновление и добавление. В этом случае все операции сначала
     * валидируют входные данные, подготавливают изменения, а затем последовательно применяются.
     * Если каждая операция для валидации и подготовки изменений будет опираться на состояние
     * базы данных, то к моменту их выполнения результаты валидации и подготовки будут уже
     * не актуальны. Поэтому, по завершении этапа подготовки каждой операции, создается
     * слепок ключевых фраз, актуальный на момент ее применения (то есть на момент в будущем),
     * и этот слепок передается в последующую операцию для ее валидации и подготовки.
     * <p>
     * Список ключевых фраз должен содержать все фразы из всех заграгиваемых групп объявлений.
     * Фразы из остальных групп объявлений могут присутствовать и должны игнорироваться.
     *
     * @param existingKeywords     существующие ключевые фразы к моменту вызова {@link #apply()},
     *                             список должен содержать существующие фразы из групп объявлений,
     *                             затронутых запросом.
     * @param fixedAutoPrices      контейнер с фиксированными ставками, которые нужно выставить у фразы, если ставка
     *                             явно не указана, но нужна в текущей стратегии. Могут быть ставки не для всех фраз.
     *                             Должен быть не {@code null}, если в параметрах операции включен
     *                             режим {@code autoPrices}. Cм. коммент к классу
     * @param autoPricesCalculator калькулятор недостающих ставок во фразах. Может быть {@code null}, тогда операция
     *                             создаст свой калькулятор
     */
    public KeywordsAddOperation(Applicability applicability, KeywordsAddOperationParams operationParams,
                                List<Keyword> models,
                                FixStopwordsService fixStopwordsService,
                                KeywordNormalizer keywordNormalizer,
                                KeywordUngluer keywordUngluer,
                                KeywordsAddValidationService validationService,
                                ClientService clientService,
                                ModerationService moderationService,
                                AutoPriceCampQueueService autoPriceCampQueueService,
                                MailNotificationEventService mailNotificationEventService,
                                LogPriceService logPriceService,
                                KeywordBsAuctionService keywordBsAuctionService,
                                ClientLimitsService clientLimitsService,
                                AutobudgetAlertService autobudgetAlertService,
                                SingleKeywordsCache singleKeywordsCache,
                                InternalKeywordFactory internalKeywordFactory,
                                KeywordShowsForecastService keywordShowsForecastService,
                                DslContextProvider dslContextProvider,
                                KeywordRepository keywordRepository,
                                BannerRepository bannerRepository,
                                AdGroupRepository adGroupRepository, CampaignRepository campaignRepository,
                                long operatorUid, ClientId clientId, long clientUid, int shard,
                                @Nullable List<Keyword> existingKeywords,
                                @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
                                @Nullable KeywordAutoPricesCalculator autoPricesCalculator) {
        super(applicability, models);
        this.operationParams = operationParams;
        this.fixStopwordsService = fixStopwordsService;
        this.keywordNormalizer = keywordNormalizer;
        this.keywordUngluer = keywordUngluer;
        this.validationService = validationService;
        this.clientService = clientService;
        this.moderationService = moderationService;
        this.autoPriceCampQueueService = autoPriceCampQueueService;
        this.mailNotificationEventService = mailNotificationEventService;
        this.logPriceService = logPriceService;
        this.keywordBsAuctionService = keywordBsAuctionService;
        this.clientLimitsService = clientLimitsService;
        this.autobudgetAlertService = autobudgetAlertService;
        this.singleKeywordsCache = singleKeywordsCache;
        this.internalKeywordFactory = internalKeywordFactory;
        this.keywordShowsForecastService = keywordShowsForecastService;
        this.dslContextProvider = dslContextProvider;
        this.keywordRepository = keywordRepository;
        this.bannerRepository = bannerRepository;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.operatorUid = operatorUid;
        this.clientUid = clientUid;
        this.shard = shard;
        this.clientId = clientId;

        if (operationParams.isIgnoreOversize()) {
            checkState(!operationParams.isAdGroupsNonexistentOnPrepare(),
                    "ad groups must exist on ignore oversize operation");
        }

        if (operationParams.isAutoPrices()) {
            checkNotNull(fixedAutoPrices, "must specify fixedAutoPrices in autoPrices mode");
        }
        this.showConditionFixedAutoPrices = fixedAutoPrices;
        this.keywordAutoPricesCalculator = autoPricesCalculator;

        if (existingKeywords != null) {
            setExistingKeywordsInternal(existingKeywords);
        }
        existingKeywordsToUpdate = ImmutableList.of();
        affectedKeywordInfoList = ImmutableList.of();
        trafaretDataByIndex = ImmutableMap.of();
    }

    public KeywordsAddOperation(Applicability applicability, KeywordsAddOperationParams operationParams,
                                List<Keyword> models,
                                FixStopwordsService fixStopwordsService,
                                KeywordNormalizer keywordNormalizer,
                                KeywordUngluer keywordUngluer,
                                KeywordsAddValidationService validationService,
                                ClientService clientService,
                                ModerationService moderationService,
                                AutoPriceCampQueueService autoPriceCampQueueService,
                                MailNotificationEventService mailNotificationEventService,
                                LogPriceService logPriceService,
                                KeywordBsAuctionService keywordBsAuctionService,
                                ClientLimitsService clientLimitsService,
                                AutobudgetAlertService autobudgetAlertService,
                                SingleKeywordsCache singleKeywordsCache,
                                InternalKeywordFactory internalKeywordFactory,
                                KeywordShowsForecastService keywordShowsForecastService,
                                DslContextProvider dslContextProvider,
                                KeywordRepository keywordRepository,
                                BannerRepository bannerRepository,
                                AdGroupRepository adGroupRepository, CampaignRepository campaignRepository,
                                long operatorUid, ClientId clientId, long clientUid, int shard) {
        this(applicability, operationParams, models, fixStopwordsService, keywordNormalizer,
                keywordUngluer,
                validationService, clientService, moderationService,
                autoPriceCampQueueService, mailNotificationEventService, logPriceService, keywordBsAuctionService,
                clientLimitsService, autobudgetAlertService, singleKeywordsCache, internalKeywordFactory,
                keywordShowsForecastService,
                dslContextProvider, keywordRepository, bannerRepository, adGroupRepository, campaignRepository,
                operatorUid, clientId, clientUid, shard, null, null, null);
    }

    private void setExistingKeywordsInternal(List<Keyword> existingKeywords) {
        checkArgument(!operationParams.isAdGroupsNonexistentOnPrepare(), "keywords can exist only if ad groups exist");
        Set<Long> uniqueIds = listToSet(existingKeywords, Keyword::getId);
        checkArgument(uniqueIds.size() == existingKeywords.size(),
                "existing keywords list has keywords with duplicated ids");
        this.existingKeywordsMap = ImmutableMap.copyOf(StreamEx.of(intRange(0, existingKeywords.size()))
                .mapToEntry(identity(), existingKeywords::get)
                .mapValues(internalKeywordFactory::createInternalKeyword)
                .nonNullValues()
        );
    }

    public void setKeywordsAdGroupInfo(Map<Integer, AdGroupInfoForKeywordAdd> keywordsAdGroupInfo) {
        checkState(operationParams.isAdGroupsNonexistentOnPrepare(), "ad group info can't be set if groups exist");
        checkState(!isPrepared(), "operation is already prepared");
        this.keywordsAdGroupInfo = keywordsAdGroupInfo;
        checkAdGroupInfo(intRange(0, getModels().size()));
    }

    public void setAdGroupsIds(Map<Integer, Long> adGroupIds) {
        checkState(operationParams.isAdGroupsNonexistentOnPrepare(),
                "ids can be set only when ad groups non existent on prepare");
        checkState(isPrepared(), "operation is not prepared yet");
        checkState(!isExecuted(), "operation is already executed");
        newKeywordsMap
                .forEach((index, internalKeyword) -> internalKeyword.getKeyword().setAdGroupId(adGroupIds.get(index)));
    }


    /**
     * Информация о ключевых словах, которые не влезают в группу.
     * В отличии от валидации лимита на группу в мапе нет индексов новых слов, которые потенциально могут поместиться.
     * Используется для получения КФ и замены принадлежащей группы, чтобы не превышать лимит.
     * <p>
     * Можно вызвать если флаг operationParams.adGroupsNonexistentOnPrepare не выставлен
     */
    public Map<Long, List<Integer>> getOversizeKeywordsIndexesByAdGroupId() {
        checkState(!operationParams.isAdGroupsNonexistentOnPrepare(),
                "oversize map is filled only when adgroups exist on prepare");
        checkState(isPrepareSuccessful(), "operation is not valid");

        Map<Long, List<Integer>> oversizeKeywordIndexesByAdGroupId = new HashMap<>();
        Map<Long, Integer> adGroupIdToExistingKeywordsNumber = EntryStream.of(existingKeywordsMap)
                .values()
                .map(InternalKeyword::getAdGroupId)
                .sorted()
                .runLengths()
                .mapValues(Long::intValue)
                .toMap();

        Map<Long, Integer> validKeywordsToAddCounterMap = new HashMap<>();
        newKeywordsMap.forEach((index, keyword) -> {
            Long adGroupId = keyword.getAdGroupId();
            int newKeywordsInAdGroup = validKeywordsToAddCounterMap.getOrDefault(adGroupId, 0);
            int existingKeywordsInAdGroup = adGroupIdToExistingKeywordsNumber.getOrDefault(adGroupId, 0);
            long freePlace = keywordsLimit - (existingKeywordsInAdGroup + newKeywordsInAdGroup);

            if (freePlace > 0) {
                validKeywordsToAddCounterMap.put(adGroupId, newKeywordsInAdGroup + 1);
            } else {
                oversizeKeywordIndexesByAdGroupId.computeIfAbsent(adGroupId, e -> new ArrayList<>())
                        .add(index);
            }
        });

        return oversizeKeywordIndexesByAdGroupId;
    }

    /**
     * Метод для обновления групп тех валидных ключевых слов, которые вызывали бы переполнение при сохранении.
     * cоответствие новых идентификаторов групп по индексу
     * <p>
     * Выставление новых групп доступно, если операция была создана с игнорированием ошибок переполнения.
     */
    public void setNewAdGroupIdsByIndex(Map<Integer, Long> newAdGroupIdsByIndex) {
        checkState(operationParams.isIgnoreOversize(), "set new adgroups allowed only for ignore oversize operation");
        checkState(isPrepared(), "operation is not prepared yet");
        checkState(!isExecuted(), "operation is already executed");

        checkState(newKeywordsMap.keySet().containsAll(newAdGroupIdsByIndex.keySet()),
                "new adgroup ids allowed to change only in valid keywords");

        Set<Long> newAdGroupIds = new HashSet<>(newAdGroupIdsByIndex.values());
        SetMultimap<Long, Long> keywordIdsByAdGroupIds =
                keywordRepository.getKeywordIdsByAdGroupIds(shard, clientId, newAdGroupIds);
        checkState(keywordIdsByAdGroupIds.values().isEmpty(), "some new adgroups has keywords");

        newAdGroupIdsByIndex.forEach((index, adGroupId) ->
                newKeywordsMap.get(index).getKeyword().setAdGroupId(adGroupId));
    }

    /**
     * Возвращает список объектов, описывающих, какие существующие фразы и каким образом
     * были затронуты при выполнении данной операции. Например, каким существующим фразам
     * какие минус-слова были добавлены при расклейке.
     * <p>
     * Может быть вызван только после вызова {@link #prepare()}.
     */
    public ImmutableList<AffectedKeywordInfo> getAffectedKeywordInfoList() {
        checkState(isPrepared(), "affected keywords info list is available only after preparing");
        return affectedKeywordInfoList;
    }

    /**
     * Возвращает информацию о торгах по индексам добавленных КФ.
     * <p>
     * Может быть вызван только после вызова {@link #apply()}.
     * <p>
     * Может не вернуть данные для некоторых фраз, для которых не удалось получить данные торгов.
     */
    public ImmutableMap<Integer, List<TrafaretBidItem>> getTrafaretBidsByIndexMap() {
        checkState(isExecuted(), "result of auction is available only after executing");
        return ImmutableMap.copyOf(EntryStream.of(trafaretDataByIndex)
                .mapValues(KeywordTrafaretData::getBidItems)
                .toMap());
    }

    @Override
    protected ValidationResult<List<Keyword>, Defect> preValidate(List<Keyword> models) {
        if (operationParams.isAdGroupsNonexistentOnPrepare()) {
            checkState(keywordsAdGroupInfo != null, "ad groups info must be set if ad groups don't exist");
        }
        clientCurrency = clientService.getWorkCurrency(clientId);
        return validationService.preValidate(models, clientId);
    }

    @Override
    protected void onPreValidated(ModelsPreValidatedStep<Keyword> modelsPreValidatedStep) {
        initMaps(modelsPreValidatedStep.getPreValidModelsMap());
        deduplicateMinusWords();
        unfixStopwordsInQuotes();
        fixStopwords();
        normalizeKeywords();
    }

    /**
     * Инициализация промежуточных данных и результатов для всех предварительно валидных элементов:
     * мапа {@link #newKeywordsMap} с промежуточными данными и мапа {@link #resultMap}
     * с постепенно заполняемыми результатами.
     * <p>
     * К моменту выполнения операции ({@link #apply()}) в мапе {@link #newKeywordsMap} останутся
     * только элементы, которые будут отправлены в репозиторий на добавление,
     * а в мапе {@link #resultMap} останутся только результаты для валидных
     * (после основной валидации) элементов.
     *
     * @param preValidKeywordsMap мапа успешно предварительно провалидированных входных объектов.
     */
    private void initMaps(Map<Integer, Keyword> preValidKeywordsMap) {
        newKeywordsMap = EntryStream.of(preValidKeywordsMap)
                .mapValues(kw -> new InternalKeyword(kw, parseWithMinuses(kw.getPhrase())))
                .toMap();
        resultMap = EntryStream.of(preValidKeywordsMap)
                .mapValues(kw -> new AddedKeywordInfo())
                .toMap();
    }

    /**
     * Удаляет дублирующиеся минус-слова в исходных фразах.
     * Дубликаты считаются по тому же алгоритму, что и для минус-фраз.
     */
    private void deduplicateMinusWords() {
        Map<Integer, List<ru.yandex.direct.libs.keywordutils.model.Keyword>> deduplicatedMinusWordsLists =
                EntryStream.of(newKeywordsMap)
                        .mapValues(InternalKeyword::getParsedKeyword)
                        .mapValues(kw -> removeDuplicates(singleKeywordsCache, kw.getMinusKeywords()))
                        .toMap();

        deduplicatedMinusWordsLists.forEach((index, deduplicatedMinusWords) -> {
            InternalKeyword internalKeyword = newKeywordsMap.get(index);

            KeywordWithMinuses oldKeyword = internalKeyword.getParsedKeyword();
            if (oldKeyword.getMinusKeywords().size() != deduplicatedMinusWords.size()) {
                KeywordWithMinuses resultKeyword =
                        new KeywordWithMinuses(oldKeyword.getKeyword(), deduplicatedMinusWords);
                internalKeyword.withParsedKeyword(resultKeyword);
                internalKeyword.getKeyword().setPhrase(resultKeyword.toString());

                AddedKeywordInfo addedKeywordInfo = resultMap.get(index);
                addedKeywordInfo.withResultPhrase(resultKeyword.toString());
            }
        });
    }

    /**
     * Удаляет знак "+" в исходных фразах внутри кавычек.
     */
    private void unfixStopwordsInQuotes() {
        Map<Integer, KeywordWithMinuses> unfixStopwordsResultsMap = EntryStream.of(newKeywordsMap)
                .mapValues(InternalKeyword::getParsedKeyword)
                .mapValues(KeywordProcessingUtils::cleanWordsInBracketsFromPlus)
                .toMap();


        unfixStopwordsResultsMap.forEach((index, unfixStopwords) -> {
            InternalKeyword internalKeyword = newKeywordsMap.get(index);

            internalKeyword.withParsedKeyword(unfixStopwords);
            internalKeyword.getKeyword().setPhrase(unfixStopwords.toString());

            AddedKeywordInfo addedKeywordInfo = resultMap.get(index);
            addedKeywordInfo.withResultPhrase(unfixStopwords.toString());

        });
    }

    /**
     * Фиксирует стоп-слова в исходных фразах в моделях, добавляя информацию о фиксации в результаты.
     */
    private void fixStopwords() {
        Map<Integer, FixStopwordsResult> fixStopwordsResultsMap = EntryStream.of(newKeywordsMap)
                .mapValues(InternalKeyword::getParsedKeyword)
                .mapValues(fixStopwordsService::fixStopwords)
                .toMap();

        /*
            Во входных данных заменяются даже те фразы, в которых ни одно стоп-слово не было зафиксировано;
            это приводит к тому, что криво написанные фразы, например, без пробелов в нужных местах
            или с лишними пробелами, приводятся к правильному виду.
         */
        fixStopwordsResultsMap.forEach((index, fixResult) -> {
            KeywordWithMinuses resultKeyword = fixResult.getResult();

            InternalKeyword internalKeyword = newKeywordsMap.get(index);
            internalKeyword.withParsedKeyword(resultKeyword);
            internalKeyword.getKeyword().setPhrase(resultKeyword.toString());

            AddedKeywordInfo addedKeywordInfo = resultMap.get(index);
            addedKeywordInfo.withResultPhrase(resultKeyword.toString());
            if (fixResult.isFixed()) {
                addedKeywordInfo.withFixations(fixResult.getFixations());
            }
        });
    }

    /**
     * Нормализует ключевые слова с предварительно зафиксированными по определенным
     * правилам стоп-словами и добавляет их в соответствующие контейнеры {@link InternalKeyword}.
     */
    private void normalizeKeywords() {
        newKeywordsMap.forEach((index, kw) -> {
            KeywordWithMinuses rawKw = kw.getParsedKeyword();
            KeywordWithMinuses normalizedKw = keywordNormalizer.normalizeKeywordWithMinuses(rawKw);
            kw.withParsedNormalKeyword(normalizedKw);
        });
    }

    @Override
    protected void validate(ValidationResult<List<Keyword>, Defect> preValidationResult) {
        boolean mustHavePrices = !operationParams.isAutoPrices();

        if (operationParams.isAdGroupsNonexistentOnPrepare()) {
            Set<Long> campaignsIds = listToSet(keywordsAdGroupInfo.values(), AdGroupInfoForKeywordAdd::getCampaignId);
            Map<Long, Campaign> campaigns =
                    listToMap(campaignRepository.getCampaigns(shard, campaignsIds), Campaign::getId);

            validationService.validateWithNonexistentAdGroups(preValidationResult, keywordsAdGroupInfo,
                    campaigns, newKeywordsMap, mustHavePrices);
        } else {
            validationService.validate(preValidationResult, newKeywordsMap, operatorUid, clientId, mustHavePrices,
                    operationParams.isWeakenValidation());
        }
        Map<Integer, Keyword> validKeywords = getValidItemsWithIndex(preValidationResult);

        clearMapsFromInvalidItems(validKeywords);
        initAdGroupInfo();
        initAdGroupIndexById();
        initExistingKeywordsMap();
        computeDuplicatedIndexes();
        determineNewDuplicatesToAddAndCleanNewKeywordsMap();

        keywordsLimit = clientLimitsService.massGetClientLimits(singletonList(clientId))
                .iterator().next()
                .getKeywordsCountLimitOrDefault();

        if (operationParams.isAdGroupsNonexistentOnPrepare()) {
            validationService.validateSizeWithNonexistentAdGroups(validationResult, keywordsAdGroupInfo, keywordsLimit);
        } else if (!operationParams.isIgnoreOversize()) {
            validationService
                    .validateSizePerAdGroup(validationResult, newKeywordsMap, existingKeywordsMap, keywordsLimit);
        }
    }

    /**
     * Очищает мапы с промежуточными данными по ключевым фразам и результатами от невалидных элементов.
     * <p>
     * Вызывается первый раз после основной валидации перед дедупликацией,
     * а затем второй раз после последнего этапа валидации
     *
     * @param validKeywordsMap мапа валидных элементов (после основной валидации, 2-й шаг из 3-х).
     */
    private void clearMapsFromInvalidItems(Map<Integer, Keyword> validKeywordsMap) {
        newKeywordsMap = EntryStream.of(newKeywordsMap)
                .filterKeys(validKeywordsMap::containsKey)
                .toMap();
        resultMap = EntryStream.of(resultMap)
                .filterKeys(validKeywordsMap::containsKey)
                .toMap();
    }

    /**
     * Если индексы групп не были выставлены извне, выводит их из id групп
     */
    private void initAdGroupInfo() {
        if (operationParams.isAdGroupsNonexistentOnPrepare()) {
            return;
        }
        Map<Integer, Long> adGroupIds = EntryStream.of(newKeywordsMap)
                .mapValues(InternalKeyword::getAdGroupId)
                .toMap();
        Map<Integer, Integer> keywordAdGroupIndexes = compressNumbers(adGroupIds);

        Set<Long> adGroupIdsList = new HashSet<>(adGroupIds.values());
        Map<Long, AdGroupSimple> simpleAdGroups =
                adGroupRepository.getAdGroupSimple(shard, clientId, adGroupIdsList);

        this.keywordsAdGroupInfo = EntryStream.of(adGroupIds)
                .mapToValue((keywordIndex, adGroupId) -> new AdGroupInfoForKeywordAdd(
                        keywordAdGroupIndexes.get(keywordIndex),
                        simpleAdGroups.get(adGroupId).getCampaignId(),
                        simpleAdGroups.get(adGroupId).getType()))
                .toMap();

        checkAdGroupInfo(newKeywordsMap.keySet());
    }

    /**
     * Для получения индекса группы у существующих фраз, нужна его зависимость от id
     */
    private void initAdGroupIndexById() {
        adGroupIndexById = new HashMap<>();
        newKeywordsMap.forEach((keywordIndex, internalKeyword) ->
                adGroupIndexById
                        .put(internalKeyword.getAdGroupId(), keywordsAdGroupInfo.get(keywordIndex).getAdGroupIndex())
        );
    }

    /**
     * Проверяет, что для всех добавляемых ключевых слов есть информация о группах, которая содержит id кампаний
     */
    private void checkAdGroupInfo(Collection<Integer> existingKeywordIndexes) {
        Set<Integer> keywordsWithoutCampaignId = new HashSet<>(filterList(existingKeywordIndexes,
                keywordIndex -> keywordsAdGroupInfo.get(keywordIndex) == null
                        || keywordsAdGroupInfo.get(keywordIndex).getCampaignId() == null));
        checkState(keywordsWithoutCampaignId.isEmpty(),
                "can't get ad group info or campaign ids for keywords with indexes: " + keywordsWithoutCampaignId);
    }

    /**
     * Загружает существующие ключевые фразы пользователя
     * и инициализирует для них фразы в разобранном и нормализованном виде.
     */
    private void initExistingKeywordsMap() {
        //если существующие ключевые фразы заданы извне, очищаем их от фраз из невалидных групп
        if (existingKeywordsMap != null) {
            clearExistingKeywordsFromInvalidAdGroups();
            return;
        }
        if (operationParams.isAdGroupsNonexistentOnPrepare()) {
            existingKeywordsMap = ImmutableMap.<Integer, InternalKeyword>builder().build();
            return;
        }

        Set<Long> adGroupIds = listToSet(newKeywordsMap.values(), InternalKeyword::getAdGroupId);
        List<Keyword> existingKeywords = getKeywordsByAdGroupIds(shard, clientId, adGroupIds);
        setExistingKeywordsInternal(existingKeywords);
    }

    /**
     * Очищаем от невалидных групп, чтобы не было npe в дедупликации и расклейке при попытке получить
     * по id группы индекс (индексы есть только для валидных групп)
     */
    private void clearExistingKeywordsFromInvalidAdGroups() {
        Set<Long> adGroupIds = listToSet(newKeywordsMap.values(), InternalKeyword::getAdGroupId);
        existingKeywordsMap = ImmutableMap.copyOf(EntryStream.of(existingKeywordsMap)
                .filterValues(internalKeyword -> adGroupIds.contains(internalKeyword.getAdGroupId()))
                .toMap());
    }

    /**
     * Вычисляет индексы дубликатов.
     */
    private void computeDuplicatedIndexes() {
        //в случае несуществующих в prepare групп existingKeywordsMap пустая,
        // поэтому из-за internalKeyword.getAdGroupId() равного null не будет исключения
        List<DuplicatingContainer> existingDuplContainers = EntryStream.of(existingKeywordsMap)
                .mapKeyValue((index, internalKeyword) -> new DuplicatingContainer(index,
                        adGroupIndexById.get(internalKeyword.getAdGroupId()),
                        internalKeyword.getParsedNormalKeyword(), internalKeyword.getKeyword().getIsAutotargeting()))
                .toList();

        List<DuplicatingContainer> newDuplContainers = EntryStream.of(newKeywordsMap)
                .mapKeyValue((index, internalKeyword) -> new DuplicatingContainer(index,
                        keywordsAdGroupInfo.get(index).getAdGroupIndex(),
                        internalKeyword.getParsedNormalKeyword(), internalKeyword.getKeyword().getIsAutotargeting()))
                .toList();

        duplicatesWithExisting = ImmutableMap.copyOf(computeDuplicates(newDuplContainers, existingDuplContainers));
        newDuplicates = ImmutableMap.copyOf(computeDuplicates(newDuplContainers).asMap());
    }

    /**
     * Определяет, какие ключевые фразы, продублированные в запросе,
     * будут отправлены в репозиторий на сохранение, а какие - нет.
     * <p>
     * Фразы из запроса, дублирующие существующие фразы, безусловно
     * не будут отправлены на сохранение в репозиторий, они просто удаляются
     * из мапы {@link #newKeywordsMap}.
     * <p>
     * Для фраз из запроса, которые не дублируют существующие фразы,
     * но при этом дублируются с другими фразами из запроса, действует
     * следующее правило: выбирается одна фраза, которая будет фактически
     * добавлена, а остальные игнорируются, при этом запоминается маппинг
     * индексов проигнорированных фраз на фактически добавленные
     * ({@link #newDuplicatesLinkingMap}), чтобы после добавления
     * в их результаты проставить id фактически добавленной фразы
     * (и предупреждение).
     */
    private void determineNewDuplicatesToAddAndCleanNewKeywordsMap() {
        duplicatesWithExisting.keySet().forEach(newKeywordsMap::remove);

        Map<Integer, Integer> newDuplicatesLinkingMapLocal = new HashMap<>();
        newDuplicates.forEach((duplicatedIndex, indexesOfDuplicates) -> {
            // если данная фраза не дублируется с существующей,
            // тогда имеет смысл думать о ее добавлении
            if (newKeywordsMap.containsKey(duplicatedIndex)) {
                // если в мапе уже присутствует данный индекс в качестве проигнорированного,
                // то пропускаем итерацию
                if (!newDuplicatesLinkingMapLocal.containsKey(duplicatedIndex)) {
                    // если данный индекс еще не попадался среди дубликатов, то добавляем его,
                    // а его дубликаты линкуем на него, чтобы в результате для них отдать его id;
                    // при этом удаляем проигнорированные дубликаты из мапы добавляемых фраз
                    indexesOfDuplicates.forEach(indexOfDuplicate -> {
                        newDuplicatesLinkingMapLocal.put(indexOfDuplicate, duplicatedIndex);
                        newKeywordsMap.remove(indexOfDuplicate);
                    });
                }
            }
        });
        newDuplicatesLinkingMap = ImmutableMap.copyOf(newDuplicatesLinkingMapLocal);
    }

    /**
     * ВАЖНО! На этом этапе нельзя расклеивать и как-то изменять сами фразы,
     * т.к. может измениться разбивка по группам.
     * <p>
     * Перед сохранением данных производятся дополнительные действия: замена групп для КФ, которые могут переполнить текущую группу,
     * расклейка фраз в пределах новых разбитых групп и др. операции, которые могут повлиять на общую валидность.
     */
    @Override
    protected void onModelsValidated(ModelsValidatedStep<Keyword> modelsValidatedStep) {
        Map<Integer, Keyword> validKeywordsMap = modelsValidatedStep.getValidModelsMap();

        clearMapsFromInvalidItems(validKeywordsMap);
        clearDuplicationMapsFromInvalidItems(validKeywordsMap);

        checkExecutionEntryPointConsistency(validKeywordsMap);

        prepareSystemFields();
    }

    /**
     * Очищает мапы с посчитанными дубликатами от невалидных элементов,
     * которые были помечены как невалидные после вычисления дубликатов.
     *
     * @param validKeywordsMap мапа валидных элементов (после последнего этапа валидации).
     */
    private void clearDuplicationMapsFromInvalidItems(Map<Integer, Keyword> validKeywordsMap) {
        Map<Integer, Integer> duplicatesWithExistingCopy = EntryStream.of(duplicatesWithExisting)
                .filterKeys(validKeywordsMap::containsKey)
                .toMap();
        duplicatesWithExisting = ImmutableMap.copyOf(duplicatesWithExistingCopy);

        Map<Integer, Collection<Integer>> newDuplicatesCopy = EntryStream.of(newDuplicates)
                .filterKeys(validKeywordsMap::containsKey)
                .toMap();
        newDuplicates = ImmutableMap.copyOf(newDuplicatesCopy);

        Map<Integer, Integer> newDuplicatesLinkingMapCopy = EntryStream.of(newDuplicatesLinkingMap)
                .filterKeys(validKeywordsMap::containsKey)
                .toMap();
        newDuplicatesLinkingMap = ImmutableMap.copyOf(newDuplicatesLinkingMapCopy);
    }

    /**
     * Индексы валидных элементов должны быть объединением множеств индексов из следующих мап:
     * {@link #newKeywordsMap}, {@link #duplicatesWithExisting}, {@link #newDuplicatesLinkingMap},
     * при этом эти множества не должны пересекаться.
     * <p>
     * Множество индексов результатов в мапе {@link #resultMap} должно совпадать с множеством
     * индексов валидных элементов.
     *
     * @param validKeywordsMap мапа валидных ключевых фраз.
     */
    private void checkExecutionEntryPointConsistency(Map<Integer, Keyword> validKeywordsMap) {
        Set<Integer> validIndexes = validKeywordsMap.keySet();
        Set<Integer> keywordsToAddIndexes = newKeywordsMap.keySet();
        Set<Integer> duplicatedWithExistingIndexes = duplicatesWithExisting.keySet();
        Set<Integer> newDuplicatesIndexes = newDuplicatesLinkingMap.keySet();
        checkState(keywordsToAddIndexes.size() +
                        duplicatedWithExistingIndexes.size() +
                        newDuplicatesIndexes.size() == validIndexes.size(),
                "can't execute operation: inconsistent state; "
                        + "this can happen for example if deduplication is not symmetric");

        Set<Integer> summaryIndexesSet = new HashSet<>();
        summaryIndexesSet.addAll(keywordsToAddIndexes);
        summaryIndexesSet.addAll(duplicatedWithExistingIndexes);
        summaryIndexesSet.addAll(newDuplicatesIndexes);
        checkState(summaryIndexesSet.equals(validIndexes), "can't execute operation: inconsistent state");

        Set<Integer> externalResultIndexes = resultMap.keySet();
        checkState(externalResultIndexes.equals(validIndexes), "can't execute operation: inconsistent state");
    }

    /**
     * Вычисление служебных полей {@link Keyword}.
     */
    private void prepareSystemFields() {
        LocalDateTime now = LocalDateTime.now();

        newKeywordsMap.forEach((index, internalKeyword) -> {
            Keyword keyword = internalKeyword.getKeyword();
            KeywordWithMinuses parsedNormalKeyword = internalKeyword.getParsedNormalKeyword();

            Long campaignId = keywordsAdGroupInfo.get(index).getCampaignId();
            String normalPhraseWithoutMinuses = parsedNormalKeyword.getKeyword().toString();
            int normalWordsCount = parsedNormalKeyword.getKeyword().getAllKeywords().size();

            keyword.withCampaignId(campaignId)
                    .withIsSuspended(false)
                    .withNormPhrase(normalPhraseWithoutMinuses)
                    .withWordsCount(normalWordsCount)
                    .withModificationTime(now)
                    .withStatusModerate(StatusModerate.NEW)
                    .withStatusBsSynced(StatusBsSynced.NO)
                    .withShowsForecast(0L)
                    .withNeedCheckPlaceModified(true)
                    .withPhraseBsId(BigInteger.ZERO)
                    .withPhraseIdHistory(null);
        });
    }

    @Override
    protected void beforeExecution(Map<Integer, Keyword> validModelsMapToApply) {
        if (operationParams.isIgnoreOversize()) {
            checkOversizeAdGroupIdsUpdateValid();
        }

        setPhrasesBeforeUnglueInResults();
        if (operationParams.isUnglueEnabled()) {
            unglue();
        }

        if (operationParams.isAdGroupsNonexistentOnPrepare()) {
            checkAllAdGroupIdsAreSet();
        }

        Collection<Long> campaignIds = mapList(keywordsAdGroupInfo.values(), AdGroupInfoForKeywordAdd::getCampaignId);
        Map<Long, Campaign> campaignByIds = campaignRepository.getCampaignsMap(shard, campaignIds);

        Map<Long, Campaign> newKeywordsCampaigns = operationParams.isAutoPrices() ?
                getKeywordsCampaigns(campaignByIds) : null;

        boolean needToCalcAutoPrices = operationParams.isAutoPrices() && needToCalcAutoPrices(newKeywordsCampaigns);

        // поход за прогнозами показов должен быть до похода в торги,
        // т.к. торгам нужны прогнозы
        getShowsForecast(campaignByIds, needToCalcAutoPrices);

        // Поход в торги должен обязательно производиться на этапе apply(),
        // т.к. для этого ходим в базу за данными по баннерам.
        // Если выполняется комплексная операция добавления/обновления группы,
        // всех баннеров в базе может еще не быть.
        // Потенциальная точка рефакторинга.
        requestTrafaretAuction(campaignByIds);

        // Расчет недостающих ставок и позиций показов на поиске должен
        // выполняться после похода в торги, т.к. они там используются
        if (operationParams.isAutoPrices()) {
            setFixedAutoPrices(newKeywordsCampaigns);
            if (needToCalcAutoPrices) {
                calcAutoPrices(newKeywordsCampaigns);
            }
            setMissingAutobudgetPriorities(newKeywordsCampaigns);
        }
        calcSearchPlaces();

        setMissingPrices();

        computeAdditionalTasks();
    }

    private Map<Long, Campaign> getKeywordsCampaigns(Map<Long, Campaign> campaignByIds) {
        Set<Long> newKeywordsCampaignIds = listToSet(newKeywordsMap.values(), InternalKeyword::getCampaignId);
        return EntryStream.of(campaignByIds)
                .filterKeys(newKeywordsCampaignIds::contains)
                .toMap();
    }

    // проверять лимит ключевиков с новыми группами, только если реально было переполнение
    private void checkOversizeAdGroupIdsUpdateValid() {
        // используем существующий метод валидации как проверка корректности во время выполнения операции,
        // без возвращения результата, если что-то пошло не так.
        // Считаем, что мы сами должны корректно определить новые группы.
        ValidationResult<List<Keyword>, Defect> vr = new ValidationResult<>(getModels());
        validationService.validateSizePerAdGroup(vr, newKeywordsMap, existingKeywordsMap, keywordsLimit);

        checkState(!vr.hasAnyErrors(), "no oversize errors expected, set new adGroup ids by indexes correctly.");
    }

    private void setPhrasesBeforeUnglueInResults() {
        newKeywordsMap.keySet().forEach(index -> {
            AddedKeywordInfo addedKeywordInfo = resultMap.get(index);
            String phraseBeforeUnglue = addedKeywordInfo.getResultPhrase();
            addedKeywordInfo.withPhraseBeforeUnglue(phraseBeforeUnglue);
        });
    }

    /**
     * Расклеивает фактически добавляемые фразы (то есть валидные за вычетом недобавляемых дубликатов)
     * с существующими и между собой. Подробнее о работе расклейки читать здесь: {@link KeywordUngluer}.
     * <p>
     * Сначала вызывает утилиту расклейки ({@link KeywordUngluer}) и получает
     * от нее результат, описывающий, как потенциально можно расклеить фразы.
     * <p>
     * Далее происходит фактическая расклейка: новым фразам и существующим добавляются минус-слова,
     * при этом минус-слова добавляются таким образом, чтобы не появились новые дубликаты,
     * и чтобы фразы не превысили ограничение на максимальную длину.
     */
    private void unglue() {
        List<UnglueContainer> newUnglueContainers = EntryStream.of(newKeywordsMap)
                .mapKeyValue(
                        (index, kw) -> new UnglueContainer(index, keywordsAdGroupInfo.get(index).getAdGroupIndex(),
                                kw.getParsedNormalKeyword()))
                .toList();
        //в случае несуществующих в prepare групп existingKeywordsMap пустая,
        // поэтому из-за kw.getAdGroupId() равного null не будет исключения
        List<UnglueContainer> existingUnglueContainers = EntryStream.of(existingKeywordsMap)
                .mapKeyValue(
                        (index, kw) -> new UnglueContainer(index, adGroupIndexById.get(kw.getAdGroupId()),
                                kw.getParsedNormalKeyword()))
                .toList();

        List<UnglueResult> unglueResults = keywordUngluer.unglue(newUnglueContainers, existingUnglueContainers);

        // сет нормальных форм фраз для отслеживания дубликатов
        Set<String> currentSummaryNormalKeywords = StreamEx.of(newKeywordsMap.values())
                .append(existingKeywordsMap.values())
                .map(InternalKeyword::getParsedNormalKeyword)
                .map(KeywordWithMinuses::toString)
                .toSet();

        unglueNewKeywords(unglueResults, currentSummaryNormalKeywords);
        unglueExistingKeywords(unglueResults, currentSummaryNormalKeywords);
    }

    /**
     * Добавление минус-слов к добавляемым фразам на основе результатов расклейки.
     * <p>
     * Результаты расклейки ({@link UnglueResult}) носят рекомендательный характер,
     * и, таким образом, к каждой фразе может быть добавлено любое подмножество
     * рекомендованных минус-слов. Эти минус-слова добавляются добавляются таким образом,
     * чтобы не появились новые дубликаты, и чтобы фразы не превысили ограничение на максимальную длину.
     *
     * @param unglueResults результаты расклейки.
     */
    private void unglueNewKeywords(List<UnglueResult> unglueResults, Set<String> currentSummaryNormalKeywords) {
        for (UnglueResult unglueResult : unglueResults) {
            if (unglueResult.getAddedMinusWords().isEmpty()) {
                continue;
            }

            int index = unglueResult.getIndex();
            InternalKeyword internalKeyword = newKeywordsMap.get(index);
            KeywordWithMinuses parsedKeyword = internalKeyword.getParsedKeyword();
            KeywordWithMinuses parsedNormalKeyword = internalKeyword.getParsedNormalKeyword();

            List<ru.yandex.direct.libs.keywordutils.model.Keyword> minusWordsToAdd =
                    mapList(unglueResult.getNormalizedAddedMinusWords(),
                            skw -> new ru.yandex.direct.libs.keywordutils.model.Keyword(false, singletonList(skw)));

            while (!minusWordsToAdd.isEmpty()) {
                KeywordWithMinuses newParsedKeyword = parsedKeyword.appendMinuses(minusWordsToAdd);
                int newLength =
                        KeywordProcessingUtils.getLengthWithoutMinusExclamationAndSpaces(newParsedKeyword.toString());

                if (newLength <= PhraseConstraints.KEYWORD_MAX_LENGTH) {
                    KeywordWithMinuses newParsedNormalKeyword =
                            keywordNormalizer.normalizeKeywordWithMinuses(newParsedKeyword);
                    String newNormalPhrase = newParsedNormalKeyword.toString();

                    // проверяем, что у нормальной формы новой фразы нет дубликатов
                    if (!currentSummaryNormalKeywords.contains(newNormalPhrase)) {

                        internalKeyword.getKeyword().setPhrase(newParsedKeyword.toString());
                        internalKeyword.withParsedKeyword(newParsedKeyword)
                                .withParsedNormalKeyword(newParsedNormalKeyword);

                        // добавляем информацию о расклейке в соответствующий результат
                        resultMap.get(index).withAddedMinuses(mapList(minusWordsToAdd, Object::toString));

                        // обновляем сет с текущим набором нормальных форм
                        currentSummaryNormalKeywords.remove(parsedNormalKeyword.toString());
                        currentSummaryNormalKeywords.add(newNormalPhrase);

                        break;
                    }
                }

                // пытаемся добавить на 1 слово меньше в следующей итерации
                minusWordsToAdd.remove(minusWordsToAdd.size() - 1);
            }
        }
    }

    /**
     * Добавление минус-слов к существующим фразам на основе результатов расклейки.
     * <p>
     * Результаты расклейки ({@link UnglueResult}) носят рекомендательный характер,
     * и, таким образом, к каждой фразе может быть добавлено любое подмножество
     * рекомендованных минус-слов. Эти минус-слова добавляются с учетом ограничения
     * на максимальную длину, а так же таким образом, чтобы не создать новые дубликаты.
     *
     * @param unglueResults результаты расклейки.
     */
    private void unglueExistingKeywords(List<UnglueResult> unglueResults, Set<String> currentSummaryNormalKeywords) {
        List<AppliedChanges<Keyword>> existingKeywordsToUpdateLocal = new ArrayList<>();
        List<AffectedKeywordInfo> affectedKeywordInfoListLocal = new ArrayList<>();

        Map<Integer, List<SingleKeyword>> minusWordsForExistingKeywords = StreamEx.of(unglueResults)
                .flatMapToEntry(UnglueResult::getNormalizedAddedMinusWordsToExistingMap)
                .grouping();

        minusWordsForExistingKeywords.forEach((index, minusWords) -> {

            InternalKeyword internalKeyword = existingKeywordsMap.get(index);
            KeywordWithMinuses parsedKeyword = internalKeyword.getParsedKeyword();
            KeywordWithMinuses parsedNormalKeyword = internalKeyword.getParsedNormalKeyword();

            List<ru.yandex.direct.libs.keywordutils.model.Keyword> minusWordsToAdd =
                    mapList(minusWords,
                            skw -> new ru.yandex.direct.libs.keywordutils.model.Keyword(false, singletonList(skw)));

            while (!minusWordsToAdd.isEmpty()) {
                KeywordWithMinuses newParsedKeyword = parsedKeyword.appendMinuses(minusWordsToAdd);
                int newLength =
                        KeywordProcessingUtils.getLengthWithoutMinusExclamationAndSpaces(newParsedKeyword.toString());

                if (newLength <= PhraseConstraints.KEYWORD_MAX_LENGTH) {
                    KeywordWithMinuses newParsedNormalKeyword =
                            keywordNormalizer.normalizeKeywordWithMinuses(newParsedKeyword);
                    String newNormalPhrase = newParsedNormalKeyword.toString();

                    // проверяем, что у нормальной формы новой фразы нет дубликатов
                    if (!currentSummaryNormalKeywords.contains(newNormalPhrase)) {
                        ModelChanges<Keyword> modelChanges = new ModelChanges<>(internalKeyword.getId(), Keyword.class);
                        modelChanges.process(newParsedKeyword.toString(), Keyword.PHRASE);
                        modelChanges.process(LocalDateTime.now(), Keyword.MODIFICATION_TIME);

                        AppliedChanges<Keyword> appliedChanges = modelChanges.applyTo(internalKeyword.getKeyword());
                        existingKeywordsToUpdateLocal.add(appliedChanges);

                        // добавляем дополнительную информацию о расклейке,
                        // которую клиент сможет получить после выполнения операции
                        String sourcePhrase = parsedKeyword.toString();
                        List<String> addedMinusWords = mapList(minusWordsToAdd, Object::toString);
                        affectedKeywordInfoListLocal
                                .add(new AffectedKeywordInfo(internalKeyword.getId(), internalKeyword.getAdGroupId(),
                                        sourcePhrase, addedMinusWords, internalKeyword.getKeyword().getIsSuspended()));

                        // обновляем сет с текущим набором нормальных форм
                        currentSummaryNormalKeywords.remove(parsedNormalKeyword.toString());
                        currentSummaryNormalKeywords.add(newNormalPhrase);

                        break;
                    }
                }

                // пытаемся добавить на 1 слово меньше в следующей итерации
                minusWordsToAdd.remove(minusWordsToAdd.size() - 1);
            }
        });

        Set<Long> uniqueAffectedIds = listToSet(affectedKeywordInfoListLocal, AffectedKeywordInfo::getId);
        checkState(affectedKeywordInfoListLocal.size() == uniqueAffectedIds.size(),
                "affectedKeywordInfoList contains duplicated ids");

        existingKeywordsToUpdate = ImmutableList.copyOf(existingKeywordsToUpdateLocal);
        affectedKeywordInfoList = ImmutableList.copyOf(affectedKeywordInfoListLocal);
    }

    /**
     * Проверяет, что у всех фраз были выставлены id групп, если их не было в {@link #prepare()}
     */
    private void checkAllAdGroupIdsAreSet() {
        Set<Integer> keywordWithoutAdGroupId = EntryStream.of(newKeywordsMap)
                .filterValues(keyword -> keyword.getAdGroupId() == null)
                .keys()
                .toSet();
        checkState(keywordWithoutAdGroupId.isEmpty(),
                "no ad group id for keywords with indexes: " + keywordWithoutAdGroupId);
    }

    /**
     * Получить прогнозы показов для новых фраз, и выставить их в поле {@code showsForecast}
     */
    private void getShowsForecast(Map<Long, Campaign> campaignByIds, boolean needToCalcAutoPrices) {
        Duration forecastTimeout =
                needToCalcAutoPrices ? AUTO_PRICE_FORECAST_TIMEOUT : DEFAULT_FORECAST_TIMEOUT;
        IdentityHashMap<Keyword, SearchItem> showForecasts =
                keywordShowsForecastService.getPhrasesShowsSafe(
                        mapList(newKeywordsMap.values(), InternalKeyword::getKeyword),
                        forecastTimeout,
                        clientId,
                        campaignByIds
                );
        newKeywordsMap.forEach((index, ikw) -> {
            SearchItem showsForecast = showForecasts.get(ikw.getKeyword());
            if (showsForecast != null) {
                ikw.getKeyword().setShowsForecast(showsForecast.getTotalCount());
            }
        });
    }

    /**
     * Получение результата торгов для новых ключевиков.
     * Запрос и сохранение похода в торги.
     */
    private void requestTrafaretAuction(Map<Long, Campaign> campaignByIds) {
        List<Keyword> keywordsForAuction = EntryStream.of(newKeywordsMap)
                .filter(e -> !ADGROUP_TYPES_NO_AUCTION.contains(keywordsAdGroupInfo.get(e.getKey()).getAdGroupType()))
                .values()
                .map(InternalKeyword::getKeyword)
                .toList();

        IdentityHashMap<Keyword, KeywordTrafaretData> trafaretDataMap =
                keywordBsAuctionService.getTrafaretAuctionMapSafe(clientId, keywordsForAuction, campaignByIds);
        trafaretDataByIndex = ImmutableMap.copyOf(
                EntryStream.of(newKeywordsMap)
                        .mapValues(InternalKeyword::getKeyword)
                        .mapValues(trafaretDataMap::get)
                        .nonNullValues()
                        .toMap());
    }

    /**
     * Выставление фиксированных ставок, если они есть в контейнере
     * {@code showConditionFixedAutoPrices}, отсутствуют в запросе,
     * и нужны в текущей стратегии
     *
     * @param newKeywordsCampaigns мапа с кампаниями, которым принадлежат фразы {@code newKeywordsMap}
     */
    private void setFixedAutoPrices(Map<Long, Campaign> newKeywordsCampaigns) {
        checkNotNull(showConditionFixedAutoPrices);
        newKeywordsMap.forEach((index, ikw) -> {
            BigDecimal fixedPrice = getPriceFromShowConditionAutoPrices(ikw);

            if (fixedPrice == null) {
                return;
            }

            Keyword kw = ikw.getKeyword();
            DbStrategy campStrategy = newKeywordsCampaigns.get(ikw.getCampaignId()).getStrategy();

            if (KEYWORD_MISSING_SEARCH_PRICE.test(kw, campStrategy)) {
                kw.setPrice(fixedPrice);
            }
            if (KEYWORD_MISSING_CONTEXT_PRICE.test(kw, campStrategy)) {
                kw.setPriceContext(fixedPrice);
            }
        });
    }

    private boolean needToCalcAutoPrices(@Nullable Map<Long, Campaign> newKeywordsCampaigns) {
        if (newKeywordsCampaigns == null || showConditionFixedAutoPrices == null) {
            return false;
        }
        for (InternalKeyword ikw : newKeywordsMap.values()) {
            BigDecimal fixedPrice = getPriceFromShowConditionAutoPrices(ikw);

            Keyword kw = ikw.getKeyword();
            DbStrategy campStrategy = newKeywordsCampaigns.get(ikw.getCampaignId()).getStrategy();

            if (fixedPrice == null &&
                    (KEYWORD_MISSING_SEARCH_PRICE.test(kw, campStrategy) ||
                            KEYWORD_MISSING_CONTEXT_PRICE.test(kw, campStrategy))) {
                return true;
            }
        }
        return false;
    }

    private BigDecimal getPriceFromShowConditionAutoPrices(InternalKeyword ikw) {
        if (showConditionFixedAutoPrices.hasGlobalFixedPrice()) {
            return showConditionFixedAutoPrices.getGlobalFixedPrice();
        } else if (!operationParams.isAdGroupsNonexistentOnPrepare()
                && showConditionFixedAutoPrices.hasAdGroupFixedPrice(ikw.getAdGroupId())) {
            return showConditionFixedAutoPrices.getAdGroupFixedPrice(ikw.getAdGroupId());
        }
        return null;
    }

    /**
     * Вычисление автоматических ставок для фраз, у которых нет явно указанной
     * ставки, обязательной в текущей стратегии. Вычисление производится
     * калькулятором {@code keywordAutoPricesCalculator}
     *
     * @param newKeywordsCampaigns мапа с кампаниями, которым принадлежат фразы {@code newKeywordsMap}
     */
    private void calcAutoPrices(Map<Long, Campaign> newKeywordsCampaigns) {
        if (keywordAutoPricesCalculator == null) {
            keywordAutoPricesCalculator = new KeywordAutoPricesCalculator(
                    clientCurrency,
                    mapList(existingKeywordsMap.values(), InternalKeyword::getKeyword),
                    keywordNormalizer
            );
        }

        newKeywordsMap.forEach((index, ikw) -> {
            Keyword kw = ikw.getKeyword();
            Campaign campaign = newKeywordsCampaigns.get(ikw.getCampaignId());

            if (KEYWORD_MISSING_SEARCH_PRICE.test(kw, campaign.getStrategy())) {
                KeywordBidBsAuctionData auctionData = null;
                if (trafaretDataByIndex.containsKey(index)) {
                    auctionData = convertToPositionsAuctionData(trafaretDataByIndex.get(index), clientCurrency);
                }

                kw.setPrice(keywordAutoPricesCalculator
                        .calcSearchAutoPrice(ikw, campaign.getType(), auctionData)
                        .bigDecimalValue()
                );
            }

            if (KEYWORD_MISSING_CONTEXT_PRICE.test(kw, campaign.getStrategy())) {
                kw.setPriceContext(keywordAutoPricesCalculator
                        .calcContextAutoPrice(ikw)
                        .bigDecimalValue()
                );
            }
        });
    }

    /**
     * Выставление приоритета по-умолчанию для всех фраз, где он явно не указан,
     * но нужен в текущей стратегии.
     *
     * @param newKeywordsCampaigns мапа с кампаниями, которым принадлежат фразы {@code newKeywordsMap}
     */
    private void setMissingAutobudgetPriorities(Map<Long, Campaign> newKeywordsCampaigns) {
        newKeywordsMap.forEach((index, ikw) -> {
            Keyword kw = ikw.getKeyword();
            Campaign campaign = newKeywordsCampaigns.get(ikw.getCampaignId());
            if (KEYWORD_MISSING_AUTOBUDGET_PRIORITY.test(kw, campaign.getStrategy())) {
                kw.setAutobudgetPriority(Constants.DEFAULT_AUTOBUDGET_PRIORITY);
            }
        });
    }

    /**
     * Расчет позиции показов на поиске для новых фраз
     */
    private void calcSearchPlaces() {
        IdentityHashMap<Keyword, Place> places = getKeywordPlaces(trafaretDataByIndex.values(), clientCurrency);

        newKeywordsMap.forEach((index, internalKeyword) -> {
            Keyword keyword = internalKeyword.getKeyword();
            Place place = places.get(keyword);
            place = place != null ? place : Place.ROTATION;
            keyword.withPlace(place);
        });
    }

    /**
     * Выставление цен по умолчанию для всех фраз, где они явно не указаны
     */
    private void setMissingPrices() {
        newKeywordsMap.forEach((index, internalKeyword) -> {
            Keyword keyword = internalKeyword.getKeyword();
            keyword.setPrice(nvl(keyword.getPrice(), BigDecimal.ZERO));
            keyword.setPriceContext(nvl(keyword.getPriceContext(), BigDecimal.ZERO));
        });
    }

    /**
     * Вычисление действий, которые будут произведены
     * над смежными объектами при выполнении операции.
     * <p>
     * Этот метод должен выполняться после {@link #prepareSystemFields},
     * так как пользуется служебными полями Keyword.
     */
    private void computeAdditionalTasks() {
        Set<Long> affectedAdGroupIds = listToSet(newKeywordsMap.values(), InternalKeyword::getAdGroupId);
        Set<Long> affectedCampaignIds = listToSet(newKeywordsMap.values(), InternalKeyword::getCampaignId);
        Set<Long> affectedTemplateBannerIds = bannerRepository.relations
                .getTemplateBannerIdsByAdGroupIds(shard, clientId, affectedAdGroupIds);
        List<KeywordEvent> addKeywordMailEvents = computeMailEvents(affectedAdGroupIds);

        transactionalAdditionalAddTask = conf -> {
            if (!affectedAdGroupIds.isEmpty()) {
                adGroupRepository.dropStatusModerateExceptDrafts(conf, affectedAdGroupIds);
                adGroupRepository.setSignificantlyChangedCoverageState(conf, affectedAdGroupIds);
            }

            if (!affectedTemplateBannerIds.isEmpty()) {
                bannerRepository.common.dropTemplateBannersStatusesOnKeywordsChange(conf, affectedTemplateBannerIds);
                moderationService.deleteBannerPostAndAuto(conf, affectedTemplateBannerIds);
                bannerRepository.moderation.deleteMinusGeo(conf.dsl(), affectedTemplateBannerIds);
            }
        };

        additionalAddTask = () -> {
            if (!affectedCampaignIds.isEmpty()) {
                campaignRepository.setAutobudgetForecastDate(shard, affectedCampaignIds, null);
                autoPriceCampQueueService.clearAutoPriceQueue(affectedCampaignIds);
                autobudgetAlertService.freezeAlertsOnKeywordsChange(clientId, affectedCampaignIds);
            }

            if (!addKeywordMailEvents.isEmpty()) {
                mailNotificationEventService.queueEvents(operatorUid, clientId, addKeywordMailEvents);
            }

            Collection<Keyword> keywords = mapList(newKeywordsMap.values(), InternalKeyword::getKeyword);
            List<LogPriceData> priceDataList = computeLogPriceDataList(keywords);
            if (!priceDataList.isEmpty()) {
                logPriceService.logPrice(priceDataList, operatorUid);
            }
        };
    }

    private List<KeywordEvent> computeMailEvents(Set<Long> affectedAdGroupIds) {
        Map<Long, AdGroupName> adGroupIdToAdGroup = getAdGroupIdToAdGroupNameWithCheck(affectedAdGroupIds);
        Map<Long, List<Keyword>> adGroupIdToKeywords = StreamEx.of(newKeywordsMap.values())
                .map(InternalKeyword::getKeyword)
                .mapToEntry(identity())
                .mapKeys(Keyword::getAdGroupId)
                .grouping();

        List<KeywordEvent> addKeywordEvents = new ArrayList<>();
        adGroupIdToKeywords.forEach((adGroupId, keywords) -> {
            long campaignId = adGroupIdToAdGroup.get(adGroupId).getCampaignId();
            String adGroupName = adGroupIdToAdGroup.get(adGroupId).getName();
            keywords.forEach(keyword -> {
                KeywordEvent event = addedKeywordEvent(operatorUid, clientUid,
                        campaignId, adGroupId, adGroupName, keyword.getPhrase());
                addKeywordEvents.add(event);
            });
        });
        return addKeywordEvents;
    }

    private Map<Long, AdGroupName> getAdGroupIdToAdGroupNameWithCheck(Set<Long> adGroupIds) {
        Map<Long, AdGroupName> adGroupIdToAdGroup =
                adGroupRepository.getAdGroupNames(shard, clientId, adGroupIds);
        Set<Long> notFoundAdGroupIds = Sets.difference(adGroupIds, adGroupIdToAdGroup.keySet());
        checkState(notFoundAdGroupIds.isEmpty(), "can't get adGroups for adGroup ids: " + notFoundAdGroupIds);
        return adGroupIdToAdGroup;
    }

    private List<LogPriceData> computeLogPriceDataList(Collection<Keyword> newKeywords) {
        Function<Keyword, LogPriceData> keywordToLogFn = keyword -> {
            checkState(keyword.getId() != null,
                    "attempt to log keyword without id (may be repository does not set id to added keywords "
                            + "or computing log records is called before keywords is added.");
            Double priceSearch = keyword.getPrice() != null ? keyword.getPrice().doubleValue() : 0;
            Double priceContext = keyword.getPriceContext() != null ? keyword.getPriceContext().doubleValue() : 0;
            return new LogPriceData(
                    keyword.getCampaignId(),
                    keyword.getAdGroupId(),
                    keyword.getId(),
                    priceContext,
                    priceSearch,
                    clientCurrency.getCode(),
                    LogPriceData.OperationType.INSERT_1);
        };
        return mapList(newKeywords, keywordToLogFn);
    }

    /**
     * В метод приходит мапа валидных ключевых слов по их индексам.
     * В качестве возвращаемого значения метод должен вернуть аналогичную
     * мапу результатов (для каждого входного элемента) по соответствующим индексам.
     * <p>
     * При этом в базе нужно создавать не все валидные ключевые слова, а только те,
     * которые не дублируются с существующими или другими добавляемыми, таковые
     * находятся в мапе newKeywordsMap. А для остальных валидных фраз необходимо
     * вернуть результат, в котором будет id их дубликатов и указание на то,
     * что элемент фактически не создан, а для него возвращен id другого элемента.
     * Маппинг входных элементов на существующие дубликаты находится в мапе
     * {@link #duplicatesWithExisting}. Маппинг входных элементов, которые имеют
     * дубликаты среди других входных элементов и не будут добавлены, на их дубликаты,
     * которые будут добавлены, находится в мапе {@link #newDuplicatesLinkingMap}.
     *
     * @param validModelsMapToApply мапа валидных ключевых фраз, к которым должна примениться операция
     * @return мапа результатов для всех элементов во входной мапе
     */
    @Override
    protected Map<Integer, AddedKeywordInfo> execute(Map<Integer, Keyword> validModelsMapToApply) {
        Map<Integer, AddedModelId> addedIdsMap = new HashMap<>();

        TransactionalRunnable saveFn = conf -> {
            Map<Integer, InternalKeyword> keywordsToSave = newKeywordsMap;
            Set<Long> affectedAdGroupIds = listToSet(keywordsToSave.values(), InternalKeyword::getAdGroupId);
            adGroupRepository.getLockOnAdGroups(conf, affectedAdGroupIds);

            Map<Long, Keyword> deletedKeywords = keywordRepository
                    .addAndUpdateWithDeduplication(conf, mapList(keywordsToSave.values(), InternalKeyword::getKeyword),
                            existingKeywordsToUpdate);

            keywordsToSave.forEach((i, keyword) -> {
                if (deletedKeywords.containsKey(keyword.getId())) {
                    addedIdsMap.put(i, AddedModelId.ofExisting(keyword.getId()));
                } else {
                    addedIdsMap.put(i, AddedModelId.ofNew(keyword.getId()));
                }
            });

            computeAdditionalTasksForDelete(deletedKeywords.values());

            transactionalAdditionalAddTask.run(conf);
            transactionalAdditionalDeleteTask.run(conf);
        };

        dslContextProvider.ppcTransaction(shard, saveFn);
        //дополнительные таски должны идти после saveFn, т.к. там создается additionalDeleteTask
        additionalAddTask.run();
        additionalDeleteTask.run();

        finishResultMapFilling(addedIdsMap);
        checkExecutionExitPointConsistency(resultMap);
        return resultMap;
    }

    /**
     * Вычисление действий, которые будут произведены над удаленными фразами.
     * <p>
     * Соответствует дополнительным операциям {@link KeywordDeleteOperation}, за исключением посылки писем
     * <p>
     * Расчет дополнительных операций для удаления выполняется в execute,
     * т.к. полное число удаленных ключевиков можно определить только после вызова метода {@code addAndUpdateWithDeduplication}
     */
    private void computeAdditionalTasksForDelete(Collection<Keyword> keywordsToDelete) {
        List<LogPriceData> priceDataList = computeLogDeletePriceDataList(keywordsToDelete);

        Set<Long> adGroupIdsToBsSynced = listToSet(keywordsToDelete, Keyword::getAdGroupId);
        Set<Long> campaignIdsToResetForecast = listToSet(keywordsToDelete, Keyword::getCampaignId);

        transactionalAdditionalDeleteTask = conf -> {
            adGroupRepository.updateAfterKeywordsDeleted(conf, adGroupIdsToBsSynced);
            campaignRepository.setAutobudgetForecastDate(conf, campaignIdsToResetForecast, null);
        };

        additionalDeleteTask = () -> {
            if (!keywordsToDelete.isEmpty()) {
                logPriceService.logPrice(priceDataList, operatorUid);
            }
        };
    }

    private List<LogPriceData> computeLogDeletePriceDataList(Collection<Keyword> deleteKeywords) {
        Function<Keyword, LogPriceData> keywordToLogFn = keyword -> new LogPriceData(
                keyword.getCampaignId(),
                keyword.getAdGroupId(),
                keyword.getId(),
                null,
                null,
                null,
                LogPriceData.OperationType.DELETE_1
        );
        return mapList(deleteKeywords, keywordToLogFn);
    }


    private void finishResultMapFilling(Map<Integer, AddedModelId> addedIdsMap) {
        // дозаполняем результаты для отправленных в репозиторий элементов
        addedIdsMap.forEach((index, addedId) -> {
            AddedKeywordInfo keywordInfo = resultMap.get(index);
            keywordInfo
                    .withId(addedId.getId())
                    .withAdGroupId(newKeywordsMap.get(index).getAdGroupId())
                    .withAdded(addedId.isAdded())
                    .withResultPhrase(newKeywordsMap.get(index).getPhrase());
            if (!addedId.isAdded()) {
                addWarning(index, duplicatedWithExisting());
            }
        });

        // дозаполняем результаты для элементов, которые дублировались с существующими
        duplicatesWithExisting.forEach((newIndex, existingIndex) -> {
            AddedKeywordInfo keywordInfo = resultMap.get(newIndex);
            InternalKeyword existingKeyword = existingKeywordsMap.get(existingIndex);
            keywordInfo
                    .withId(existingKeyword.getId())
                    .withAdGroupId(existingKeyword.getAdGroupId())
                    .withAdded(false);
            addWarning(newIndex, duplicatedWithExisting());
        });

        // дозаполняем результаты для элементов, которые дублировались с другими добавляемыми
        newDuplicatesLinkingMap.forEach((ignoredItemIndex, createdItemIndex) -> {
            AddedKeywordInfo keywordInfo = resultMap.get(ignoredItemIndex);
            AddedModelId addedId = addedIdsMap.get(createdItemIndex);
            keywordInfo.withId(addedId.getId())
                    .withAdGroupId(newKeywordsMap.get(createdItemIndex).getAdGroupId())
                    .withAdded(false);

            if (addedId.isAdded()) {
                addWarning(ignoredItemIndex, duplicatedWithNew());
            } else {
                addWarning(ignoredItemIndex, duplicatedWithExisting());
            }
        });
    }

    private void checkExecutionExitPointConsistency(Map<Integer, AddedKeywordInfo> addedKeywordInfoMap) {
        addedKeywordInfoMap.forEach((index, addedKeywordInfo) -> {
            checkState(addedKeywordInfo.getId() != null, "id is null in result[%s]", index);
            checkState(addedKeywordInfo.getResultPhrase() != null, "result phrase is null in result[%s]", index);
            if (!addedKeywordInfo.isAdded()) {
                List<Keyword> list = validationResult.getValue();
                ValidationResult<Keyword, Defect> keywordVr =
                        validationResult.getOrCreateSubValidationResult(index(index), list.get(index));
                checkState(keywordVr.hasAnyWarnings(),
                        "(result[%s].isAdded == false) and it has no warnings", index);
            }
        });
    }

    private void addWarning(int index, Defect warning) {
        validationResult.getOrCreateSubValidationResult(index(index), getModels().get(index))
                .addWarning(warning);
    }

    private List<Keyword> getKeywordsByAdGroupIds(int shard, ClientId clientId, Set<Long> adGroupIds) {
        Map<Long, List<Keyword>> keywordsByAdGroupIds =
                keywordRepository.getKeywordsByAdGroupIds(shard, clientId, adGroupIds);

        BinaryOperator<List<Keyword>> accumulator =
                (result, adGroupKeywords) -> {
                    result.addAll(adGroupKeywords);
                    return result;
                };

        return StreamEx.of(keywordsByAdGroupIds.values())
                .reduce(new ArrayList<>(), accumulator);
    }

    /**
     * Запрещаем частичное исполнение операции.
     */
    @Override
    public MassResult<AddedKeywordInfo> apply(Set<Integer> elementIndexesToApply) {
        throw new UnsupportedOperationException("Partial apply is unsupported by the operation");
    }
}
