package ru.yandex.direct.core.entity.keyword.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupAddOperationFactory;
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupService;
import ru.yandex.direct.core.entity.adgroup.service.complex.text.ComplexTextAdGroupAddOperation;
import ru.yandex.direct.core.entity.auction.container.bs.TrafaretBidItem;
import ru.yandex.direct.core.entity.banner.service.DatabaseMode;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.client.service.ClientLimitsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.keyword.container.AddedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.container.AffectedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.container.KeywordsAddOperationParams;
import ru.yandex.direct.core.entity.keyword.container.KeywordsModificationContainer;
import ru.yandex.direct.core.entity.keyword.container.KeywordsModificationResult;
import ru.yandex.direct.core.entity.keyword.container.KeywordsModifyOperationParams;
import ru.yandex.direct.core.entity.keyword.container.KeywordsUpdateOperationParams;
import ru.yandex.direct.core.entity.keyword.container.UpdatedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.processing.KeywordNormalizer;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.lang.Math.min;
import static java.util.Collections.singletonList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupModelUtils.cloneComplexAdGroupInCampaignOnOversize;
import static ru.yandex.direct.core.entity.keyword.container.KeywordsModificationContainer.ADD_LIST_NAME;
import static ru.yandex.direct.core.entity.keyword.container.KeywordsModificationContainer.COPY_AD_GROUPS_LIST_NAME;
import static ru.yandex.direct.core.entity.keyword.container.KeywordsModificationContainer.DELETE_LIST_NAME;
import static ru.yandex.direct.core.entity.keyword.container.KeywordsModificationContainer.UPDATE_LIST_NAME;
import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.cloneKeywords;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.ValidationResult.transferSubNodesWithIssues;

/**
 * Комплексная операция добавления/обновления/удаления ключевых фраз.
 * <p>
 * Состоит из трех базовых операций: добавления, обновления и удаления ключевых фраз.
 * Сначала все операции проходят этап валидации и подготовки изменений, и только
 * если все операции успешно прошли этот этап, проводится последовательное применение
 * провалидированных и подготовленных данных.
 * <p>
 * Порядок операций важен.
 * <p>
 * Удаление не зависит от добавления или обновления, поскольку для него важно только то,
 * присутствуют ли в базе указанные в запросе фразы. При этом обратная зависимость существует:
 * рассчитывать дубликаты и расклеивать фразы нужно с учетом того, какие фразы будут удалены
 * (получится не правильно, если мы расклеим добавляемую фразу с существующей, которая будет удалена),
 * обновлять можно только те фразы, которые не отправлены одновременно с этим на удаление,
 * а так же ограничение на максимальное количество фраз на группу должно срабатывать с учетом
 * удаленных фраз. Таким образом, удаление должно подготавливаться и выполняться первым.
 * <p>
 * Обновление зависит от списка существующих фраз, по нему производится дедупликация и расклейка.
 * То же можно сказать и про добавление. Обе эти операции зависят от существующих
 * ключевых фраз и обе вносят в них изменения. В отрыве от конкретных операций дедупликация
 * и расклейка - симметричны, и с этой точки зрения не важно в каком порядке будут выполнены
 * обновление и добавление, все равно фразы будут расклеены, а дубликатов не будет.
 * Однако, обновление порождает удаление существующих фраз, а добавление - нет.
 * И эти удаленные фразы влияют как на дедупликацию и расклейку, так и на валидацию добавления
 * (общее количество фраз на группу). Это и создает асимметрию между этими операциями и
 * устанавливает порядок их выполнения. Таким образом, обновление должно выполняться раньше добавления.
 * <p>
 * Важной особенностью данной операции является зависимость операций друг от друга:
 * в каждой операции при валидации и подготовке изменений учитываются существующие
 * в базе ключевые фразы, и при этом каждая из операций изменяет их. Получается,
 * что валидация/подготовка каждой операции должна опираться не на существующие
 * в базе фразы в этот момент времени, а на существующие в базе фразы к моменту
 * ее выполнения. Чтобы обеспечить это требование, при подготовке первым делом
 * создается виртуальный список существующих а базе фраз, передается в первую операцию,
 * после подготовки первой операции к нему применяются потенциальные изменения, которые
 * претерпят существующие фразы после применения данной операции, и уже этот
 * модифицированный список передается в следующую операцию для подготовки, и так далее.
 * Таким образом, каждая операция при подготовке и валидации видит тот список фраз,
 * который будет существовать в базе к моменту ее выполнения.
 * <p>
 * Детали процесса описаны в javadoc к методу {@code #prepare()}.
 * <p>
 * <p>
 * Параметр {@code autoPrices} включает режим автоматического вычисления ставок
 * в условиях показов, если они явно не указаны в запросе, но нужны в текущей
 * стратегии. При этом параметр {@code showConditionAutoPriceParams} должен
 * быть не {@code null}.
 * <p>
 * Автоматические ставки сначала ищутся в контейнере с фиксированными ставками,
 * который находится в {@code showConditionAutoPriceParams}. Если там ставки не
 * найдено, она считается хитрым образом с помощью калькулятора
 * {@link KeywordAutoPricesCalculator}.
 * <p>
 * Если режим {@code autoPrices} не включен, наличие нужных ставок проверяется
 * валидацией.
 */
public class KeywordsModifyOperation {
    private static final Logger logger = LoggerFactory.getLogger(KeywordsModifyOperation.class);
    private final KeywordsModifyOperationParams operationParams;

    private final KeywordOperationFactory keywordOperationFactory;
    private final KeywordRepository keywordRepository;
    private final KeywordNormalizer keywordNormalizer;
    private final AdGroupRepository adGroupRepository;
    private final ComplexAdGroupService complexAdGroupService;
    private final ComplexAdGroupAddOperationFactory complexAdGroupAddOperationFactory;
    private final ClientLimitsService clientLimitsService;
    private final RbacService rbacService;
    private final GeoTree geoTree;
    private final Currency clientCurrency;

    private final long operatorUid;
    private final ClientId clientId;
    private final long clientUid;
    private final int shard;

    private final ShowConditionAutoPriceParams showConditionAutoPriceParams;
    private final KeywordsModificationContainer inputContainer;

    private final NullableOperationDecorator<KeywordDeleteOperation, Long> nullableDeleteOperation;
    private final NullableOperationDecorator<KeywordsUpdateOperation, UpdatedKeywordInfo> nullableUpdateOperation;
    private final NullableOperationDecorator<KeywordsAddOperation, AddedKeywordInfo> nullableAddOperation;
    private final NullableOperationDecorator<ComplexTextAdGroupAddOperation, Long> nullableComplexAdGroupAddOperation;


    /**
     * Заполняется на этапе подготовки
     * Получение по индексу группы в операции копирвания групп исходного идентфикатора группы
     */
    private Map<Integer, Long> sourceAdGroupIdByCopiedAdGroupIndex;

    /**
     * заполняется на этапе подготовки
     * ключ - индекс КФ, значение - индекс новой группы в операции сохранения группы.
     * По индексу КФ получаем новую группу для сохранения на этапе apply()
     */
    private Map<Integer, Integer> newAdGroupIndexByKeywordIndex;


    private boolean prepared;
    private boolean executed;
    private Result<KeywordsModificationResult> result;

    /**
     * @param showConditionAutoPriceParams параметры для вычисления недостающих ставок в условиях показов.
     *                                     Должен быть не {@code null}, если в параметрах операции включен
     *                                     режим {@code autoPrices}.
     *                                     N.B.: это параметры не только для фраз, т.к. при копировании группы
     *                                     (при переполнении) нужны параметры для всех условий показов.
     */
    public KeywordsModifyOperation(KeywordsModifyOperationParams operationParams,
                                   KeywordOperationFactory keywordOperationFactory,
                                   KeywordRepository keywordRepository,
                                   KeywordNormalizer keywordNormalizer,
                                   AdGroupRepository adGroupRepository,
                                   ComplexAdGroupService complexAdGroupService,
                                   ClientLimitsService clientLimitsService,
                                   ClientGeoService clientGeoService,
                                   RbacService rbacService,
                                   ClientService clientService,
                                   ComplexAdGroupAddOperationFactory complexAdGroupAddOperationFactory,
                                   @Nullable ShowConditionAutoPriceParams showConditionAutoPriceParams,
                                   long operatorUid, ClientId clientId, long clientUid, int shard,
                                   KeywordsModificationContainer inputContainer) {
        if (inputContainer.isCopyOnOversize()) {
            checkArgument(!isEmpty(inputContainer.getAddList())
                            && isEmpty(inputContainer.getUpdateList())
                            && isEmpty(inputContainer.getDeleteList()),
                    "only add input data for copy on oversize operation is required");
        } else {
            checkArgument(!isEmpty(inputContainer.getAddList()) ||
                            !isEmpty(inputContainer.getUpdateList()) ||
                            !isEmpty(inputContainer.getDeleteList()),
                    "input data for add, update or delete is required");
        }

        this.operationParams = operationParams;
        this.keywordOperationFactory = keywordOperationFactory;
        this.keywordRepository = keywordRepository;
        this.keywordNormalizer = keywordNormalizer;
        this.adGroupRepository = adGroupRepository;
        this.complexAdGroupService = complexAdGroupService;
        this.complexAdGroupAddOperationFactory = complexAdGroupAddOperationFactory;
        this.clientLimitsService = clientLimitsService;
        this.rbacService = rbacService;
        this.geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);
        this.clientCurrency = clientService.getWorkCurrency(clientId);

        this.operatorUid = operatorUid;
        this.clientId = clientId;
        this.clientUid = clientUid;
        this.shard = shard;
        this.inputContainer = inputContainer;

        this.nullableDeleteOperation = new NullableOperationDecorator<>();
        this.nullableUpdateOperation = new NullableOperationDecorator<>();
        this.nullableAddOperation = new NullableOperationDecorator<>();
        this.nullableComplexAdGroupAddOperation = new NullableOperationDecorator<>();

        this.sourceAdGroupIdByCopiedAdGroupIndex = new HashMap<>();
        this.newAdGroupIndexByKeywordIndex = new HashMap<>();

        if (operationParams.isAutoPrices()) {
            checkArgument(showConditionAutoPriceParams != null,
                    "showConditionAutoPriceParams must be specified in autoPrices mode");
        }
        this.showConditionAutoPriceParams = showConditionAutoPriceParams;
    }

    public ImmutableList<AffectedKeywordInfo> getAffectedKeywordInfoListByUpdate() {
        checkState(prepared, "operation must be prepared");
        checkState(nullableUpdateOperation.isOperationPresent(), "update operation doesn't exist, check input data");
        return nullableUpdateOperation.getOperation().getAffectedKeywordInfoList();
    }

    public ImmutableMap<Integer, List<TrafaretBidItem>> getTrafaretBidsByAddIndexMap() {
        checkState(executed, "result of auction is available only after executing");
        return nullableAddOperation.isOperationPresent()
                ? nullableAddOperation.getOperation().getTrafaretBidsByIndexMap()
                : null;
    }

    public ImmutableMap<Integer, List<TrafaretBidItem>> getTrafaretBidsByUpdateIndexMap() {
        checkState(executed, "result of auction is available only after executing");
        return nullableUpdateOperation.isOperationPresent()
                ? nullableUpdateOperation.getOperation().getTrafaretBidsByIndexMap()
                : null;
    }

    /**
     * Общее описание процесса смотреть в javadoc класса.
     * <p>
     * На этапе подготовки из базы загружаются все ключевые фразы
     * из всех потенциально затронутых запросом групп объявлений.
     * Потенциально, а не фактически, потому что id ключевых фраз
     * берутся из непровалидированного запроса.
     * <p>
     * Этот слепок далее используется как виртуальное состояние
     * ключевых фраз в группах: после подготовки каждой операции
     * в него вносятся изменения, которые соответствуют потенциальному
     * применению операции, и измененный список передаетя в следующую
     * операцию. Таким образом, валидация и подготовка в каждой операции
     * проводятся с учетом состояния групп после применения предыдущей
     * операции.
     * <p>
     * Каждая операция должна брать информацию о существующих фразах
     * в группе только на основе переданного списка.
     * <p>
     * Каждая операция должна корректно обрабатывать кейс, когда
     * в переданном списке существующих фраз присутствуют группы,
     * не затрагиваемые ей.
     * <p>
     * Каждая операция по отдельности является необязательной,
     * поэтому работа с ними для удобства ведется через декоратор
     * {@link NullableOperationDecorator}.
     */
    public Optional<Result<KeywordsModificationResult>> prepare() {
        checkState(!prepared, "prepare() can be called only once");
        prepared = true;

        List<Keyword> virtualExistingKeywords = getExistingKeywordsForAffectedAdGroups();
        KeywordAutoPricesCalculator autoPricesCalculator = getAutoPricesCalculator(virtualExistingKeywords);
        virtualExistingKeywords = prepareDeleteOperation(virtualExistingKeywords);
        virtualExistingKeywords = prepareUpdateOperation(virtualExistingKeywords, autoPricesCalculator);
        prepareAddOperation(virtualExistingKeywords, autoPricesCalculator);

        boolean failed = nullableDeleteOperation.getResult().isPresent() ||
                nullableUpdateOperation.getResult().isPresent() ||
                nullableAddOperation.getResult().isPresent();

        if (!failed && inputContainer.isCopyOnOversize()) {
            prepareComplexAdGroupAddOperationForCopy();
            failed = nullableComplexAdGroupAddOperation.getResult().isPresent();
        }

        if (failed) {
            ValidationResult<KeywordsModificationContainer, Defect> validationResult =
                    buildValidationResult(inputContainer,
                            nullableDeleteOperation.getNullableResult(),
                            nullableUpdateOperation.getNullableResult(),
                            nullableAddOperation.getNullableResult(),
                            nullableComplexAdGroupAddOperation.getNullableResult());
            String errors = StreamEx.of(validationResult.flattenErrors())
                            .limit(20)
                            .joining("; ");
            logger.info("Keywords errors length: {}", errors.length());
            logger.error("Keywords prepare errors: {}", errors.substring(0, min(errors.length(), 10000)));
            this.result = Result.broken(validationResult);
        }

        return Optional.ofNullable(result);
    }

    /**
     * Инициализация калькулятора автоматических недостающих ставок.
     * Его полезно создавать в этой операции, т.к. в нем нужна информация по
     * состоянию всех фраз в группах до удаления и обновления.
     * Этот калькулятор потом должен быть передан в операции обновления и
     * добавления фраз.
     *
     * @param virtualExistingKeywords список фраз в группах до выполнения операции
     * @return калькулятор, если включен режим {@code autoPrices}. Иначе {@code null}.
     */
    @Nullable
    private KeywordAutoPricesCalculator getAutoPricesCalculator(List<Keyword> virtualExistingKeywords) {
        if (operationParams.isAutoPrices()) {
            return new KeywordAutoPricesCalculator(
                    clientCurrency,
                    cloneKeywords(virtualExistingKeywords),
                    keywordNormalizer
            );
        } else {
            return null;
        }
    }

    private List<Keyword> prepareDeleteOperation(List<Keyword> virtualExistingKeywords) {
        List<Keyword> newVirtualExistingKeywords = cloneKeywords(virtualExistingKeywords);
        if (isEmpty(inputContainer.getDeleteList())) {
            return newVirtualExistingKeywords;
        }

        List<Keyword> workingVirtualExistingKeywords = cloneKeywords(virtualExistingKeywords);
        KeywordDeleteOperation deleteOperation = keywordOperationFactory
                .createKeywordsDeleteOperation(Applicability.FULL, inputContainer.getDeleteList(),
                        workingVirtualExistingKeywords, operatorUid, clientId);
        nullableDeleteOperation.setOperation(deleteOperation);

        Optional<MassResult<Long>> deleteResult = deleteOperation.prepare();
        if (!deleteResult.isPresent()) {
            Set<Long> potentiallyRemovedIds = deleteOperation.getPotentiallyDeletedKeywordIds();
            newVirtualExistingKeywords.removeIf(kw -> potentiallyRemovedIds.contains(kw.getId()));
        }
        return newVirtualExistingKeywords;
    }

    /**
     * @param autoPricesCalculator калькулятор для вычисления автоматических недостающих ставок.
     *                             Должен быть не {@code null}, если включен режим {@code autoPrices}
     */
    private List<Keyword> prepareUpdateOperation(List<Keyword> virtualExistingKeywords,
                                                 @Nullable KeywordAutoPricesCalculator autoPricesCalculator) {
        List<Keyword> newVirtualExistingKeywords = cloneKeywords(virtualExistingKeywords);
        if (isEmpty(inputContainer.getUpdateList())) {
            return newVirtualExistingKeywords;
        }

        List<Keyword> workingVirtualExistingKeywords = cloneKeywords(virtualExistingKeywords);
        KeywordsUpdateOperationParams operationParams = KeywordsUpdateOperationParams.builder()
                .withUnglueEnabled(true)
                .withAutoPrices(this.operationParams.isAutoPrices())
                .withModificationDisabled(false)
                .build();
        ShowConditionFixedAutoPrices fixedAutoPrices =
                ifNotNull(showConditionAutoPriceParams, ShowConditionAutoPriceParams::getFixedAutoPrices);
        KeywordsUpdateOperation updateOperation = keywordOperationFactory.createKeywordsUpdateOperation(
                Applicability.FULL, operationParams, inputContainer.getUpdateList(), workingVirtualExistingKeywords,
                fixedAutoPrices, autoPricesCalculator, operatorUid, clientId, clientUid);
        nullableUpdateOperation.setOperation(updateOperation);

        Optional<MassResult<UpdatedKeywordInfo>> updateResult = updateOperation.prepare();
        if (!updateResult.isPresent()) {
            Set<Long> idsToDelete = updateOperation.getPotentiallyDeletedKeywordIds();
            newVirtualExistingKeywords.removeIf(kw -> idsToDelete.contains(kw.getId()));

            Map<Long, Keyword> keywordIdToExistingKeyword =
                    listToMap(newVirtualExistingKeywords, Keyword::getId);

            updateOperation.getPotentiallyUpdatedKeywords().forEach(appliedChanges -> {
                Long keywordId = appliedChanges.getModel().getId();
                Keyword keyword = keywordIdToExistingKeyword.get(keywordId);
                modify(keyword, appliedChanges);
            });
        }
        return newVirtualExistingKeywords;
    }

    /**
     * @param autoPricesCalculator калькулятор для вычисления автоматических недостающих ставок.
     *                             Должен быть не {@code null}, если включен режим {@code autoPrices}
     */
    private void prepareAddOperation(List<Keyword> virtualExistingKeywords,
                                     @Nullable KeywordAutoPricesCalculator autoPricesCalculator) {
        if (isEmpty(inputContainer.getAddList())) {
            return;
        }

        KeywordsAddOperationParams operationParams = KeywordsAddOperationParams.builder()
                .withAdGroupsNonexistentOnPrepare(false)
                .withUnglueEnabled(true)
                .withIgnoreOversize(inputContainer.isCopyOnOversize())
                .withAutoPrices(this.operationParams.isAutoPrices())
                .withWeakenValidation(false)
                .build();
        ShowConditionFixedAutoPrices fixedAutoPrices =
                ifNotNull(showConditionAutoPriceParams, ShowConditionAutoPriceParams::getFixedAutoPrices);
        List<Keyword> workingVirtualExistingKeywords = cloneKeywords(virtualExistingKeywords);
        KeywordsAddOperation addOperation = keywordOperationFactory.createKeywordsAddOperation(Applicability.FULL,
                operationParams, inputContainer.getAddList(), workingVirtualExistingKeywords,
                fixedAutoPrices,
                autoPricesCalculator,
                operatorUid, clientId, clientUid);

        nullableAddOperation.setOperation(addOperation);
        addOperation.prepare();
    }

    private void prepareComplexAdGroupAddOperationForCopy() {
        checkState(nullableAddOperation.isOperationPresent());

        Map<Long, List<Integer>> oversizeKeywordsIndexesByAdGroupId =
                nullableAddOperation.getOperation().getOversizeKeywordsIndexesByAdGroupId();
        if (oversizeKeywordsIndexesByAdGroupId.isEmpty()) {
            return;
        }

        boolean isOperatorInternal = rbacService.getUidRole(operatorUid).isInternal();

        Set<Long> adGroupIds = oversizeKeywordsIndexesByAdGroupId.keySet();
        List<ComplexTextAdGroup> complexAdGroups = complexAdGroupService
                .getComplexAdGroupsWithoutKeywords(operatorUid, UidAndClientId.of(clientUid, clientId), adGroupIds);

        Long keywordsLimit = clientLimitsService.massGetClientLimits(singletonList(clientId))
                .iterator().next()
                .getKeywordsCountLimitOrDefault();

        int currentCopyAdGroupIndex = 0;
        List<ComplexTextAdGroup> newComplexAdGroups = new ArrayList<>();
        for (ComplexTextAdGroup complexAdGroup : complexAdGroups) {
            Long sourceAdGroupId = complexAdGroup.getAdGroup().getId();
            List<Integer> oversizeKeywordsIndexes = oversizeKeywordsIndexesByAdGroupId.get(sourceAdGroupId);

            int keywordsInCurrentGroup = 0;
            Iterator<Integer> keywordIndexIterator = oversizeKeywordsIndexes.iterator();
            while (keywordIndexIterator.hasNext()) {
                Integer keywordIndex = keywordIndexIterator.next();

                if (!sourceAdGroupIdByCopiedAdGroupIndex.containsKey(currentCopyAdGroupIndex)) {
                    newComplexAdGroups.add(cloneComplexAdGroupInCampaignOnOversize(complexAdGroup, isOperatorInternal));
                    sourceAdGroupIdByCopiedAdGroupIndex.put(currentCopyAdGroupIndex, sourceAdGroupId);
                }
                newAdGroupIndexByKeywordIndex.put(keywordIndex, currentCopyAdGroupIndex);
                keywordsInCurrentGroup++;
                if (keywordsInCurrentGroup == keywordsLimit && keywordIndexIterator.hasNext()) {
                    currentCopyAdGroupIndex++;
                    keywordsInCurrentGroup = 0;
                }
            }
            // увеличиваем индекс для группы, когда переходим к следующей переполненной группе.
            currentCopyAdGroupIndex++;
        }

        ComplexTextAdGroupAddOperation complexTextAdGroupAddOperation = complexAdGroupAddOperationFactory
                .createTextAdGroupAddOperation(false, newComplexAdGroups, geoTree,
                        operationParams.isAutoPrices(), showConditionAutoPriceParams,
                        operatorUid, clientId, clientUid, DatabaseMode.ONLY_MYSQL);
        complexTextAdGroupAddOperation.prepare();

        nullableComplexAdGroupAddOperation.setOperation(complexTextAdGroupAddOperation);
    }

    // todo может пригодиться, если придется возвращать затронутые операцией add фразы
    // Gри этом в текущем виде не подойдет, так как из операции update могут возвращаться
    // устаревшие фразы с результатами расклейки из-за того, что они могут быть подвергнуты
    // повторной расклейке в операции add. Поэтому правильнее делать метод, который будет возвращать
    // единый список всех затронутых запросом фраз, как фигурирующих в запросе, так и косвенно затронутых,
    // с конечным серженным результатом комплексной операции.
//    private void computeAffectedKeywordInfoList() {
//        Map<Long, String> sourcePhrases = new HashMap<>();
//        ListMultimap<Long, String> addedMinusesMap = MultimapBuilder.hashKeys().arrayListValues().build();
//
//        if (nullableUpdateOperation.isOperationPresent()) {
//            List<AffectedKeywordInfo> affectedList =
//                    nullableUpdateOperation.getOperation().getAffectedKeywordInfoList();
//            affectedList.forEach(affectedInfo -> {
//                sourcePhrases.put(affectedInfo.getId(), affectedInfo.getSourcePhrase());
//                addedMinusesMap.putAll(affectedInfo.getId(), affectedInfo.getAddedMinuses());
//            });
//        }
//        if (nullableAddOperation.isOperationPresent()) {
//            List<AffectedKeywordInfo> affectedList =
//                    nullableAddOperation.getOperation().getAffectedKeywordInfoList();
//            affectedList.forEach(affectedInfo -> {
//                // приоритет у исходной фразы из операции обновления, так как она выполняется перед добавлением
//                sourcePhrases.putIfAbsent(affectedInfo.getId(), affectedInfo.getSourcePhrase());
//                addedMinusesMap.putAll(affectedInfo.getId(), affectedInfo.getAddedMinuses());
//            });
//        }
//
//        List<AffectedKeywordInfo> affectedKeywordInfoListLocal = new ArrayList<>();
//        addedMinusesMap.asMap().forEach((id, minuses) -> {
//            String sourcePhrase = sourcePhrases.get(id);
//            //noinspection unchecked
//            AffectedKeywordInfo mergedAffectedInfo = new AffectedKeywordInfo(id, sourcePhrase, (List) minuses);
//            affectedKeywordInfoListLocal.add(mergedAffectedInfo);
//        });
//
//        affectedKeywordInfoList = ImmutableList.copyOf(affectedKeywordInfoListLocal);
//    }

    public Result<KeywordsModificationResult> apply() {
        checkState(prepared, "prepare() must be called before apply()");
        checkState(!executed, "apply() can be called only once");
        if (result != null) {
            logResult();
        }
        checkState(result == null, "result is already computed by prepare()");
        executed = true;

        copyOversizeAdGroups();
        nullableDeleteOperation.apply();
        nullableUpdateOperation.apply();
        nullableAddOperation.apply();
        buildExecutedResult();
        return result;
    }

    private void copyOversizeAdGroups() {
        if (inputContainer.isCopyOnOversize() && nullableComplexAdGroupAddOperation.isOperationPresent()) {
            nullableComplexAdGroupAddOperation.apply();

            Optional<MassResult<Long>> copiedAdGroupIdsResult = nullableComplexAdGroupAddOperation.getResult();
            checkState(copiedAdGroupIdsResult.isPresent(), "копирование новых групп необходимо, ожидается результат");
            checkState(copiedAdGroupIdsResult.get().isSuccessful(), "операция должна быть успешной");

            List<Long> copiedAdGroupIds = mapList(copiedAdGroupIdsResult.get().getResult(), Result::getResult);
            Map<Integer, Long> newAdGroupIdByKeywordToSaveIndex = EntryStream.of(newAdGroupIndexByKeywordIndex)
                    .mapValues(copiedAdGroupIds::get)
                    .toMap();
            nullableAddOperation.getOperation().setNewAdGroupIdsByIndex(newAdGroupIdByKeywordToSaveIndex);
        }
    }

    public Result<KeywordsModificationResult> prepareAndApply() {
        return prepare().orElseGet(this::apply);
    }


    @SuppressWarnings("ConstantConditions")
    private void buildExecutedResult() {
        List<Long> deletedIds = null;
        List<AddedKeywordInfo> addedKeywordInfos = null;
        List<UpdatedKeywordInfo> updatedKeywordInfos = null;
        Map<Long, Long> sourceAdGroupIdByCopiedAdGroupId = null;
        if (nullableDeleteOperation.isOperationPresent()) {
            deletedIds = mapList(nullableDeleteOperation.getResult().get().getResult(), Result::getResult);
        }
        if (nullableUpdateOperation.isOperationPresent()) {
            updatedKeywordInfos = mapList(nullableUpdateOperation.getResult().get().getResult(), Result::getResult);
        }
        if (nullableAddOperation.isOperationPresent()) {
            addedKeywordInfos = mapList(nullableAddOperation.getResult().get().getResult(), Result::getResult);
        }
        if (nullableComplexAdGroupAddOperation.isOperationPresent()) {
            List<Long> copiedAdGroupIds =
                    mapList(nullableComplexAdGroupAddOperation.getResult().get().getResult(), Result::getResult);

            sourceAdGroupIdByCopiedAdGroupId = EntryStream.of(sourceAdGroupIdByCopiedAdGroupIndex)
                    .mapKeys(copiedAdGroupIds::get)
                    .toMap();
        }

        KeywordsModificationResult resultValue =
                new KeywordsModificationResult(addedKeywordInfos, updatedKeywordInfos, deletedIds,
                        sourceAdGroupIdByCopiedAdGroupId);

        ValidationResult<KeywordsModificationContainer, Defect> modificationContainerVr =
                buildValidationResult(inputContainer,
                        nullableDeleteOperation.getNullableResult(),
                        nullableUpdateOperation.getNullableResult(),
                        nullableAddOperation.getNullableResult(),
                        nullableComplexAdGroupAddOperation.getNullableResult());

        this.result = Result.successful(resultValue, modificationContainerVr);
    }

    @SuppressWarnings("ConstantConditions")
    private static ValidationResult<KeywordsModificationContainer, Defect> buildValidationResult(
            KeywordsModificationContainer inputContainer,
            @Nullable MassResult<Long> deleteResult,
            @Nullable MassResult<UpdatedKeywordInfo> updateResult,
            @Nullable MassResult<AddedKeywordInfo> addResult,
            @Nullable MassResult<Long> copyAdGroupsResult) {
        ValidationResult<KeywordsModificationContainer, Defect> modificationContainerVr =
                new ValidationResult<>(inputContainer);

        if (deleteResult != null) {
            ValidationResult<List<Long>, Defect> deleteKeywordsVr = modificationContainerVr
                    .getOrCreateSubValidationResult(field(DELETE_LIST_NAME), inputContainer.getDeleteList());
            transferSubNodesWithIssues(deleteResult.getValidationResult(), deleteKeywordsVr);
        }

        if (updateResult != null) {
            // для апдейте в качестве валидируемого значения должен быть
            // не исходный список ModelChanges, а список моделей
            //noinspection unchecked
            List<Keyword> validatedModels = (List<Keyword>) updateResult.getValidationResult().getValue();
            ValidationResult<List<Keyword>, Defect> updateKeywordsVr = modificationContainerVr
                    .getOrCreateSubValidationResult(field(UPDATE_LIST_NAME), validatedModels);
            transferSubNodesWithIssues(updateResult.getValidationResult(), updateKeywordsVr);
        }

        if (addResult != null) {
            ValidationResult<List<Keyword>, Defect> addKeywordsVr = modificationContainerVr
                    .getOrCreateSubValidationResult(field(ADD_LIST_NAME), inputContainer.getAddList());
            transferSubNodesWithIssues(addResult.getValidationResult(), addKeywordsVr);
        }

        if (copyAdGroupsResult != null) {
            ValidationResult<List<Long>, Defect> copyAdGroupsVr = modificationContainerVr
                    .getOrCreateSubValidationResult(field(COPY_AD_GROUPS_LIST_NAME),
                            mapList(copyAdGroupsResult.getResult(), Result::getResult));
            transferSubNodesWithIssues(copyAdGroupsResult.getValidationResult(), copyAdGroupsVr);
        }

        return modificationContainerVr;
    }

    private List<Keyword> getExistingKeywordsForAffectedAdGroups() {
        Set<Long> potentiallyAffectedAdGroupIds = getPotentiallyAffectedAdGroupIds();
        Map<Long, List<Keyword>> adGroupIdToKeywords =
                keywordRepository.getKeywordsByAdGroupIds(shard, clientId, potentiallyAffectedAdGroupIds);
        return StreamEx.of(adGroupIdToKeywords.values())
                .reduce(new ArrayList<>(), (res, adGroupKeywords) -> {
                    res.addAll(adGroupKeywords);
                    return res;
                });
    }

    private Set<Long> getPotentiallyAffectedAdGroupIds() {
        int reqSize = inputContainer.getRequestSize();
        Set<Long> keywordIdsForUpdateAndDelete = inputContainer.getKeywordIds();

        Set<Long> adGroupIds = new HashSet<>(reqSize);

        if (!isEmpty(inputContainer.getAddList())) {
            Collection<Long> adGroupIdsForAdd = StreamEx.of(inputContainer.getAddList())
                    .nonNull()
                    .map(Keyword::getAdGroupId)
                    .nonNull()
                    .toList();
            adGroupIdsForAdd = adGroupRepository.getClientExistingAdGroupIds(shard, clientId, adGroupIdsForAdd);
            adGroupIds.addAll(adGroupIdsForAdd);
        }

        Collection<Long> adGroupIdsForUpdateAndDelete =
                keywordRepository.getAdGroupIdsByKeywordsIds(shard, clientId, keywordIdsForUpdateAndDelete);
        adGroupIds.addAll(adGroupIdsForUpdateAndDelete);
        return adGroupIds;
    }

    @SuppressWarnings("unchecked")
    private static void modify(Keyword keyword, AppliedChanges<Keyword> appliedChanges) {
        appliedChanges.getActuallyChangedProps().forEach(prop -> {
            Object value = appliedChanges.getNewValue(prop);
            ModelProperty<Keyword, Object> property = (ModelProperty<Keyword, Object>) prop;
            property.set(keyword, value);
        });
    }

    private void logResult() {
        logger.info("Start logging of keyword modify result");
        String deleteResult = result.getResult() != null && result.getResult().getDeleteResults() != null
                ? StreamEx.of(result.getResult().getDeleteResults())
                    .limit(20)
                    .joining(", ")
                : "";
        String addResult = result.getResult() != null && result.getResult().getAddResults() != null
                ? StreamEx.of(result.getResult().getAddResults())
                    .limit(20)
                    .map(AddedKeywordInfo::getResultPhrase)
                    .joining(", ")
                : "";
        String updateResult = result.getResult() != null && result.getResult().getUpdateResults() != null
                ? StreamEx.of(result.getResult().getUpdateResults())
                    .limit(20)
                    .map(UpdatedKeywordInfo::getResultPhrase)
                    .joining(", ")
                : "";

        String errors = result.getValidationResult() != null
                ? StreamEx.of(result.getValidationResult().flattenErrors())
                    .limit(20)
                    .joining("; ")
                : "";
        if (result.getValidationResult() != null) {
            logger.info("Keywords modify has any errors: {}", result.getValidationResult().hasAnyErrors());
            if (!result.getValidationResult().flattenErrors().isEmpty()) {
                String firstError = result.getValidationResult().flattenErrors().get(0).toString();
                logger.info("Keywords first error length: {}", firstError.length());
                logger.info("Keywords first error: {}", firstError.substring(0, min(10000, firstError.length())));
            } else {
                logger.info("Can't find any error");
            }
        }
        logger.info("Keywords add result: {}, delete result: {}, update result: {}, prepare errors: {}", addResult,
                deleteResult, updateResult, errors);

        logger.info("Finish logging of keyword modify result");
    }

    /**
     * Декоратор для взаимодействия с операцией, которая может быть равна {@code null}.
     * Нужен для удобства, чтобы не проверять ее при каждом обращении на {@code null}.
     */
    private static class NullableOperationDecorator<O extends Operation<R>, R> implements Operation<R> {
        private O operation;

        public void setOperation(O operation) {
            checkState(this.operation == null, "operation is already set");
            this.operation = checkNotNull(operation, "operation is required");
        }

        public boolean isOperationPresent() {
            return operation != null;
        }

        public O getOperation() {
            return operation;
        }

        @Override
        public Optional<MassResult<R>> prepare() {
            if (operation == null) {
                return Optional.empty();
            }
            return operation.prepare();
        }

        @Override
        @Nullable
        public MassResult<R> apply() {
            if (operation == null) {
                return null;
            }
            return operation.apply();
        }

        @Override
        @Nullable
        public MassResult<R> cancel() {
            if (operation == null) {
                return null;
            }
            return operation.cancel();
        }

        @Override
        public Optional<MassResult<R>> getResult() {
            if (operation == null) {
                return Optional.empty();
            }
            return operation.getResult();
        }

        public MassResult<R> getNullableResult() {
            if (operation == null) {
                return null;
            }
            return operation.getResult().orElse(null);
        }
    }
}
