package ru.yandex.direct.core.entity.keyword.service;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.TransactionalRunnable;

import ru.yandex.direct.advq.search.SearchItem;
import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupName;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.StatusShowsForecast;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordBidBsAuctionData;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordTrafaretData;
import ru.yandex.direct.core.entity.auction.container.bs.TrafaretBidItem;
import ru.yandex.direct.core.entity.autobudget.service.AutobudgetAlertService;
import ru.yandex.direct.core.entity.autoprice.service.AutoPriceCampQueueService;
import ru.yandex.direct.core.entity.banner.repository.BannerRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.keyword.container.AffectedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.container.CampaignIdAndKeywordIdPair;
import ru.yandex.direct.core.entity.keyword.container.FixStopwordsResult;
import ru.yandex.direct.core.entity.keyword.container.InternalKeyword;
import ru.yandex.direct.core.entity.keyword.container.KeywordsUpdateOperationParams;
import ru.yandex.direct.core.entity.keyword.container.UpdatedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.model.Place;
import ru.yandex.direct.core.entity.keyword.model.StatusModerate;
import ru.yandex.direct.core.entity.keyword.processing.KeywordNormalizer;
import ru.yandex.direct.core.entity.keyword.processing.KeywordProcessingUtils;
import ru.yandex.direct.core.entity.keyword.processing.deduplication.DuplicatingContainer;
import ru.yandex.direct.core.entity.keyword.processing.unglue.KeywordUngluer;
import ru.yandex.direct.core.entity.keyword.processing.unglue.UnglueContainer;
import ru.yandex.direct.core.entity.keyword.processing.unglue.UnglueResult;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.keyword.service.internal.InternalKeywordFactory;
import ru.yandex.direct.core.entity.keyword.service.validation.UpdateKeywordValidationService;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseConstraints;
import ru.yandex.direct.core.entity.mailnotification.model.KeywordEvent;
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService;
import ru.yandex.direct.core.entity.moderation.service.ModerationService;
import ru.yandex.direct.core.entity.showcondition.Constants;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.libs.keywordutils.helper.SingleKeywordsCache;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;
import ru.yandex.direct.libs.keywordutils.model.SingleKeyword;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.AbstractUpdateOperation;
import ru.yandex.direct.operation.update.AppliedChangesValidatedStep;
import ru.yandex.direct.operation.update.ChangesAppliedStep;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.FunctionalUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.auction.utils.BsAuctionConverter.convertToPositionsAuctionData;
import static ru.yandex.direct.core.entity.keyword.processing.KeywordProcessingUtils.getLengthWithoutMinusExclamationAndSpaces;
import static ru.yandex.direct.core.entity.keyword.processing.MinusKeywordsDeduplicator.removeDuplicates;
import static ru.yandex.direct.core.entity.keyword.processing.deduplication.AdGroupKeywordDeduplicationUtils.computeDuplicates;
import static ru.yandex.direct.core.entity.keyword.service.KeywordBsAuctionService.getKeywordPlaces;
import static ru.yandex.direct.core.entity.keyword.service.KeywordOperationFactory.ADGROUP_TYPES_NO_AUCTION;
import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.SENSITIVE_PROPERTIES;
import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.cloneKeywords;
import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.isPhraseCoverageSignificantlyChanged;
import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.safeParseWithMinuses;
import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordDefects.duplicatedWithExisting;
import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordDefects.duplicatedWithUpdated;
import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordPredicates.KEYWORD_MISSING_AUTOBUDGET_PRIORITY;
import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordPredicates.KEYWORD_MISSING_CONTEXT_PRICE;
import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordPredicates.KEYWORD_MISSING_SEARCH_PRICE;
import static ru.yandex.direct.core.entity.mailnotification.model.KeywordEvent.changedContextPriceEvent;
import static ru.yandex.direct.core.entity.mailnotification.model.KeywordEvent.changedKeywordPhraseEvent;
import static ru.yandex.direct.core.entity.mailnotification.model.KeywordEvent.changedSearchPriceEvent;
import static ru.yandex.direct.utils.CommonUtils.compressNumbers;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.intRange;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.index;

/**
 * Операция обновления ключевых фраз.
 * <p>
 * В качестве результатов возвращает объекты {@link UpdatedKeywordInfo},
 * которые помимо id добавленного объекта так же содержат информацию
 * о преобразованиях, которые были проведены над фразами.
 * Мапа с результатами создается в начале выполнения операции и заполняется по мере выполнения действий.
 * <p>
 * <p>
 * Параметр {@code autoPrices} включает режим автоматического пересчитывания ставок
 * у фраз, где хотя бы немного поменялся текст. Ставки явно указанные в запросе не
 * пересчитываются. При включении этого режима параметр {@code fixedAutoPrices}
 * конструктора должен быть не {@code null}.
 * <p>
 * При пересчитывании, если стратегия не требует ручной ставки, она
 * сбрасывается в 0. Иначе ставка расчитывается как недостающая в операции
 * добавления.
 * <p>
 * Cначала пытаемся взять фиксированную ставку из контейнера
 * {@code fixedAutoPrices}, но если там ее не нашлось, считаем
 * ставки хитрым образом на основе старых фраз в группе и торгов, с помощью
 * калькулятора {@link KeywordAutoPricesCalculator}. Калькулятор можно передать
 * снаружи через конструктор.
 * <p>
 * Также у фраз будет выставлен приоритет автобюджета по умолчанию, если
 * его нет в запросе, но он нужен в текущей стратегии.
 * <p>
 * <p>
 * Если режим {@code autoPrices} выключен, ставки не пересчитываются.
 */
public class KeywordsUpdateOperation extends AbstractUpdateOperation<Keyword, UpdatedKeywordInfo> {

    /**
     * Таймаут на получение прогнозов из ADVQ по умолчанию.
     * Может не хватать, если много минус фраз.
     */
    private static final Duration DEFAULT_FORECAST_TIMEOUT = Duration.ofSeconds(5);

    /**
     * Таймаут на полученеи прогнозов в случае, если включено вычисление
     * автоставки для новых фраз. При этом ставка, возможно, будет считаться
     * на основе ответа от торгов, а для торгов важны прогнозы.
     * Если в торги передать нулевой прогноз, ставки будут завышены.
     */
    private static final Duration AUTO_PRICE_FORECAST_TIMEOUT = Duration.ofSeconds(15);

    private final FixStopwordsService fixStopwordsService;
    private final KeywordNormalizer keywordNormalizer;
    private final KeywordUngluer keywordUngluer;
    private final UpdateKeywordValidationService updateKeywordValidationService;
    private final ClientService clientService;
    private final ModerationService moderationService;
    private final AutoPriceCampQueueService autoPriceCampQueueService;
    private final AutobudgetAlertService autobudgetAlertService;
    private final MailNotificationEventService mailNotificationEventService;
    private final LogPriceService logPriceService;
    private final KeywordBsAuctionService keywordBsAuctionService;
    private final KeywordModerationService keywordModerationService;
    private final SingleKeywordsCache singleKeywordsCache;
    private final InternalKeywordFactory internalKeywordFactory;
    private final KeywordShowsForecastService keywordShowsForecastService;

    private final DslContextProvider dslContextProvider;
    private final KeywordRepository keywordRepository;
    private final BannerRepository bannerRepository;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final AggregatedStatusesRepository aggregatedStatusesRepository;

    private final KeywordsUpdateOperationParams operationParams;

    /**
     * Фиксированные ставки для новых фраз. Должен быть не {@code null}, если
     * в параметрах включен режим {@code autoPrices}.
     * Контейнер может содержать ставки не для всех фраз, тогда ставки будут
     * вычислены автоматически, см. коммент к классу
     */
    @Nullable
    private final ShowConditionFixedAutoPrices fixedAutoPrices;

    private final long operatorUid;
    private final ClientId clientId;
    private final long clientUid;
    private final int shard;
    private List<Keyword> existingKeywords;
    private Currency clientCurrency;

    /**
     * Мапа результатов для каждой входной ключевой фразы, которые заполняются данными
     * в процессе выполнения операции. Ключами являются индексы соответствующих
     * ключевых фраз во входном списке.
     * <p>
     * На начальном этапе результаты создаются для всех успешно предвалидированных
     * элементов, но после выполнения основной валидации невалидные элементы
     * удаляются и в мапе остаются результаты только для валидных.
     * <p>
     * Важно, что к моменту вызова {@link #apply()} мапа содержит результаты -
     * хоть и не до конца заполненные, - для всех валидных элементов запроса.
     * Таким образом, во время выполнения операции результаты дополняются недостающими
     * данными и отдаются в качестве возвращаемого значения метода {@link #execute(ExecutionStep)}.
     */
    private Map<Integer, UpdatedKeywordInfo> resultMap;

    /**
     * Мапа контейнеров для каждой входной ключевой фразы, которые заполняются
     * вспомогательными данными в процессе выполнения операции. Ключами являются
     * индексы соответствующих ключевых фраз во входном списке.
     * <p>
     * На начальном этапе контейнеры создаются для всех успешно предвалидированных
     * элементов. Затем из нее удаляются элементы, не прошедшие основной этап валидации.
     * Затем из нее удаляются дубликаты существующих фраз, а для каждого набора дубликатов
     * среди новых фраз остается только один, который должен быть обновлен в репозиторий.
     * <p>
     * Ключевым моментом является то, что к моменту вызова {@link #apply()} в ней остаются
     * контейнеры только для тех фраз, которые должны быть отправлены для обновления в репозиторий.
     */
    private Map<Integer, InternalKeyword> updatedKeywordsMap;

    /**
     * Мапа контейнеров для существующих фраз из групп объявлений,
     * которые затронуты валидными элементами запроса,
     * за исключением самих валидных ключевых фраз ({@link #updatedKeywordsMap}).
     * Создается и заполняется в процессе выполнения операции. Нужна для расклейки, поиска дубликатов,
     * и определения фраз, которые необходимо удалить.
     */
    private ImmutableMap<Integer, InternalKeyword> otherExistingKeywordsMap;

    /**
     * Мапа дубликатов добавляемых фраз с существующими, за исключением валидных фраз из запроса
     * <p>
     * Ключами являются индексы фраз из входного списка, а значениями -
     * индексы из мапы существующих ({@link #otherExistingKeywordsMap}).
     * <p>
     * Нужна, чтобы понять, какая валидная обновляемая фраза дублирует другие фразы из группы,
     * и вернуть в ответе для нее id соответствующей фразы и предупреждение.
     */
    private ImmutableMap<Integer, Integer> duplicatesWithOtherExisting;

    /**
     * Мапа дубликатов обновляемых фраз между собой.
     * <p>
     * Ключами и значениями являются индексы фраз из входного списка.
     * <p>
     * Нужна, чтобы добавить только одну фразу из набора таких фраз-дубликатов,
     * а все остальные фразы удалить и вернуть в ответе id фактически обновленной и предупреждение.
     */
    private ImmutableMap<Integer, Collection<Integer>> newDuplicates;

    /**
     * Когда во входном списке присутствуют дублирующиеся между собой фразы,
     * одна из дубликатов обновляются, а остальные дубликаты удаляются и для них
     * возвращаются id оставшейся фразы и предупреждение. Эта мапа в качестве ключей содержит
     * индексы удаляемых дубликатов, а в качестве значений - индексы
     * обновленных. Она нужна, чтобы для проигнорированных дубликатов вернуть
     * id оставшейся обновленной фразы.
     */
    private ImmutableMap<Integer, Integer> newDuplicatesLinkingMap;

    /**
     * Валидные изменения, сохраняемые в данном поле после этапа подготовки,
     * так как они нужны для отдачи наружу всех потенциальных изменений.
     */
    private ImmutableMap<Integer, AppliedChanges<Keyword>> validAppliedChangesMap;

    /**
     * Список изменений, которые необходимо применить к другим существующим фразам.
     * На данный момент изменения в существующих фразах могут быть только
     * результатом расклейки.
     */
    private ImmutableList<AppliedChanges<Keyword>> otherExistingKeywordsToUpdate;

    /**
     * Список объектов, описывающих, какие существующие у клиента ключевые фразы
     * претерпели изменения в результате обновления ключевых фраз,
     * и в чем состояли эти изменения. На данный момент содержит только информацию
     * о расклейке.
     */
    private ImmutableList<AffectedKeywordInfo> affectedKeywordInfoList;

    /**
     * Список моделей ключевых фраз, подлежащих удалению.
     */
    private ImmutableList<Keyword> keywordsToDelete;

    /**
     * Дополнительные действия, которые необходимо выполнить в одной транзакции
     * с обновлением фраз для обновляемых фраз.
     */
    private TransactionalRunnable transactionalAdditionalUpdateTask;

    /**
     * Дополнительные действия, которые необходимо выполнить в одной транзакции
     * с обновлением фраз для удаляемых фраз.
     */
    private TransactionalRunnable transactionalAdditionalDeleteTask;

    /**
     * Индексы групп по индексам ключевых фраз. Используется для линковки фраз и групп на этапе {@link #prepare()}.
     * Будут выведены из id групп (см. {@link ru.yandex.direct.utils.CommonUtils#compressNumbers(Map)})
     */
    private Map<Integer, Integer> keywordAdGroupIndexes;

    /**
     * Соответствие индексов групп из {@link #keywordAdGroupIndexes} их id.
     * Используется для вычисления индексов групп для существующих фраз.
     */
    private Map<Long, Integer> adGroupIndexById;

    /**
     * Информация о торгах по индексам ключевых фраз, содержится только по индексам валидных изменённых КФ.
     * <p>
     * Результат не добавляется напрямую в ответ, но сохраняется на этапе выполнения операции,
     * чтобы в случае необходимости не запрашивать повторно.
     * <p>
     * Может не содержать данных для некоторых фраз, для которых не удалось получить данные торгов.
     */
    private ImmutableMap<Integer, KeywordTrafaretData> trafaretDataByIndex;

    /**
     * Калькулятор недостающих ставок. Может быть передан снаружи через
     * конструктор, тогда он будет не {@code null} до вызова
     * {@code calcAutoPrices}.
     * Если калькулятор не передали снаружи, и в параметрах операции включен
     * режим {@code autoPrices}, метод {@code calcAutoPrices} сам создаст
     * калькулятор и положит в это поле.
     */
    @Nullable
    private KeywordAutoPricesCalculator keywordAutoPricesCalculator;

    /**
     * Дополнительные действия, которые выполняются вне транзакции.
     */
    private Runnable additionalUpdateTask;
    private Runnable additionalDeleteTask;

    /**
     * @param fixedAutoPrices      контейнер с фиксированными ставками, которые нужно выставить у фразы, если ставку
     *                             нужно пересчитать, см коммент к классу. Могут быть ставки не для всех фраз.
     *                             Должен быть не {@code null}, если в параметрах операции включен
     *                             режим {@code autoPrices}
     * @param autoPricesCalculator калькулятор недостающих ставок во фразах. Может быть {@code null}, тогда операция
     *                             создаст свой калькулятор
     */
    public KeywordsUpdateOperation(Applicability applicability, KeywordsUpdateOperationParams operationParams,
                                   List<ModelChanges<Keyword>> modelChangesList,
                                   FixStopwordsService fixStopwordsService,
                                   KeywordNormalizer keywordNormalizer,
                                   KeywordUngluer keywordUngluer,
                                   UpdateKeywordValidationService updateKeywordValidationService,
                                   ClientService clientService,
                                   ModerationService moderationService,
                                   AutoPriceCampQueueService autoPriceCampQueueService,
                                   AutobudgetAlertService autobudgetAlertService,
                                   MailNotificationEventService mailNotificationEventService,
                                   LogPriceService logPriceService,
                                   KeywordBsAuctionService keywordBsAuctionService,
                                   KeywordModerationService keywordModerationService,
                                   SingleKeywordsCache singleKeywordsCache,
                                   InternalKeywordFactory internalKeywordFactory,
                                   KeywordShowsForecastService keywordShowsForecastService,
                                   DslContextProvider dslContextProvider,
                                   KeywordRepository keywordRepository,
                                   BannerRepository bannerRepository,
                                   AdGroupRepository adGroupRepository,
                                   CampaignRepository campaignRepository,
                                   AggregatedStatusesRepository aggregatedStatusesRepository,
                                   long operatorUid, ClientId clientId, long clientUid, int shard,
                                   @Nullable List<Keyword> existingKeywords,
                                   @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
                                   @Nullable KeywordAutoPricesCalculator autoPricesCalculator) {
        super(applicability, modelChangesList, id -> new Keyword().withId(id), SENSITIVE_PROPERTIES);

        this.operationParams = operationParams;
        this.fixStopwordsService = fixStopwordsService;
        this.keywordNormalizer = keywordNormalizer;
        this.keywordUngluer = keywordUngluer;
        this.updateKeywordValidationService = updateKeywordValidationService;
        this.clientService = clientService;
        this.moderationService = moderationService;
        this.autoPriceCampQueueService = autoPriceCampQueueService;
        this.autobudgetAlertService = autobudgetAlertService;
        this.mailNotificationEventService = mailNotificationEventService;
        this.logPriceService = logPriceService;
        this.keywordBsAuctionService = keywordBsAuctionService;
        this.keywordModerationService = keywordModerationService;
        this.singleKeywordsCache = singleKeywordsCache;
        this.internalKeywordFactory = internalKeywordFactory;
        this.keywordShowsForecastService = keywordShowsForecastService;
        this.dslContextProvider = dslContextProvider;
        this.keywordRepository = keywordRepository;
        this.bannerRepository = bannerRepository;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;

        this.operatorUid = operatorUid;
        this.clientId = clientId;
        this.clientUid = clientUid;
        this.shard = shard;

        if (existingKeywords != null) {
            Set<Long> uniqueIds = listToSet(existingKeywords, Keyword::getId);
            checkArgument(uniqueIds.size() == existingKeywords.size(),
                    "existing keywords list has keywords with duplicated ids");
            this.existingKeywords = existingKeywords;
        }

        if (operationParams.isAutoPrices()) {
            checkNotNull(fixedAutoPrices, "must specify fixedAutoPrices in autoPrices mode");
        }
        this.fixedAutoPrices = fixedAutoPrices;
        this.keywordAutoPricesCalculator = autoPricesCalculator;

        otherExistingKeywordsToUpdate = ImmutableList.of();
        affectedKeywordInfoList = ImmutableList.of();
    }

    /**
     * Возвращает список объектов, описывающих, какие существующие фразы и каким образом
     * были затронуты при выполнении данной операции. Например, каким существующим фразам
     * какие минус-слова были добавлены при расклейке.
     * <p>
     * Может быть вызван только после вызова {@link #prepare()}.
     */
    public ImmutableList<AffectedKeywordInfo> getAffectedKeywordInfoList() {
        checkState(isPrepared(), "affected keywords info list is available only after preparing");
        return affectedKeywordInfoList;
    }

    /**
     * Возвращает информацию о торгах по индексам обновлённых КФ.
     * <p>
     * Может быть вызван только после вызова {@link #apply()}.
     * <p>
     * Может не вернуть данные для некоторых фраз, для которых не удалось получить данные торгов.
     */
    ImmutableMap<Integer, List<TrafaretBidItem>> getTrafaretBidsByIndexMap() {
        checkState(isExecuted(), "result of auction is available only after executing");
        return ImmutableMap.copyOf(EntryStream.of(trafaretDataByIndex)
                .mapValues(KeywordTrafaretData::getBidItems)
                .toMap());
    }

    /**
     * Возвращает список ключевых фраз, которые которые будут обновлены при вызове {@link #apply()}
     * <p>
     * Может быть вызван только после вызова {@link #prepare()}.
     */
    Collection<AppliedChanges<Keyword>> getPotentiallyUpdatedKeywords() {
        checkState(isPrepared(), "potentially updated keywords list is available only after preparing");
        return EntryStream.of(validAppliedChangesMap)
                .filterKeys(updatedKeywordsMap::containsKey)
                .values()
                .append(otherExistingKeywordsToUpdate)
                .toList();
    }

    /**
     * Возвращает сет id ключевых фраз, которые которые будут удалены при вызове {@link #apply()}
     * <p>
     * Может быть вызван только после вызова {@link #prepare()}.
     */
    public Set<Long> getPotentiallyDeletedKeywordIds() {
        checkState(isPrepared(), "potentially deleted keyword ids list is available only after preparing");
        return listToSet(keywordsToDelete, Keyword::getId);
    }

    @Override
    protected ValidationResult<List<ModelChanges<Keyword>>, Defect> validateModelChanges(
            List<ModelChanges<Keyword>> modelChanges) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:validateModelChanges")) {
            clientCurrency = clientService.getWorkCurrency(clientId);
            Set<Long> keywordIds = listToSet(modelChanges, ModelChanges::getId);
            initExistingKeywords(keywordIds);

            Set<Long> existingKeywordIds = listToSet(existingKeywords, Keyword::getId);

            Set<Long> suspendedIds = existingKeywords.stream()
                    .filter(Keyword::getIsSuspended)
                    .map(Keyword::getId)
                    .collect(toSet());
            return updateKeywordValidationService
                    .preValidate(modelChanges, existingKeywordIds, suspendedIds);
        }
    }

    /**
     * Загружает существующие фразы из групп, которые учавствуют в запросе
     *
     * @param keywordIds список id ключевых фраз из запроса
     */
    private void initExistingKeywords(Set<Long> keywordIds) {
        if (existingKeywords == null) {
            try (TraceProfile ignore = Trace.current().profile("keywords.update:initExistingKeywords")) {
                Set<Long> adGroupIds = keywordRepository.getAdGroupIdsByKeywordsIds(shard, clientId, keywordIds);
                existingKeywords = getKeywordsByAdGroupIds(shard, clientId, adGroupIds);
            }
        }
    }

    @Override
    protected Collection<Keyword> getModels(Collection<Long> ids) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:getModels")) {
            Set<Long> idsSet = new HashSet<>(ids);
            // т.к. над этими ключевыми фразами будут применяться изменения, то создаем копию существующих
            return cloneKeywords(filterList(existingKeywords, keyword -> idsSet.contains(keyword.getId())));
        }
    }

    @Override
    protected void onChangesApplied(ChangesAppliedStep<Keyword> changesAppliedStep) {
        Map<Integer, AppliedChanges<Keyword>> appliedChangesMap =
                changesAppliedStep.getAppliedChangesForValidModelChangesWithIndex();
        initMaps(appliedChangesMap);
        deduplicateMinusWords(appliedChangesMap);
        unfixStopwordsInQuotes();
        fixStopwords(appliedChangesMap);
    }

    private void initMaps(Map<Integer, AppliedChanges<Keyword>> preValidKeywordsMap) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:initMaps")) {
            updatedKeywordsMap = EntryStream.of(preValidKeywordsMap)
                    .mapValues(AppliedChanges::getModel)
                    .mapValues(kw -> new InternalKeyword(kw, safeParseWithMinuses(kw.getPhrase())))
                    .filterValues(kw -> kw.getParsedKeyword() != null)
                    .toMap();
            resultMap = EntryStream.of(preValidKeywordsMap)
                    .mapValues(kw -> new UpdatedKeywordInfo()
                            .withId(kw.getModel().getId())
                            .withIsSuspended(kw.getModel().getIsSuspended()))
                    .toMap();
        }
    }

    /**
     * Удаляет дублирующиеся минус-слова в исходных фразах.
     * Дубликаты считаются по тому же алгоритму, что и для минус-фраз.
     */
    private void deduplicateMinusWords(Map<Integer, AppliedChanges<Keyword>> appliedChangesMap) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:deduplicateMinusWords")) {
            Map<Integer, List<ru.yandex.direct.libs.keywordutils.model.Keyword>> deduplicatedMinusWordsLists =
                    EntryStream.of(updatedKeywordsMap)
                            .mapValues(InternalKeyword::getParsedKeyword)
                            .mapValues(kw -> removeDuplicates(singleKeywordsCache, kw.getMinusKeywords()))
                            .toMap();

            deduplicatedMinusWordsLists.forEach((index, deduplicatedMinusWords) -> {
                InternalKeyword internalKeyword = updatedKeywordsMap.get(index);

                KeywordWithMinuses oldKeyword = internalKeyword.getParsedKeyword();
                if (oldKeyword.getMinusKeywords().size() != deduplicatedMinusWords.size()) {
                    KeywordWithMinuses resultKeyword =
                            new KeywordWithMinuses(oldKeyword.getKeyword(), deduplicatedMinusWords);
                    internalKeyword.withParsedKeyword(resultKeyword);

                    AppliedChanges<Keyword> changes = appliedChangesMap.get(index);
                    changes.modify(Keyword.PHRASE, resultKeyword.toString());

                    UpdatedKeywordInfo updatedKeywordInfo = resultMap.get(index);
                    updatedKeywordInfo.withResultPhrase(resultKeyword.toString());
                }
            });
        }
    }


    /**
     * Удаляет знак "+" во фразах внутри кавычек.
     */
    private void unfixStopwordsInQuotes() {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:unfixStopwordsInQuotes")) {
            Map<Integer, KeywordWithMinuses> unfixStopwordsResultsMap = EntryStream.of(updatedKeywordsMap)
                    .mapValues(InternalKeyword::getParsedKeyword)
                    .mapValues(KeywordProcessingUtils::cleanWordsInBracketsFromPlus)
                    .toMap();

            unfixStopwordsResultsMap.forEach((index, unfixStopwords) -> {
                InternalKeyword internalKeyword = updatedKeywordsMap.get(index);

                internalKeyword.withParsedKeyword(unfixStopwords);
                internalKeyword.getKeyword().setPhrase(unfixStopwords.toString());

                UpdatedKeywordInfo updatedKeywordInfo = resultMap.get(index);
                updatedKeywordInfo.withResultPhrase(unfixStopwords.toString());
            });
        }
    }

    /**
     * Фиксирует стоп-слова в исходных фразах в моделях, путем модификации примененных изменений.
     * Добавляет информацию о фиксации в результаты
     */
    private void fixStopwords(Map<Integer, AppliedChanges<Keyword>> appliedChangesMap) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:fixStopwords")) {
            Map<Integer, FixStopwordsResult> fixStopwordsResultsMap = EntryStream.of(updatedKeywordsMap)
                    .mapValues(InternalKeyword::getParsedKeyword)
                    .mapValues(fixStopwordsService::fixStopwords)
                    .toMap();

        /*
         Во входных данных заменяются даже те фразы, в которых ни одно стоп-слово не было зафиксировано
         или не было изменений во фразе.
         Это приводит к тому, что криво написанные фразы, например, без пробелов в нужных местах
         или с лишними пробелами, приводятся к правильному виду.
         */
            fixStopwordsResultsMap.forEach((index, fixResult) -> {
                KeywordWithMinuses resultKeyword = fixResult.getResult();

                InternalKeyword internalKeyword = this.updatedKeywordsMap.get(index);
                internalKeyword.withParsedKeyword(resultKeyword);

                AppliedChanges<Keyword> changes = appliedChangesMap.get(index);
                changes.modify(Keyword.PHRASE, resultKeyword.toString());

                UpdatedKeywordInfo updatedKeywordInfo = resultMap.get(index);
                updatedKeywordInfo.withResultPhrase(resultKeyword.toString());
                if (fixResult.isFixed()) {
                    updatedKeywordInfo.withFixations(fixResult.getFixations());
                }
            });
        }
    }

    @Override
    protected ValidationResult<List<Keyword>, Defect> validateAppliedChanges(
            ValidationResult<List<Keyword>, Defect> validationResult) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:validateAppliedChanges")) {
            return updateKeywordValidationService
                    .validate(validationResult, updatedKeywordsMap, operatorUid, clientId);
        }
    }

    @Override
    protected void onAppliedChangesValidated(AppliedChangesValidatedStep<Keyword> appliedChangesValidatedStep) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:onAppliedChangesValidated")) {
            Map<Integer, AppliedChanges<Keyword>> appliedChangesMap =
                    appliedChangesValidatedStep.getValidAppliedChangesWithIndex();
            this.validAppliedChangesMap = ImmutableMap.copyOf(appliedChangesMap);

            clearMapsFromInvalidItems(appliedChangesMap);
            initAdGroupInfo();
            initAdGroupIndexById();
            normalizeKeywords();
            initOtherExistingKeywordsMap();
            if (operationParams.isPhraseModificationDisabled()) {
                duplicatesWithOtherExisting = ImmutableMap.of();
                newDuplicates = ImmutableMap.of();
            } else {
                computeDuplicatedIndexes();
            }
            determineNewDuplicatesToUpdateOrDeleteAndCleanUpdatedKeywordsMap();
            setPhrasesBeforeUnglueInResults();

            if (operationParams.isUnglueEnabled() && !operationParams.isPhraseModificationDisabled()) {
                unglue(appliedChangesMap);
            }
        }
    }

    /**
     * Очищает мапы с промежуточными данными по ключевым фразам и результатами от невалидных элементов.
     * <p>
     * Вызывается после валидации примененных изменений
     *
     * @param validAppliedChangesMap мапа валидных примененных изменений
     */
    private void clearMapsFromInvalidItems(Map<Integer, AppliedChanges<Keyword>> validAppliedChangesMap) {
        updatedKeywordsMap = EntryStream.of(updatedKeywordsMap)
                .filterKeys(validAppliedChangesMap::containsKey)
                .toMap();
        resultMap = EntryStream.of(resultMap)
                .filterKeys(validAppliedChangesMap::containsKey)
                .toMap();
    }

    /**
     * Выводит индексы групп из id'шников
     */
    private void initAdGroupInfo() {
        Map<Integer, Long> adGroupIds = EntryStream.of(updatedKeywordsMap)
                .mapValues(InternalKeyword::getAdGroupId)
                .toMap();
        keywordAdGroupIndexes = compressNumbers(adGroupIds);
    }

    /**
     * Для получения индекса группы у существующих фраз, нужна его зависимость от id
     */
    private void initAdGroupIndexById() {
        adGroupIndexById = new HashMap<>();
        updatedKeywordsMap.forEach((keywordIndex, internalKeyword) -> adGroupIndexById
                .put(internalKeyword.getAdGroupId(), keywordAdGroupIndexes.get(keywordIndex)));
    }

    /**
     * Нормализует ключевые слова с предварительно зафиксированными по определенным
     * правилам стоп-словами и добавляет их в соответствующие контейнеры {@link InternalKeyword}.
     * Нормализация должна идти после валидации для того, чтобы в нее не попадали невалидные ключевые фразы
     */
    private void normalizeKeywords() {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:normalizeKeywords")) {
            updatedKeywordsMap.forEach((index, kw) -> {
                KeywordWithMinuses rawKw = kw.getParsedKeyword();
                KeywordWithMinuses normalizedKw = keywordNormalizer.normalizeKeywordWithMinuses(rawKw);
                kw.withParsedNormalKeyword(normalizedKw);
            });
        }
    }

    /**
     * Выбираем существующие ключевые фразы пользователя,
     * за исключением валидных ключевые фраз из запроса ({@link #updatedKeywordsMap})
     * и инициализирует для них фразы в разобранном и нормализованном виде.
     */
    private void initOtherExistingKeywordsMap() {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:initOtherExistingKeywordsMap")) {
            Set<Long> validModelIds = listToSet(updatedKeywordsMap.values(), InternalKeyword::getId);
            Set<Long> validModelAdGroupIds = listToSet(updatedKeywordsMap.values(), InternalKeyword::getAdGroupId);
            List<Keyword> otherExistingKeywordsLocal = new ArrayList<>(existingKeywords);
            otherExistingKeywordsLocal
                    .removeIf(k -> validModelIds.contains(k.getId()) || !validModelAdGroupIds.contains(k.getAdGroupId()));

            this.otherExistingKeywordsMap = ImmutableMap.copyOf(StreamEx.of(intRange(0,
                    otherExistingKeywordsLocal.size()))
                    .mapToEntry(identity(), otherExistingKeywordsLocal::get)
                    .mapValues(internalKeywordFactory::createInternalKeyword)
                    .nonNullValues()
            );
        }
    }

    /**
     * Вычисляет индексы дубликатов.
     */
    private void computeDuplicatedIndexes() {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:computeDuplicatedIndexes")) {
            List<DuplicatingContainer> existingDuplContainers = EntryStream.of(otherExistingKeywordsMap)
                    .mapKeyValue((index, internalKeyword) -> new DuplicatingContainer(index,
                            adGroupIndexById.get(internalKeyword.getAdGroupId()),
                            internalKeyword.getParsedNormalKeyword(),
                            internalKeyword.getKeyword().getIsAutotargeting()))
                    .toList();

            List<DuplicatingContainer> newDuplContainers = EntryStream.of(updatedKeywordsMap)
                    .mapKeyValue((index, internalKeyword) -> new DuplicatingContainer(index,
                            keywordAdGroupIndexes.get(index), internalKeyword.getParsedNormalKeyword(),
                            internalKeyword.getKeyword().getIsAutotargeting()))
                    .toList();

            duplicatesWithOtherExisting = ImmutableMap.copyOf(computeDuplicates(newDuplContainers,
                    existingDuplContainers));
            newDuplicates = ImmutableMap.copyOf(computeDuplicates(newDuplContainers).asMap());
        }
    }

    /**
     * Определяет, какие ключевые фразы, продублированные в запросе,
     * будут отправлены в репозиторий на сохранение, а какие - на удаление.
     * <p>
     * Фразы из запроса, дублирующие другие существующие фразы, безусловно
     * не будут отправлены на сохранение в репозиторий,
     * они будут удалены из базы, а также из мапы {@link #updatedKeywordsMap}.
     * <p>
     * Для фраз из запроса, которые не дублируют другие существующие фразы,
     * но при этом дублируются с фразами из запроса, действует
     * следующее правило: выбирается одна фраза, которая будет фактически
     * обновлена, а остальные удаляются, при этом запоминается маппинг
     * индексов проигнорированных фраз на фактически обновленные
     * ({@link #newDuplicatesLinkingMap}), чтобы после обновления
     * в их результаты проставить id фактически обновленной фразы
     * (и предупреждение).
     */
    private void determineNewDuplicatesToUpdateOrDeleteAndCleanUpdatedKeywordsMap() {
        try (TraceProfile ignore = Trace.current()
                .profile("keywords.update:determineNewDuplicatesToUpdateOrDeleteAndCleanUpdatedKeywordsMap")) {
            List<Keyword> keywordsToDeleteLocal = new ArrayList<>();
            duplicatesWithOtherExisting.keySet().forEach(kw -> {
                InternalKeyword removedKeyword = updatedKeywordsMap.remove(kw);
                checkNotNull(removedKeyword, "inconsistent state");
                keywordsToDeleteLocal.add(removedKeyword.getKeyword());
            });

            Map<Integer, Integer> newDuplicatesLinkingMapLocal = new HashMap<>();
            newDuplicates.forEach((duplicatedIndex, indexesOfDuplicates) -> {
                // если данная фраза не дублируется с существующей,
                // тогда имеет смысл думать о ее обновлении
                if (updatedKeywordsMap.containsKey(duplicatedIndex)) {
                    // если в мапе уже присутствует данный индекс в качестве удаленного,
                    // то пропускаем итерацию
                    if (!newDuplicatesLinkingMapLocal.containsKey(duplicatedIndex)) {
                        // если данный индекс еще не попадался среди дубликатов, то добавляем его,
                        // а его дубликаты линкуем на него, чтобы в результате для них отдать его id
                        // при этом убираем удаляемые дубликаты из мапы обновления фраз
                        indexesOfDuplicates.forEach(indexOfDuplicate -> {
                            newDuplicatesLinkingMapLocal.put(indexOfDuplicate, duplicatedIndex);
                            InternalKeyword removedKeyword = updatedKeywordsMap.remove(indexOfDuplicate);
                            checkNotNull(removedKeyword, "inconsistent state");
                            keywordsToDeleteLocal.add(removedKeyword.getKeyword());
                        });
                    }
                }
            });
            newDuplicatesLinkingMap = ImmutableMap.copyOf(newDuplicatesLinkingMapLocal);
            keywordsToDelete = ImmutableList.copyOf(keywordsToDeleteLocal);
        }
    }

    private void setPhrasesBeforeUnglueInResults() {
        updatedKeywordsMap.keySet().forEach(index -> {
            UpdatedKeywordInfo updatedKeywordInfo = resultMap.get(index);
            String phraseBeforeUnglue = updatedKeywordInfo.getResultPhrase();
            updatedKeywordInfo.withPhraseBeforeUnglue(phraseBeforeUnglue);
        });
    }

    /**
     * Расклеивает фактически обновляемые фразы (то есть валидные за вычетом удаляемых дубликатов)
     * с другими существующими и между собой. Подробнее о работе расклейки читать здесь: {@link KeywordUngluer}.
     * <p>
     * Сначала вызывает утилиту расклейки ({@link KeywordUngluer}) и получает
     * от нее результат, описывающий, как потенциально можно расклеить фразы.
     * <p>
     * Далее происходит фактическая расклейка: обновляемым фразам и другим существующим добавляются минус-слова,
     * при этом минус-слова добавляются таким образом, чтобы не появились новые дубликаты,
     * и чтобы фразы не превысили ограничение на максимальную длину.
     */
    private void unglue(Map<Integer, AppliedChanges<Keyword>> appliedChangesMap) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:unglue")) {
            List<UnglueContainer> newUnglueContainers = EntryStream.of(updatedKeywordsMap)
                    .mapKeyValue((index, kw) -> new UnglueContainer(index, keywordAdGroupIndexes.get(index),
                            kw.getParsedNormalKeyword()))
                    .toList();
            List<UnglueContainer> existingUnglueContainers = EntryStream.of(otherExistingKeywordsMap)
                    .mapKeyValue((index, kw) -> new UnglueContainer(index, adGroupIndexById.get(kw.getAdGroupId()),
                            kw.getParsedNormalKeyword()))
                    .toList();

            List<UnglueResult> unglueResults = keywordUngluer.unglue(newUnglueContainers, existingUnglueContainers);

            // сет нормальных форм фраз для отслеживания дубликатов
            Set<String> currentSummaryNormalKeywords = StreamEx.of(updatedKeywordsMap.values())
                    .append(otherExistingKeywordsMap.values())
                    .map(InternalKeyword::getParsedNormalKeyword)
                    .map(KeywordWithMinuses::toString)
                    .toSet();

            unglueUpdateKeywords(appliedChangesMap, unglueResults, currentSummaryNormalKeywords);
            unglueExistingKeywords(unglueResults, currentSummaryNormalKeywords);
        }
    }

    /**
     * Добавление минус-слов к обновляемым фразам на основе результатов расклейки.
     * <p>
     * Результаты расклейки ({@link UnglueResult}) носят рекомендательный характер,
     * и, таким образом, к каждой фразе может быть добавлено любое подмножество
     * рекомендованных минус-слов. Эти минус-слова добавляются добавляются таким образом,
     * чтобы не появились новые дубликаты, и чтобы фразы не превысили ограничение на максимальную длину.
     *
     * @param unglueResults результаты расклейки.
     */
    private void unglueUpdateKeywords(Map<Integer, AppliedChanges<Keyword>> appliedChangesMap,
                                      List<UnglueResult> unglueResults, Set<String> currentSummaryNormalKeywords) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:unglueUpdateKeywords")) {
            for (UnglueResult unglueResult : unglueResults) {
                if (unglueResult.getAddedMinusWords().isEmpty()) {
                    continue;
                }

                int index = unglueResult.getIndex();
                AppliedChanges<Keyword> changes = appliedChangesMap.get(index);
                InternalKeyword internalKeyword = updatedKeywordsMap.get(index);
                KeywordWithMinuses parsedKeyword = internalKeyword.getParsedKeyword();
                KeywordWithMinuses parsedNormalKeyword = internalKeyword.getParsedNormalKeyword();

                List<ru.yandex.direct.libs.keywordutils.model.Keyword> minusWordsToAdd =
                        mapList(unglueResult.getNormalizedAddedMinusWords(),
                                skw -> new ru.yandex.direct.libs.keywordutils.model.Keyword(false, singletonList(skw)));

                while (!minusWordsToAdd.isEmpty()) {
                    KeywordWithMinuses newParsedKeyword = parsedKeyword.appendMinuses(minusWordsToAdd);
                    int newLength = getLengthWithoutMinusExclamationAndSpaces(newParsedKeyword.toString());

                    if (newLength <= PhraseConstraints.KEYWORD_MAX_LENGTH) {
                        KeywordWithMinuses newParsedNormalKeyword =
                                keywordNormalizer.normalizeKeywordWithMinuses(newParsedKeyword);
                        String newNormalPhrase = newParsedNormalKeyword.toString();

                        // проверяем, что у нормальной формы новой фразы нет дубликатов
                        if (!currentSummaryNormalKeywords.contains(newNormalPhrase)) {

                            changes.modify(Keyword.PHRASE, newParsedKeyword.toString());
                            internalKeyword.withParsedKeyword(newParsedKeyword)
                                    .withParsedNormalKeyword(newParsedNormalKeyword);

                            // добавляем информацию о расклейке в соответствующий результат
                            resultMap.get(index)
                                    .withResultPhrase(newParsedKeyword.toString())
                                    .withAddedMinuses(mapList(minusWordsToAdd, Object::toString));

                            // обновляем сет с текущим набором нормальных форм
                            currentSummaryNormalKeywords.remove(parsedNormalKeyword.toString());
                            currentSummaryNormalKeywords.add(newNormalPhrase);

                            break;
                        }
                    }

                    // пытаемся добавить на 1 слово меньше в следующей итерации
                    minusWordsToAdd.remove(minusWordsToAdd.size() - 1);
                }
            }
        }
    }

    /**
     * Добавление минус-слов к другим существующим фразам на основе результатов расклейки.
     * <p>
     * Результаты расклейки ({@link UnglueResult}) носят рекомендательный характер,
     * и, таким образом, к каждой фразе может быть добавлено любое подмножество
     * рекомендованных минус-слов. Эти минус-слова добавляются с учетом ограничения
     * на максимальную длину, а так же таким образом, чтобы не создать новые дубликаты.
     *
     * @param unglueResults результаты расклейки.
     */
    private void unglueExistingKeywords(List<UnglueResult> unglueResults, Set<String> currentSummaryNormalKeywords) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:unglueExistingKeywords")) {
            List<AppliedChanges<Keyword>> existingKeywordsToUpdateLocal = new ArrayList<>();
            List<AffectedKeywordInfo> affectedKeywordInfoListLocal = new ArrayList<>();

            Map<Integer, List<SingleKeyword>> minusWordsForExistingKeywords = StreamEx.of(unglueResults)
                    .flatMapToEntry(UnglueResult::getNormalizedAddedMinusWordsToExistingMap)
                    .grouping();

            minusWordsForExistingKeywords.forEach((index, minusWords) -> {

                InternalKeyword internalKeyword = otherExistingKeywordsMap.get(index);
                KeywordWithMinuses parsedKeyword = internalKeyword.getParsedKeyword();
                KeywordWithMinuses parsedNormalKeyword = internalKeyword.getParsedNormalKeyword();

                List<ru.yandex.direct.libs.keywordutils.model.Keyword> minusWordsToAdd =
                        mapList(minusWords,
                                skw -> new ru.yandex.direct.libs.keywordutils.model.Keyword(false, singletonList(skw)));

                while (!minusWordsToAdd.isEmpty()) {
                    KeywordWithMinuses newParsedKeyword = parsedKeyword.appendMinuses(minusWordsToAdd);
                    int newLength = getLengthWithoutMinusExclamationAndSpaces(newParsedKeyword.toString());

                    if (newLength <= PhraseConstraints.KEYWORD_MAX_LENGTH) {
                        KeywordWithMinuses newParsedNormalKeyword =
                                keywordNormalizer.normalizeKeywordWithMinuses(newParsedKeyword);
                        String newNormalPhrase = newParsedNormalKeyword.toString();

                        // проверяем, что у нормальной формы новой фразы нет дубликатов
                        if (!currentSummaryNormalKeywords.contains(newNormalPhrase)) {
                            AppliedChanges<Keyword> appliedChanges =
                                    new ModelChanges<>(internalKeyword.getId(), Keyword.class)
                                            .process(newParsedKeyword.toString(), Keyword.PHRASE)
                                            .process(LocalDateTime.now(), Keyword.MODIFICATION_TIME)
                                            .applyTo(internalKeyword.getKeyword());

                            existingKeywordsToUpdateLocal.add(appliedChanges);

                            // добавляем дополнительную информацию о расклейке,
                            // которую клиент сможет получить после выполнения операции
                            String sourcePhrase = parsedKeyword.toString();
                            List<String> addedMinusWords = mapList(minusWordsToAdd, Object::toString);
                            affectedKeywordInfoListLocal
                                    .add(new AffectedKeywordInfo(internalKeyword.getId(),
                                            internalKeyword.getAdGroupId(),
                                            sourcePhrase, addedMinusWords,
                                            internalKeyword.getKeyword().getIsSuspended()));

                            // обновляем сет с текущим набором нормальных форм
                            currentSummaryNormalKeywords.remove(parsedNormalKeyword.toString());
                            currentSummaryNormalKeywords.add(newNormalPhrase);

                            break;
                        }
                    }

                    // пытаемся добавить на 1 слово меньше в следующей итерации
                    minusWordsToAdd.remove(minusWordsToAdd.size() - 1);
                }
            });

            Set<Long> uniqueAffectedIds = listToSet(affectedKeywordInfoListLocal, AffectedKeywordInfo::getId);
            checkState(affectedKeywordInfoListLocal.size() == uniqueAffectedIds.size(),
                    "affectedKeywordInfoList contains duplicated ids");

            otherExistingKeywordsToUpdate = ImmutableList.copyOf(existingKeywordsToUpdateLocal);
            affectedKeywordInfoList = ImmutableList.copyOf(affectedKeywordInfoListLocal);
        }
    }

    @Override
    protected void beforeExecution(ExecutionStep<Keyword> executionStep) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:beforeExecution")) {
            List<ModelChanges<Keyword>> modelChanges = executionStep.getModelChanges();
            Map<Integer, AppliedChanges<Keyword>> actualKeywordsToUpdate = EntryStream.of(validAppliedChangesMap)
                    .filterKeys(updatedKeywordsMap::containsKey)
                    .toMap();

            List<Long> campaignIds = mapList(actualKeywordsToUpdate.values(),
                    changes -> changes.getModel().getCampaignId());
            Map<Long, Campaign> campaignByIds = campaignRepository.getCampaignsMap(shard, campaignIds);

            // поход за прогнозами показов должен быть до похода в торги, т.к.
            // торгам нужны прогнозы
            getShowsForecast(actualKeywordsToUpdate, campaignByIds);
            // обновление полей модели, которые связаны с текстом фразы.
            // Важно сделать это до похода в торги, т.к. там они используются.
            // Так же при автоматическом расчете ставок, который работает после этого вызова,
            // автоставка выставляется только тем фразам, у которых изменился NORM_PHRASE
            // (нормальная форма без учета минус-слов), а NORM_PHRASE выставляется в этом методе.
            processPhraseTextChange(actualKeywordsToUpdate);
            // Расчет отсутствующих ставок и позиции показов на поиске.
            // Эти расчеты должны обязательно проводиться на этапе apply(), т.к.
            // для них нужно ходить в торги, а при походе в торги мы читаем данные
            // баннеров из базы.
            // В случае, когда выполняется комплексная операция обновления группы,
            // всех нужных данных может еще не быть в базе на этапе prepare().
            // Потенциальная точка рефакторинга.
            requestTrafaretAuction(actualKeywordsToUpdate, campaignByIds);
            if (operationParams.isAutoPrices()) {
                Map<Integer, AppliedChanges<Keyword>> keywordsToRecalcPrice =
                        getKeywordsToRecalcPrice(actualKeywordsToUpdate);
                Set<Long> campaignIdsToRecalcPrice =
                        listToSet(keywordsToRecalcPrice.values(), ac -> ac.getModel().getCampaignId());
                Map<Long, Campaign> campaigns = EntryStream.of(campaignByIds)
                        .filterKeys(campaignIdsToRecalcPrice::contains)
                        .toMap();

                resetPrices(keywordsToRecalcPrice, modelChanges);
                setFixedAutoPrices(keywordsToRecalcPrice, campaigns);
                calcAutoPrices(keywordsToRecalcPrice, campaigns);
                setMissingAutobudgetPriorities(keywordsToRecalcPrice, campaigns);
            }
            calcSearchPlaces(actualKeywordsToUpdate);
            // Подготовка служебных полей обязательно должна быть после расчета
            // ставок, т.к. они могли там измениться.
            prepareSystemFields(actualKeywordsToUpdate);
            // эти методы должены следовать за prepareSystemFields,
            // так как пользуется служебными полями Keyword
            computeAdditionalTasksForUpdate(actualKeywordsToUpdate);
            computeAdditionalTasksForDelete(keywordsToDelete);
        }
    }

    /**
     * Обработка ситуаций, когда у фразы поменялся текст.
     */
    private void processPhraseTextChange(Map<Integer, AppliedChanges<Keyword>> keywordsChangedPhrases) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:processPhraseTextChange")) {
            keywordsChangedPhrases.forEach((idx, change) -> {
                // эти значения будут равны null, если старая фраза была невалидна
                KeywordWithMinuses oldKeyword = safeParseWithMinuses(change.getOldValue(Keyword.PHRASE));
                KeywordWithMinuses oldNormalKeyword = keywordNormalizer.safeNormalizeKeywordWithMinuses(oldKeyword);

                KeywordWithMinuses updatedKeyword = updatedKeywordsMap.get(idx).getParsedKeyword();
                KeywordWithMinuses updatedNormalKeyword = updatedKeywordsMap.get(idx).getParsedNormalKeyword();

                // нормальную форму фразы обновляем безусловно,
                // но обновление полей попадет в sql-запрос только если новые значения не совпадают со старыми
                int wordsCount = updatedNormalKeyword.getKeyword().getAllKeywords().size();
                change.modify(Keyword.NORM_PHRASE, updatedNormalKeyword.getKeyword().toString());
                change.modify(Keyword.WORDS_COUNT, wordsCount);

                if (oldNormalKeyword == null || oldKeyword == null ||
                        keywordModerationService.checkModerateKeyword(oldNormalKeyword, updatedNormalKeyword,
                                oldKeyword, updatedKeyword)) {
                    change.modify(Keyword.STATUS_MODERATE, StatusModerate.NEW);

                    if (oldNormalKeyword == null ||
                            isPhraseCoverageSignificantlyChanged(oldNormalKeyword, updatedNormalKeyword)) {
                        change.modify(Keyword.PHRASE_BS_ID, BigInteger.ZERO);
                        change.modify(Keyword.PHRASE_ID_HISTORY, null);
                    }
                }
            });
        }
    }

    /**
     * Получить прогнозы показов для фраз, у которых поменялся текст,
     * и обновить их в поле {@code showsForecast}
     */
    private void getShowsForecast(Map<Integer, AppliedChanges<Keyword>> actualKeywordsToUpdate,
                                  Map<Long, Campaign> campaignByIds) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:getShowsForecast")) {
            List<AppliedChanges<Keyword>> keywordsToUpdateForecast = EntryStream.of(actualKeywordsToUpdate)
                    .filterValues(changes -> changes.changed(Keyword.PHRASE))
                    .values()
                    .collect(Collectors.toList());
            if (keywordsToUpdateForecast.isEmpty()) {
                return;
            }

            // сбрасываем прогнозы у фраз, где поменялся текст, чтобы в случае,
            // если не удастся сходить в ADVQ, у фразы не остался старый прогноз
            keywordsToUpdateForecast.forEach((change) -> change.modify(Keyword.SHOWS_FORECAST, 0L));

            Duration forecastTimeout =
                    operationParams.isAutoPrices() ? AUTO_PRICE_FORECAST_TIMEOUT : DEFAULT_FORECAST_TIMEOUT;
            IdentityHashMap<Keyword, SearchItem> showForecasts =
                    keywordShowsForecastService.getPhrasesShowsSafe(
                            mapList(keywordsToUpdateForecast, AppliedChanges::getModel),
                            forecastTimeout,
                            clientId,
                            campaignByIds
                    );
            keywordsToUpdateForecast.forEach((change) -> {
                SearchItem showsForecast = showForecasts.get(change.getModel());
                if (showsForecast != null) {
                    change.modify(Keyword.SHOWS_FORECAST, showsForecast.getTotalCount());
                }
            });
        }
    }

    /**
     * Получение результата торгов для изменяемых ключевиков.
     * Запрос, сохранение похода в торги.
     */
    private void requestTrafaretAuction(Map<Integer, AppliedChanges<Keyword>> appliedChangesMap,
                                        Map<Long, Campaign> campaignByIds) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:requestTrafaretAuction")) {
            List<Keyword> keywordsForAuction = mapList(appliedChangesMap.values(), AppliedChanges::getModel);

            Set<Long> adGroupIds = FunctionalUtils.listToSet(keywordsForAuction, Keyword::getAdGroupId);
            Map<Long, AdGroupSimple> adGroupSimpleMap = adGroupRepository.getAdGroupSimple(shard, clientId, adGroupIds);

            keywordsForAuction = filterList(keywordsForAuction,
                    k -> !ADGROUP_TYPES_NO_AUCTION.contains(adGroupSimpleMap.get(k.getAdGroupId()).getType()));

            IdentityHashMap<Keyword, KeywordTrafaretData> trafaretDataMap =
                    keywordBsAuctionService.getTrafaretAuctionMapSafe(clientId, keywordsForAuction, campaignByIds);

            trafaretDataByIndex = ImmutableMap.copyOf(
                    EntryStream.of(appliedChangesMap)
                            .mapValues(AppliedChanges::getModel)
                            .mapValues(trafaretDataMap::get)
                            .nonNullValues()
                            .toMap());
        }
    }

    /**
     * Получение мапы с фразами, которым нужно пересчитать ставку, когда
     * включен режим {@code autoPrices}
     */
    private Map<Integer, AppliedChanges<Keyword>> getKeywordsToRecalcPrice(
            Map<Integer, AppliedChanges<Keyword>> actualKeywordsToUpdate) {
        return EntryStream.of(actualKeywordsToUpdate)
                .filterValues(change -> change.changed(Keyword.NORM_PHRASE))
                .toMap();
    }

    /**
     * Сбрасывание ставок и приоритетов автобюджета указанным фразам, если
     * ставки не указаны явно в запросе.
     *
     * @param keywordsToResetPrice мапа с фразами, которым нужно сбросить ставки
     */
    private void resetPrices(Map<Integer, AppliedChanges<Keyword>> keywordsToResetPrice,
                             List<ModelChanges<Keyword>> modelChanges) {
        keywordsToResetPrice.forEach((index, appliedChanges) -> {
            ModelChanges<Keyword> mc = modelChanges.get(index);

            if (!mc.isPropChanged(Keyword.PRICE)) {
                appliedChanges.modify(Keyword.PRICE, null);
            }
            if (!mc.isPropChanged(Keyword.PRICE_CONTEXT)) {
                appliedChanges.modify(Keyword.PRICE_CONTEXT, null);
            }
            if (!mc.isPropChanged(Keyword.AUTOBUDGET_PRIORITY)) {
                appliedChanges.modify(Keyword.AUTOBUDGET_PRIORITY, null);
            }
        });
    }

    /**
     * Попытка выставить фиксированные ставки указанным фразам, если ставки
     * нет, но она нужна в текущей стратегии.
     *
     * @param keywords  мапа с фразами, которым нужно выставить фиксированные ставки
     * @param campaigns кампании, в которые входят фразы {@code keywords}
     */
    private void setFixedAutoPrices(Map<Integer, AppliedChanges<Keyword>> keywords, Map<Long, Campaign> campaigns) {
        checkNotNull(fixedAutoPrices);
        keywords.forEach((index, changes) -> {
            Keyword kw = changes.getModel();

            BigDecimal fixedPrice = null;
            if (fixedAutoPrices.hasGlobalFixedPrice()) {
                fixedPrice = fixedAutoPrices.getGlobalFixedPrice();
            } else if (fixedAutoPrices.hasAdGroupFixedPrice(kw.getAdGroupId())) {
                fixedPrice = fixedAutoPrices.getAdGroupFixedPrice(kw.getAdGroupId());
            }
            if (fixedPrice == null) {
                return;
            }

            DbStrategy campStrategy = campaigns.get(kw.getCampaignId()).getStrategy();
            if (KEYWORD_MISSING_SEARCH_PRICE.test(kw, campStrategy)) {
                changes.modify(Keyword.PRICE, fixedPrice);
            }
            if (KEYWORD_MISSING_CONTEXT_PRICE.test(kw, campStrategy)) {
                changes.modify(Keyword.PRICE_CONTEXT, fixedPrice);
            }
        });
    }

    /**
     * Расчет автоматических ставок для фраз, если ставки нет, но она нужна
     * в текущей стратегии.
     *
     * @param keywords  мапа с фразами, которым нужно расчитать автоматические ставки
     * @param campaigns кампании, в которые входят фразы
     */
    private void calcAutoPrices(Map<Integer, AppliedChanges<Keyword>> keywords, Map<Long, Campaign> campaigns) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:calcAutoPrices")) {
            if (keywordAutoPricesCalculator == null) {
                keywordAutoPricesCalculator = new KeywordAutoPricesCalculator(
                        clientCurrency,
                        cloneKeywords(existingKeywords),
                        keywordNormalizer
                );
            }

            keywords.forEach((index, changes) -> {
                Keyword kw = changes.getModel();
                InternalKeyword ikw = updatedKeywordsMap.get(index);
                Campaign campaign = campaigns.get(kw.getCampaignId());

                if (KEYWORD_MISSING_SEARCH_PRICE.test(kw, campaign.getStrategy())) {
                    KeywordBidBsAuctionData auctionData = null;
                    if (trafaretDataByIndex.containsKey(index)) {
                        auctionData = convertToPositionsAuctionData(trafaretDataByIndex.get(index), clientCurrency);
                    }

                    changes.modify(Keyword.PRICE, keywordAutoPricesCalculator
                            .calcSearchAutoPrice(ikw, campaign.getType(), auctionData)
                            .bigDecimalValue()
                    );
                }

                if (KEYWORD_MISSING_CONTEXT_PRICE.test(kw, campaign.getStrategy())) {
                    changes.modify(Keyword.PRICE_CONTEXT, keywordAutoPricesCalculator
                            .calcContextAutoPrice(ikw)
                            .bigDecimalValue()
                    );
                }
            });
        }
    }

    /**
     * Выставление приоритета по-умолчанию для фраз, у которых его нет, но
     * он нужен в текущей стратегии.
     *
     * @param keywords  фразы, которым нужно пересчитать приоритет автобюджета
     * @param campaigns кампании, в которые входят фразы
     */
    private void setMissingAutobudgetPriorities(Map<Integer, AppliedChanges<Keyword>> keywords,
                                                Map<Long, Campaign> campaigns) {
        keywords.forEach((index, changes) -> {
            Keyword kw = changes.getModel();
            Campaign campaign = campaigns.get(kw.getCampaignId());
            if (KEYWORD_MISSING_AUTOBUDGET_PRIORITY.test(kw, campaign.getStrategy())) {
                changes.modify(Keyword.AUTOBUDGET_PRIORITY, Constants.DEFAULT_AUTOBUDGET_PRIORITY);
            }
        });
    }

    /**
     * Пересчитывает позицию показов на поиске, если это нужно.
     */
    private void calcSearchPlaces(Map<Integer, AppliedChanges<Keyword>> actualKeywordsToUpdate) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:calcSearchPlaces")) {
            IdentityHashMap<Keyword, Place> places = getKeywordPlaces(trafaretDataByIndex.values(), clientCurrency);
            Map<Integer, AppliedChanges<Keyword>> keywordsForComputePlace = EntryStream.of(actualKeywordsToUpdate)
                    .filterValues(keyword -> keyword.changed(Keyword.PHRASE) || keyword.changed(Keyword.PRICE))
                    .toMap();

            keywordsForComputePlace.forEach((index, change) -> {
                Keyword keyword = actualKeywordsToUpdate.get(index).getModel();
                Place place = places.get(keyword);
                place = place != null ? place : Place.ROTATION;

                change.modify(Keyword.PLACE, place);
                change.modify(Keyword.NEED_CHECK_PLACE_MODIFIED, true);
            });
        }
    }

    private void prepareSystemFields(Map<Integer, AppliedChanges<Keyword>> appliedChangesMap) {
        LocalDateTime time = LocalDateTime.now();

        appliedChangesMap.forEach((idx, change) -> {

            if (change.changed(Keyword.PRICE)
                    || change.changed(Keyword.PRICE_CONTEXT)
                    || change.changed(Keyword.AUTOBUDGET_PRIORITY)) {
                change.modify(Keyword.STATUS_BS_SYNCED, StatusBsSynced.NO);
            }

            if (change.changed(Keyword.PHRASE) || change.changed(Keyword.IS_SUSPENDED)
                    || change.changed(Keyword.HREF_PARAM1) || change.changed(Keyword.HREF_PARAM2)
                    || change.changed(Keyword.PRICE) || change.changed(Keyword.PRICE_CONTEXT)
                    || change.changed(Keyword.AUTOBUDGET_PRIORITY)) {
                change.modify(Keyword.MODIFICATION_TIME, time);
            }
        });
    }

    /**
     * Вычисление действий, которые будут произведены
     * над смежными объектами при выполнении операции.
     * <p>
     * Этот метод должен выполняться после {@link #prepareSystemFields},
     * так как пользуется служебными полями Keyword.
     * <p>
     * Все вычисления по возможности должны выполняться на этапе подготовки
     * ({@link #prepare()}, то есть вне создаваемых {@code Runnable}, чтобы
     * не затягивать по времени транзакцию и минимизировать риски падения
     * при выполнении операции.
     */
    private void computeAdditionalTasksForUpdate(Map<Integer, AppliedChanges<Keyword>> appliedChangesMap) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:computeAdditionalTasksForUpdate")) {
            Set<Long> adGroupIdsToBsSynced = new HashSet<>();
            Set<Long> adGroupIdsToModerate = new HashSet<>();
            Set<Long> adGroupIdsToModerateTemplateBanners = new HashSet<>();
            Set<Long> adGroupIdsToResetForecast = new HashSet<>();
            Set<Long> adGroupIdsToCheckConditions = new HashSet<>();
            Set<Long> campaignIdsToResetForecast = new HashSet<>();
            Set<Long> campaignIdsToFreezeAutobudgetAlert = new HashSet<>();
            Set<Long> keywordIdsToResetAggregatedStatuses = new HashSet<>();
            List<Keyword> keywordsToLogPrice = new ArrayList<>();

            for (AppliedChanges<Keyword> change : appliedChangesMap.values()) {
                if (change.changed(Keyword.PHRASE) || change.changed(Keyword.IS_SUSPENDED)
                        || change.changed(Keyword.HREF_PARAM1) || change.changed(Keyword.HREF_PARAM2)) {
                    //тут же и сбрасывается lastChange
                    adGroupIdsToBsSynced.add(change.getModel().getAdGroupId());
                }

                if (change.getNewValue(Keyword.STATUS_MODERATE) == StatusModerate.NEW) {
                    adGroupIdsToModerate.add(change.getModel().getAdGroupId());
                }

                if (change.changed(Keyword.NORM_PHRASE)) {
                    adGroupIdsToModerateTemplateBanners.add(change.getModel().getAdGroupId());
                }

                if (change.changed(Keyword.PHRASE)) {
                    adGroupIdsToResetForecast.add(change.getModel().getAdGroupId());
                }

                if (change.changed(Keyword.IS_SUSPENDED)
                        && Boolean.TRUE.equals(change.getNewValue(Keyword.IS_SUSPENDED))) {
                    adGroupIdsToCheckConditions.add(change.getModel().getAdGroupId());
                }

                if (change.changed(Keyword.PHRASE) || change.changed(Keyword.IS_SUSPENDED) || change.changed(Keyword.PRICE)
                        || change.changed(Keyword.PRICE_CONTEXT) || change.changed(Keyword.AUTOBUDGET_PRIORITY)) {
                    campaignIdsToResetForecast.add(change.getModel().getCampaignId());
                }

                if (change.changed(Keyword.PHRASE) || change.changed(Keyword.IS_SUSPENDED)
                        || change.changed(Keyword.AUTOBUDGET_PRIORITY)) {
                    campaignIdsToFreezeAutobudgetAlert.add(change.getModel().getCampaignId());
                }

                if (change.changed(Keyword.IS_SUSPENDED)) {
                    keywordIdsToResetAggregatedStatuses.add(change.getModel().getId());
                }

                if (change.changed(Keyword.PRICE) || change.changed(Keyword.PRICE_CONTEXT)) {
                    keywordsToLogPrice.add(change.getModel());
                }
            }

            Set<Long> templateBannerIds = bannerRepository.relations
                    .getTemplateBannerIdsByAdGroupIds(shard, clientId, adGroupIdsToModerateTemplateBanners);
            List<KeywordEvent> updateKeywordMailEvents = computeMailEvents(appliedChangesMap.values());
            List<LogPriceData> priceDataList = computeLogUpdatePriceDataList(keywordsToLogPrice);

            transactionalAdditionalUpdateTask = conf -> {
                if (!adGroupIdsToBsSynced.isEmpty()) {
                    adGroupRepository.updateStatusBsSyncedWithLastChange(conf, adGroupIdsToBsSynced, StatusBsSynced.NO);
                }
                if (!adGroupIdsToModerate.isEmpty()) {
                    adGroupRepository.dropStatusModerateExceptDrafts(conf, adGroupIdsToModerate);
                }
                if (!adGroupIdsToCheckConditions.isEmpty()) {
                    Set<Long> nonEmptyGroupIds =
                            adGroupRepository.getAdGroupIdsWithConditions(conf.dsl(), adGroupIdsToCheckConditions);
                    Set<Long> emptyGroupIds = Sets.difference(adGroupIdsToCheckConditions, nonEmptyGroupIds);
                    adGroupRepository.updateModerationStatusesAfterConditionsAreGone(conf, emptyGroupIds);
                }

                if (!templateBannerIds.isEmpty()) {
                    bannerRepository.common.dropTemplateBannersStatusesOnKeywordsChange(conf, templateBannerIds);
                    moderationService.deleteBannerPostAndAuto(conf, templateBannerIds);
                    bannerRepository.moderation.deleteMinusGeo(conf.dsl(), templateBannerIds);
                }

                if (!adGroupIdsToResetForecast.isEmpty()) {
                    adGroupRepository
                            .updateStatusShowsForecastExceptDrafts(conf, adGroupIdsToResetForecast,
                                    StatusShowsForecast.NEW);
                }

                if (!keywordIdsToResetAggregatedStatuses.isEmpty()) {
                    aggregatedStatusesRepository.markKeywordStatusesAsObsolete(conf.dsl(), null,
                            keywordIdsToResetAggregatedStatuses);
                }
            };

            additionalUpdateTask = () -> {
                if (!campaignIdsToResetForecast.isEmpty()) {
                    campaignRepository.setAutobudgetForecastDate(shard, campaignIdsToResetForecast, null);
                    autoPriceCampQueueService.clearAutoPriceQueue(campaignIdsToResetForecast);
                }
                if (!updateKeywordMailEvents.isEmpty()) {
                    mailNotificationEventService.queueEvents(operatorUid, clientId, updateKeywordMailEvents);
                }
                if (!campaignIdsToFreezeAutobudgetAlert.isEmpty()) {
                    autobudgetAlertService.freezeAlertsOnKeywordsChange(clientId, campaignIdsToFreezeAutobudgetAlert);
                }
                if (!keywordsToLogPrice.isEmpty()) {
                    logPriceService.logPrice(priceDataList, operatorUid);
                }
            };
        }
    }

    private List<KeywordEvent> computeMailEvents(Collection<AppliedChanges<Keyword>> changes) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:computeMailEvents")) {
            List<AppliedChanges<Keyword>> affectedChanges = filterList(changes,
                    change -> change.changed(Keyword.PHRASE) || change.changed(Keyword.PRICE)
                            || change.changed(Keyword.PRICE_CONTEXT));
            Set<Long> affectedAdGroupIds = StreamEx.of(affectedChanges)
                    .map(AppliedChanges::getModel)
                    .map(Keyword::getAdGroupId)
                    .toSet();
            Map<Long, AdGroupName> adGroupIdToAdGroup = getAdGroupIdToAdGroupNameWithCheck(affectedAdGroupIds);

            List<KeywordEvent> addKeywordEvents = new ArrayList<>();
            affectedChanges.forEach(change -> {
                Long adGroupId = change.getModel().getAdGroupId();
                long campaignId = adGroupIdToAdGroup.get(adGroupId).getCampaignId();
                String adGroupName = adGroupIdToAdGroup.get(adGroupId).getName();
                if (change.changed(Keyword.PHRASE)
                        && change.getOldValue(Keyword.PHRASE) != null
                        && change.getNewValue(Keyword.PHRASE) != null) {
                    @SuppressWarnings("ConstantConditions")
                    KeywordEvent event =
                            changedKeywordPhraseEvent(operatorUid, clientUid, campaignId, adGroupId, adGroupName,
                                    change.getOldValue(Keyword.PHRASE), change.getNewValue(Keyword.PHRASE));
                    addKeywordEvents.add(event);
                }
                if (change.changed(Keyword.PRICE)
                        && change.getOldValue(Keyword.PRICE) != null
                        && change.getNewValue(Keyword.PRICE) != null) {
                    @SuppressWarnings("ConstantConditions")
                    KeywordEvent event =
                            changedSearchPriceEvent(operatorUid, clientUid, campaignId, adGroupId, adGroupName,
                                    change.getOldValue(Keyword.PRICE), change.getNewValue(Keyword.PRICE));
                    addKeywordEvents.add(event);
                }
                if (change.changed(Keyword.PRICE_CONTEXT)
                        && change.getOldValue(Keyword.PRICE_CONTEXT) != null
                        && change.getNewValue(Keyword.PRICE_CONTEXT) != null) {
                    @SuppressWarnings("ConstantConditions")
                    KeywordEvent event =
                            changedContextPriceEvent(operatorUid, clientUid, campaignId, adGroupId, adGroupName,
                                    change.getOldValue(Keyword.PRICE_CONTEXT),
                                    change.getNewValue(Keyword.PRICE_CONTEXT));
                    addKeywordEvents.add(event);
                }
            });

            return addKeywordEvents;
        }
    }

    private Map<Long, AdGroupName> getAdGroupIdToAdGroupNameWithCheck(Set<Long> adGroupIds) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:getAdGroupIdToAdGroupNameWithCheck")) {
            Map<Long, AdGroupName> adGroupIdToAdGroup = adGroupRepository.getAdGroupNames(shard, clientId, adGroupIds);
            Set<Long> notFoundAdGroupIds = Sets.difference(adGroupIds, adGroupIdToAdGroup.keySet());
            checkState(notFoundAdGroupIds.isEmpty(), "can't get adGroups for adGroup ids: " + notFoundAdGroupIds);
            return adGroupIdToAdGroup;
        }
    }

    private List<LogPriceData> computeLogUpdatePriceDataList(Collection<Keyword> updateKeywords) {
        Function<Keyword, LogPriceData> keywordToLogFn = keyword -> {
            Double priceSearch = keyword.getPrice() != null ? keyword.getPrice().doubleValue() : 0;
            Double priceContext = keyword.getPriceContext() != null ? keyword.getPriceContext().doubleValue() : 0;
            return new LogPriceData(
                    keyword.getCampaignId(),
                    keyword.getAdGroupId(),
                    keyword.getId(),
                    priceContext,
                    priceSearch,
                    clientCurrency.getCode(),
                    LogPriceData.OperationType.UPDATE_1);
        };
        return mapList(updateKeywords, keywordToLogFn);
    }

    @Override
    protected Map<Integer, UpdatedKeywordInfo> execute(ExecutionStep<Keyword> executionStep) {
        try (TraceProfile ignore = Trace.current().profile("keywords.update:execute")) {
            Map<Integer, AppliedChanges<Keyword>> applicableAppliedChanges =
                    executionStep.getAppliedChangesForExecutionWithIndex();
            Map<Integer, AppliedChanges<Keyword>> actualKeywordsToUpdate =
                    extractActualKeywordsToUpdate(applicableAppliedChanges);
            checkExecutionEntryPointConsistency(applicableAppliedChanges);

            List<CampaignIdAndKeywordIdPair> campaignIdToKeywordIdList =
                    mapList(keywordsToDelete, keyword -> new CampaignIdAndKeywordIdPair(keyword.getCampaignId(),
                            keyword.getId()));
            TransactionalRunnable updateFn = conf -> {
                Set<Long> affectedAdGroupIds =
                        listToSet(actualKeywordsToUpdate.values(), kw -> kw.getModel().getAdGroupId());
                List<AppliedChanges<Keyword>> allKeywordsToUpdate = new ArrayList<>();
                allKeywordsToUpdate.addAll(actualKeywordsToUpdate.values());
                allKeywordsToUpdate.addAll(otherExistingKeywordsToUpdate);

                adGroupRepository.getLockOnAdGroups(conf, affectedAdGroupIds);

                List<Keyword> deletedKeywords = keywordRepository
                        .updateAndDeleteWithDeduplication(conf, allKeywordsToUpdate, campaignIdToKeywordIdList);

                List<Keyword> allDeletedKeywords = new ArrayList<>();
                allDeletedKeywords.addAll(keywordsToDelete);
                allDeletedKeywords.addAll(deletedKeywords);
                computeAdditionalTasksForDelete(allDeletedKeywords);

                transactionalAdditionalUpdateTask.run(conf);
                transactionalAdditionalDeleteTask.run(conf);
            };

            dslContextProvider.ppcTransaction(shard, updateFn);
            //дополнительные таски должны идти после updateFn, т.к. там создается additionalDeleteTask
            additionalUpdateTask.run();
            additionalDeleteTask.run();

            finishResultMapFilling(executionStep.getValidationResult(), applicableAppliedChanges,
                    actualKeywordsToUpdate);
            checkExecutionExitPointConsistency(applicableAppliedChanges, resultMap);
            return resultMap;
        }
    }

    private Map<Integer, AppliedChanges<Keyword>> extractActualKeywordsToUpdate(
            Map<Integer, AppliedChanges<Keyword>> applicableAppliedChanges) {
        return EntryStream.of(applicableAppliedChanges)
                .filterKeys(updatedKeywordsMap::containsKey)
                .toMap();
    }

    /**
     * Индексы валидных элементов должны быть объединением множеств индексов из следующих мап:
     * {@link #updatedKeywordsMap}, {@link #duplicatesWithOtherExisting}, {@link #newDuplicatesLinkingMap},
     * при этом эти множества не должны пересекаться.
     * <p>
     * Множество индексов результатов в мапе {@link #resultMap} должно совпадать с множеством
     * индексов валидных элементов.
     *
     * @param validKeywordsMap мапа валидных ключевых фраз.
     */
    private void checkExecutionEntryPointConsistency(Map<Integer, AppliedChanges<Keyword>> validKeywordsMap) {
        Set<Integer> validIndexes = validKeywordsMap.keySet();
        Set<Integer> keywordsToUpdateIndexes = updatedKeywordsMap.keySet();
        Set<Integer> duplicatedWithOtherExistingIndexes = duplicatesWithOtherExisting.keySet();
        Set<Integer> newDuplicatesIndexes = newDuplicatesLinkingMap.keySet();
        checkState(keywordsToUpdateIndexes.size() +
                        duplicatedWithOtherExistingIndexes.size() +
                        newDuplicatesIndexes.size() == validIndexes.size(),
                "can't execute operation: inconsistent state; "
                        + "this can happen for example if deduplication is not symmetric");

        Set<Integer> summaryIndexesSet = new HashSet<>();
        summaryIndexesSet.addAll(keywordsToUpdateIndexes);
        summaryIndexesSet.addAll(duplicatedWithOtherExistingIndexes);
        summaryIndexesSet.addAll(newDuplicatesIndexes);
        checkState(summaryIndexesSet.equals(validIndexes), "can't execute operation: inconsistent state");

        Set<Integer> externalResultIndexes = resultMap.keySet();
        checkState(externalResultIndexes.equals(validIndexes), "can't execute operation: inconsistent state");
    }

    /**
     * Вычисление действий, которые будут произведены над удаленными фразами.
     * <p>
     * Соответствует дополнительным операциям {@link KeywordDeleteOperation}, за исключением посылки писем
     * <p>
     * Расчет дополнительных операций для удаления выполняется в execute,
     * т.к. полное число удаленных ключевиков можно определить только после вызова метода {@code
     * updateAndDeleteWithDeduplication}
     */
    private void computeAdditionalTasksForDelete(List<Keyword> keywordsToDelete) {
        List<LogPriceData> priceDataList = computeLogDeletePriceDataList(keywordsToDelete);

        Set<Long> adGroupIdsToBsSynced = listToSet(keywordsToDelete, Keyword::getAdGroupId);
        Set<Long> campaignIdsToResetForecast = listToSet(keywordsToDelete, Keyword::getCampaignId);

        transactionalAdditionalDeleteTask = conf -> {
            adGroupRepository.updateAfterKeywordsDeleted(conf, adGroupIdsToBsSynced);
            campaignRepository.setAutobudgetForecastDate(conf, campaignIdsToResetForecast, null);
        };

        additionalDeleteTask = () -> {
            if (!keywordsToDelete.isEmpty()) {
                logPriceService.logPrice(priceDataList, operatorUid);
            }
        };
    }

    private List<LogPriceData> computeLogDeletePriceDataList(Collection<Keyword> deleteKeywords) {
        Function<Keyword, LogPriceData> keywordToLogFn = keyword -> new LogPriceData(
                keyword.getCampaignId(),
                keyword.getAdGroupId(),
                keyword.getId(),
                null,
                null,
                null,
                LogPriceData.OperationType.DELETE_1
        );
        return mapList(deleteKeywords, keywordToLogFn);
    }


    private void finishResultMapFilling(ValidationResult<List<Keyword>, Defect> validationResult,
                                        Map<Integer, AppliedChanges<Keyword>> applicableAppliedChanges,
                                        Map<Integer, AppliedChanges<Keyword>> actualKeywordsToUpdate) {
        // дозаполняем результаты для элементов, которые дублировались с другими существующими
        duplicatesWithOtherExisting.forEach((deletedItemIndex, existingItemIndex) -> {
            UpdatedKeywordInfo keywordInfo = resultMap.get(deletedItemIndex);
            InternalKeyword existingKeyword = otherExistingKeywordsMap.get(existingItemIndex);
            keywordInfo
                    .withId(existingKeyword.getId())
                    .withDeleted(true)
                    .withIsSuspended(null);
            addWarning(validationResult, deletedItemIndex, applicableAppliedChanges.get(deletedItemIndex),
                    duplicatedWithExisting());
        });

        // дозаполняем результаты для элементов, которые дублировались с обновляемыми
        newDuplicatesLinkingMap.forEach((deletedItemIndex, updatedItemIndex) -> {
            UpdatedKeywordInfo keywordInfo = resultMap.get(deletedItemIndex);
            Long updatedId = actualKeywordsToUpdate.get(updatedItemIndex).getModel().getId();
            keywordInfo.withId(updatedId)
                    .withDeleted(true)
                    .withIsSuspended(null);

            addWarning(validationResult, deletedItemIndex, applicableAppliedChanges.get(deletedItemIndex),
                    duplicatedWithUpdated());
        });
    }

    private void checkExecutionExitPointConsistency(Map<Integer, AppliedChanges<Keyword>> applicableAppliedChanges,
                                                    Map<Integer, UpdatedKeywordInfo> updatedKeywordInfoMap) {
        updatedKeywordInfoMap.forEach((index, updatedKeywordInfo) -> {
            checkState(updatedKeywordInfo.getId() != null, "id is null in result[%s]", index);
            checkState(updatedKeywordInfo.getResultPhrase() != null, "result phrase is null in result[%s]", index);
            if (updatedKeywordInfo.isDeleted()) {
                List<Keyword> list = validationResult.getValue();
                ValidationResult<Keyword, Defect> keywordVr =
                        validationResult.getOrCreateSubValidationResult(index(index), list.get(index));
                checkState(keywordVr.hasAnyWarnings(), "(result[%s].isDeleted == true) and it has no warnings", index);

                Long oldId = applicableAppliedChanges.get(index).getModel().getId();
                checkState(!Objects.equals(updatedKeywordInfo.getId(), oldId),
                        "deleted keyword id is in update response (result[%s].id) ", index);
            }
        });
    }

    private void addWarning(ValidationResult<List<Keyword>, Defect> validationResult, int index,
                            AppliedChanges<Keyword> changes, Defect warning) {
        validationResult.getOrCreateSubValidationResult(index(index), changes.getModel())
                .addWarning(warning);
    }

    private List<Keyword> getKeywordsByAdGroupIds(int shard, ClientId clientId, Set<Long> adGroupIds) {
        return StreamEx.of(keywordRepository.getKeywordsByAdGroupIds(shard, clientId, adGroupIds).values())
                .flatMap(Collection::stream)
                .toList();
    }
}
