package ru.yandex.direct.core.entity.keyword.service.validation;

import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableSet;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.keyword.container.InternalKeyword;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordDefects.adGroupNotFound;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

/**
 * Класс базовых проверок для ключевых фраз
 */
public class ClientKeywordCommonValidator {

    static final Set<AdGroupType> ACCEPTABLE_AD_GROUP_TYPES_FOR_ADD_KEYWORD =
            ImmutableSet.of(AdGroupType.BASE, AdGroupType.MOBILE_CONTENT, AdGroupType.MCBANNER, AdGroupType.CPM_BANNER,
                    AdGroupType.CONTENT_PROMOTION_VIDEO, AdGroupType.CONTENT_PROMOTION);

    private ClientKeywordCommonValidator() {
    }

    /**
     * Проверки на наличие и неархивность групп и кампаний, права записи,
     * наличие обязательных полей при автобюджетной и ручной стратегиях
     *
     * @param mustHavePrices   во фразах должны быть ставки и приоритет автобюджета, если того требует стратегия
     * @param weakenValidation ослабленная валидация, пропускает проверки, которые могут не проходить для существующих
     *                         фраз
     */
    public static Validator<Keyword, Defect> build(StopWordService stopWordService,
                                                   KeywordWithLemmasFactory keywordFactory, InternalKeyword internalKeyword,
                                                   Map<Long, Long> adGroupIdToCampaignId, Map<Long, Campaign> writableCampaignsById,
                                                   Map<Long, AdGroupType> adGroupTypesById, Map<Long, CriterionType> cpmAdGroupsWithCriterionType,
                                                   Validator<Long, Defect> adGroupAccessValidator,
                                                   boolean mustHavePrices,
                                                   boolean weakenValidation) {
        return keyword -> validateCommon(keyword, stopWordService, keywordFactory,
                internalKeyword, adGroupIdToCampaignId, writableCampaignsById,
                adGroupTypesById, cpmAdGroupsWithCriterionType, adGroupAccessValidator,
                mustHavePrices, weakenValidation);
    }

    private static ValidationResult<Keyword, Defect> validateCommon(Keyword keyword,
                                                                    StopWordService stopWordService, KeywordWithLemmasFactory keywordFactory,
                                                                    InternalKeyword internalKeyword, Map<Long, Long> adGroupIdToCampaignId,
                                                                    Map<Long, Campaign> writableCampaignsById, Map<Long, AdGroupType> adGroupTypesById,
                                                                    Map<Long, CriterionType> cpmAdGroupsWithCriterionType, Validator<Long, Defect> adGroupAccessValidator,
                                                                    boolean mustHavePrices,
                                                                    boolean weakenValidation) {
        ModelItemValidationBuilder<Keyword> vb = ModelItemValidationBuilder.of(keyword);
        vb.item(Keyword.AD_GROUP_ID)
                .check(notNull(), objectNotFound())
                .check(validId())
                .checkBy(adGroupAccessValidator, When.isValid())
                .check(inSet(adGroupIdToCampaignId.keySet()), adGroupNotFound(), When.isValid())
                .check(adGroupTypeIsAcceptable(adGroupTypesById, cpmAdGroupsWithCriterionType), When.isValid());

        Long campaignId = adGroupIdToCampaignId.get(keyword.getAdGroupId());
        AdGroupType adGroupType = adGroupTypesById.get(keyword.getAdGroupId());
        vb.checkBy(KeywordWithoutInterconnectionsValidator
                .build(stopWordService, keywordFactory, internalKeyword,
                        writableCampaignsById.get(campaignId), adGroupType,
                        mustHavePrices, weakenValidation));
        return vb.getResult();
    }

    private static Constraint<Long, Defect> adGroupTypeIsAcceptable(Map<Long, AdGroupType> adGroupTypeById,
                                                                    Map<Long, CriterionType> cpmAdGroupsWithCriterionType) {
        return adGroupId -> {
            boolean isCpmAdGroupWithKeywordsCriterion = ImmutableSet
                    .of(AdGroupType.CPM_BANNER)
                    .contains(adGroupTypeById.get(adGroupId));
            if (!isCpmAdGroupWithKeywordsCriterion) {
                return ACCEPTABLE_AD_GROUP_TYPES_FOR_ADD_KEYWORD.contains(adGroupTypeById.get(adGroupId))
                        ? null
                        : KeywordDefects.notAcceptableAdGroupType();
            }
            return cpmAdGroupsWithCriterionType.get(adGroupId) == CriterionType.KEYWORD
                    ? null
                    : KeywordDefects.notAcceptableAdGroupType();
        };
    }

    public static boolean adGroupCanHaveKeywords(AdGroupType adGroupType) {
        return ACCEPTABLE_AD_GROUP_TYPES_FOR_ADD_KEYWORD.contains(adGroupType);
    }
}
