package ru.yandex.direct.core.entity.keyword.service.validation;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.AccessDefectPresets;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessDefects;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessChecker;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.keyword.container.InternalKeyword;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignTypeNotSupported;
import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordDefects.adGroupNotFound;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRights;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

/**
 * Общий сервис валидации ключевых фраз
 * Здесь собраны все проверки моделей, не зависящие от контекста операции(add/update)
 */
@Service
@ParametersAreNonnullByDefault
public class ClientKeywordValidationService {
    private static final CampaignAccessDefects ACCESS_DEFECTS =
            AccessDefectPresets.DEFAULT_DEFECTS.toBuilder()
                    .withTypeNotAllowable(adGroupNotFound())
                    .withNotVisible(adGroupNotFound())
                    .withTypeNotSupported(campaignTypeNotSupported())
                    .withNoRights(noRights())
                    .build();

    private final ShardHelper shardHelper;
    private final RbacService rbacService;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final KeywordWithLemmasFactory keywordFactory;
    private final StopWordService stopWordService;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;

    @Autowired
    public ClientKeywordValidationService(
            ShardHelper shardHelper,
            RbacService rbacService,
            AdGroupRepository adGroupRepository,
            CampaignRepository campaignRepository,
            KeywordWithLemmasFactory keywordFactory,
            StopWordService stopWordService,
            CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory) {
        this.shardHelper = shardHelper;
        this.rbacService = rbacService;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.keywordFactory = keywordFactory;
        this.stopWordService = stopWordService;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
    }

    /**
     * Основной метод валидации ключевых фраз
     *
     * @param vr                 результат предварительной валидации ключевых фраз
     * @param internalKeywordMap Набор нормализованных ключевых фраз, сопоставляются с исходными фразами по индексу
     * @param mustHavePrices     во фразах должны быть ставки и приоритет автобюджета, если того требует стратегия
     * @param weakenValidation   ослабленная валидация, пропускает проверки, которые могут не проходить для существующих
     *                           фраз
     * @return {@link ValidationResult} результат валидации
     */
    public ValidationResult<List<Keyword>, Defect> validate(
            ValidationResult<List<Keyword>, Defect> vr, Map<Integer, InternalKeyword> internalKeywordMap,
            boolean mustHavePrices,
            boolean weakenValidation,
            long operatorUid, ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<Keyword> validKeywords = getValidItems(vr);

        Map<Keyword, Long> adGroupIdsByKeywords = StreamEx.of(validKeywords).distinct()
                .filter(t -> t.getAdGroupId() != null)
                .toMap(Keyword::getAdGroupId);
        Collection<Long> adGroupIds = StreamEx.of(adGroupIdsByKeywords.values()).toList();
        CampaignSubObjectAccessChecker checker =
                campaignSubObjectAccessCheckerFactory.newAdGroupChecker(operatorUid, clientId, adGroupIds);
        CampaignSubObjectAccessValidator adGroupAccessValidator = checker.createValidator(
                CampaignAccessType.READ_WRITE, ACCESS_DEFECTS);

        Map<Long, AdGroupSimple> adGroupsById =
                adGroupRepository.getAdGroupSimple(shard, clientId, adGroupIds);
        Map<Long, AdGroupType> adGroupTypesById =
                EntryStream.of(adGroupsById).mapValues(AdGroupSimple::getType).toMap();
        List<Long> cpmBannerAdGroups =
                mapList(filterList(adGroupsById.values(),
                        adGroup -> adGroup.getType() == AdGroupType.CPM_BANNER),
                        AdGroupSimple::getId);
        Map<Long, CriterionType> cpmAdGroupsWithCriterionType =
                adGroupRepository.getCriterionTypeByAdGroupIds(shard, cpmBannerAdGroups);
        Map<Long, Long> adGroupIdToCampaignId =
                EntryStream.of(adGroupsById).mapValues(AdGroupSimple::getCampaignId).toMap();
        Set<Long> existingCampaignIds = new HashSet<>(adGroupIdToCampaignId.values());
        Set<Long> writableCampaigns = rbacService.getWritableCampaigns(operatorUid, existingCampaignIds);
        Map<Long, Campaign> writableCampaignsById =
                listToMap(campaignRepository.getCampaigns(shard, writableCampaigns), Campaign::getId);

        List<Keyword> keywords = vr.getValue();
        for (int i = 0; i < keywords.size(); i++) {
            ValidationResult<Keyword, Defect> subResult =
                    vr.getOrCreateSubValidationResult(index(i), keywords.get(i));
            if (subResult.hasAnyErrors()) {
                continue;
            }

            ModelItemValidationBuilder<Keyword> itmVr = new ModelItemValidationBuilder<>(subResult);
            InternalKeyword internalKeyword = internalKeywordMap.get(i);

            itmVr.checkBy(ClientKeywordCommonValidator
                    .build(stopWordService, keywordFactory, internalKeyword, adGroupIdToCampaignId,
                            writableCampaignsById, adGroupTypesById, cpmAdGroupsWithCriterionType,
                            adGroupAccessValidator, mustHavePrices, weakenValidation));
        }
        return vr;
    }
}
