package ru.yandex.direct.core.entity.keyword.service.validation;

import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.service.accesschecker.AccessDefectPresets;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessDefects;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.keyword.container.KeywordDeleteInfo;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignTypeNotSupported;
import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordConstraints.keywordExists;
import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordConstraints.nonArchivedCampaign;
import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordConstraints.operatorHasRightsToChange;
import static ru.yandex.direct.core.validation.BaseDeleteValidation.idsToDeleteValidate;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Service
public class DeleteKeywordValidationService {
    private static final CampaignAccessDefects ACCESS_DEFECTS =
            AccessDefectPresets.DEFAULT_DEFECTS.toBuilder()
                    .withTypeNotSupported(campaignTypeNotSupported())
                    .build();

    private final RbacService rbacService;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;

    @Autowired
    public DeleteKeywordValidationService(RbacService rbacService,
                                          CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory) {
        this.rbacService = rbacService;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
    }

    /**
     * Валидирует ключевые фразы перед удалением
     *
     * @param ids               список id ключевых фраз
     * @param keywordDeleteInfo маппинг keywordId -> информация об удаляемой ключевой фразе
     * @return результат валидации
     */
    public ValidationResult<List<Long>, Defect> validate(List<Long> ids,
                                                         Map<Long, KeywordDeleteInfo> keywordDeleteInfo, long operatorUid, ClientId clientId) {

        Constraint<Long, Defect> accessConstraint = campaignSubObjectAccessCheckerFactory
                .newBidChecker(operatorUid, clientId, ids)
                .createValidator(CampaignAccessType.READ_WRITE, ACCESS_DEFECTS)
                .getAccessConstraint();

        List<Long> campaignIds = mapList(keywordDeleteInfo.values(), KeywordDeleteInfo::getCampaignId);
        Set<Long> writableCampaignIds = rbacService.getWritableCampaigns(operatorUid, campaignIds);

        Map<Long, Long> campaignIdByKeywordId = EntryStream.of(keywordDeleteInfo)
                .mapValues(KeywordDeleteInfo::getCampaignId)
                .toMap();

        ValidationResult<List<Long>, Defect> baseIdsDeleteValidation = idsToDeleteValidate(ids);

        return new ListValidationBuilder<>(baseIdsDeleteValidation)
                .checkEach(notNull())
                .checkEach(keywordExists(keywordDeleteInfo.keySet()), When.isValid())
                .checkEach(operatorHasRightsToChange(campaignIdByKeywordId, writableCampaignIds), When.isValid())
                .checkEach(nonArchivedCampaign(keywordDeleteInfo), When.isValid())
                .checkEach(accessConstraint, When.isValid())
                .getResult();
    }
}
