package ru.yandex.direct.core.entity.keyword.service.validation;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.bids.validation.PriceValidator;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.keyword.container.InternalKeyword;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseValidator;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.autoBudgetPriorityIsAccepted;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.autoBudgetPriorityIsNotNullForAutoStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.contextPriceNotNullForManualStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.priceContextIsAcceptedForStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.priceSearchIsAcceptedForStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.searchPriceNotNullForManualStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.strategyIsSet;
import static ru.yandex.direct.core.validation.constraints.Constraints.allowedBannerLetters;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;

public class KeywordWithoutInterconnectionsValidator {
    static final int MAX_PARAM_LENGTH = 255;

    private KeywordWithoutInterconnectionsValidator() {
    }

    /**
     * @param mustHavePrices во фразах должны быть ставки и приоритет автобюджета, если того требует стратегия
     */
    public static Validator<Keyword, Defect> build(StopWordService stopWordService,
                                                   KeywordWithLemmasFactory keywordFactory,
                                                   InternalKeyword internalKeyword,
                                                   @Nullable Campaign campaign, AdGroupType adGroupType,
                                                   boolean mustHavePrices) {
        return build(stopWordService, keywordFactory, internalKeyword, campaign, adGroupType, mustHavePrices, false);
    }

    /**
     * @param mustHavePrices во фразах должны быть ставки и приоритет автобюджета, если того требует стратегия
     * @param weakenValidation ослабленная валидация, пропускает проверки, которые могут не проходить для существующих
     *                         фраз
     */
    public static Validator<Keyword, Defect> build(StopWordService stopWordService,
                                                   KeywordWithLemmasFactory keywordFactory,
                                                   InternalKeyword internalKeyword,
                                                   @Nullable Campaign campaign, AdGroupType adGroupType,
                                                   boolean mustHavePrices,
                                                   boolean weakenValidation) {
        return keyword -> validateWithoutInterconnections(keyword, stopWordService,
                keywordFactory, internalKeyword, campaign, adGroupType, mustHavePrices, weakenValidation);
    }

    /**
     * @param mustHavePrices   во фразах должны быть ставки и приоритет автобюджета, если того требует стратегия
     * @param weakenValidation ослабленная валидация, пропускает проверки, которые могут не проходить для существующих
     *                         фраз
     */
    private static ValidationResult<Keyword, Defect> validateWithoutInterconnections(
            Keyword keyword,
            StopWordService stopWordService,
            KeywordWithLemmasFactory keywordFactory,
            InternalKeyword internalKeyword,
            @Nullable Campaign campaign,
            AdGroupType adGroupType,
            boolean mustHavePrices,
            boolean weakenValidation) {
        ModelItemValidationBuilder<Keyword> vb = ModelItemValidationBuilder.of(keyword);
        if (campaign != null) {
            DbStrategy strategy = campaign.getStrategy();
            vb.check(strategyIsSet(strategy));

            checkState(adGroupType != null, "adgroup type cannot be null");

            Currency currency = campaign.getCurrency().getCurrency();
            vb.item(Keyword.PRICE)
                    .checkBy(new PriceValidator(currency, adGroupType))
                    .check(searchPriceNotNullForManualStrategy(strategy), When.isTrue(strategy != null && mustHavePrices))
                    .weakCheck(priceSearchIsAcceptedForStrategy(strategy), When.isTrue(strategy != null));
            vb.item(Keyword.PRICE_CONTEXT)
                    .checkBy(new PriceValidator(currency, adGroupType))
                    .check(contextPriceNotNullForManualStrategy(strategy), When.isTrue(strategy != null && mustHavePrices))
                    .weakCheck(priceContextIsAcceptedForStrategy(strategy), When.isTrue(strategy != null));
            vb.item(Keyword.AUTOBUDGET_PRIORITY)
                    .check(autoBudgetPriorityIsNotNullForAutoStrategy(strategy), When.isTrue(strategy != null && mustHavePrices))
                    .check(autoBudgetPriorityIsAccepted());
        }
        vb.item(Keyword.PHRASE)
                .checkBy(new PhraseValidator(stopWordService, keywordFactory,
                        ifNotNull(internalKeyword, InternalKeyword::getParsedKeyword),
                        weakenValidation), When.isValid());
        vb.item(Keyword.HREF_PARAM1)
                .check(maxStringLength(MAX_PARAM_LENGTH))
                .check(allowedBannerLetters(), When.isValid());
        vb.item(Keyword.HREF_PARAM2)
                .check(maxStringLength(MAX_PARAM_LENGTH))
                .check(allowedBannerLetters(), When.isValid());
        return vb.getResult();
    }
}
