package ru.yandex.direct.core.entity.keyword.service.validation;

import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.keyword.container.AdGroupInfoForKeywordAdd;
import ru.yandex.direct.core.entity.keyword.container.InternalKeyword;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.hasNoAutotargetingPrefix;
import static ru.yandex.direct.core.entity.keyword.service.validation.ClientKeywordCommonValidator.ACCEPTABLE_AD_GROUP_TYPES_FOR_ADD_KEYWORD;
import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordDefects.maxKeywordsPerAdGroupExceeded;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseSyntaxValidator.keywordSyntaxValidator;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Service
public class KeywordsAddValidationService {

    private final ClientKeywordValidationService clientKeywordValidationService;
    private final KeywordWithLemmasFactory keywordFactory;
    private final StopWordService stopWordService;
    private final FeatureService featureService;

    public KeywordsAddValidationService(ClientKeywordValidationService clientKeywordValidationService,
                                        KeywordWithLemmasFactory keywordFactory,
                                        StopWordService stopWordService,
                                        FeatureService featureService) {
        this.clientKeywordValidationService = clientKeywordValidationService;
        this.keywordFactory = keywordFactory;
        this.stopWordService = stopWordService;
        this.featureService = featureService;
    }

    public ValidationResult<List<Keyword>, Defect> preValidate(List<Keyword> keywords, ClientId clientId) {
        boolean autotargetingKeywordPrefixAllowed = featureService.isEnabledForClientId(clientId,
                FeatureName.AUTOTARGETING_KEYWORD_PREFIX_ALLOWED);
        return preValidate(keywords, autotargetingKeywordPrefixAllowed);
    }

    public ValidationResult<List<Keyword>, Defect> preValidate(List<Keyword> keywords,
                                                               boolean autotargetingKeywordPrefixAllowed) {
        ListValidationBuilder<Keyword, Defect> lvb = ListValidationBuilder.of(keywords);
        lvb.checkEachBy(keyword -> preValidateKeywordPhrase(keyword, autotargetingKeywordPrefixAllowed));
        return lvb.getResult();
    }

    private ValidationResult<Keyword, Defect> preValidateKeywordPhrase(Keyword keyword,
                                                                       boolean autotargetingKeywordPrefixAllowed) {
        ModelItemValidationBuilder<Keyword> vb = ModelItemValidationBuilder.of(keyword);

        vb.item(Keyword.IS_AUTOTARGETING)
                .check(correctAutotargetingPrefix(autotargetingKeywordPrefixAllowed));
        vb.item(Keyword.PHRASE)
                .check(notNull())
                .checkBy(keywordSyntaxValidator(), When.notNull());

        return vb.getResult();
    }

    /**
     * Если добавляется фраза с префиксом "---autotargeting", то у клиента должна быть включена фича
     */
    private Constraint<Boolean, Defect> correctAutotargetingPrefix(boolean autotargetingKeywordPrefixAllowed) {
        return Constraint.fromPredicate(isAutotargeting -> hasNoAutotargetingPrefix(isAutotargeting) ||
                        autotargetingKeywordPrefixAllowed,
                KeywordDefects.autotargetingPrefixIsNotAllowed());
    }

    /**
     * @param mustHavePrices   во фразах должны быть ставки и приоритет автобюджета, если того требует стратегия
     * @param weakenValidation ослабленная валидация, пропускает проверки, которые могут не проходить для существующих
     *                         фраз
     */
    public void validate(ValidationResult<List<Keyword>, Defect> preValidationResult,
                         Map<Integer, InternalKeyword> preValidKeywordsMap, long operatorUid, ClientId clientId,
                         boolean mustHavePrices, boolean weakenValidation) {
        clientKeywordValidationService.validate(preValidationResult, preValidKeywordsMap, mustHavePrices,
                weakenValidation, operatorUid, clientId);
    }

    /**
     * @param mustHavePrices во фразах должны быть ставки и приоритет автобюджета, если того требует стратегия
     */
    public void validateWithNonexistentAdGroups(ValidationResult<List<Keyword>, Defect> preValidationResult,
                                                Map<Integer, AdGroupInfoForKeywordAdd> adGroupInfo,
                                                Map<Long, Campaign> campaigns,
                                                Map<Integer, InternalKeyword> internalKeywordsMap,
                                                boolean mustHavePrices) {
        ListValidationBuilder<Keyword, Defect> lvb = new ListValidationBuilder<>(preValidationResult);
        lvb.checkEachBy((index, keyword) -> {
            AdGroupType adGroupType = adGroupInfo.get(index).getAdGroupType();
            Campaign campaign = campaigns.get(adGroupInfo.get(index).getCampaignId());
            InternalKeyword internalKeyword = internalKeywordsMap.get(index);

            ModelItemValidationBuilder<Keyword> vb = ModelItemValidationBuilder.of(keyword);
            vb.checkBy(KeywordWithoutInterconnectionsValidator.build(stopWordService, keywordFactory,
                    internalKeyword, campaign, adGroupType, mustHavePrices));
            vb.item(Keyword.AD_GROUP_ID)
                    .check(adGroupIsAcceptableToAddKeywords(adGroupType));
            return vb.getResult();
        });
    }

    /**
     * Проверка максимального числа ключевых фраз на группу.
     * При превышении для всех фраз группы добавляются ошибки.
     *
     * @param newKeywordsForAdd мапа с фактически добавляемыми ключевыми фразами,
     *                          т.е. очищенная от недобавляемых дубликатов
     *                          (смотреть {@link ru.yandex.direct.core.entity.keyword.service.KeywordsAddOperation}).
     */
    public void validateSizePerAdGroup(ValidationResult<List<Keyword>, Defect> validationResult,
                                       Map<Integer, InternalKeyword> newKeywordsForAdd, Map<Integer, InternalKeyword>
                                               existingKeywords,
                                       Long limit) {
        Map<Long, Integer> adGroupIdToNewKeywordsNumber = StreamEx.of(newKeywordsForAdd.values())
                .map(InternalKeyword::getAdGroupId)
                .sorted()
                .runLengths()
                .mapValues(l -> (int) (long) l)
                .toMap();
        Map<Long, Integer> adGroupIdToExistingKeywordsNumber = StreamEx.of(existingKeywords.values())
                .map(InternalKeyword::getAdGroupId)
                .sorted()
                .runLengths()
                .mapValues(l -> (int) (long) l)
                .toMap();

        ListValidationBuilder<Keyword, Defect> lvb = new ListValidationBuilder<>(validationResult);
        lvb.checkEach(
                maxNumberPerAdGroupIsInBounds(adGroupIdToNewKeywordsNumber, adGroupIdToExistingKeywordsNumber, limit),
                When.isValid());
    }

    public void validateSizeWithNonexistentAdGroups(ValidationResult<List<Keyword>, Defect> validationResult,
                                                    Map<Integer, AdGroupInfoForKeywordAdd> keywordsAdGroupInfo,
                                                    Long limit) {
        Map<Integer, Integer> adGroupsKeywordsNumber = StreamEx.of(keywordsAdGroupInfo.values())
                .map(AdGroupInfoForKeywordAdd::getAdGroupIndex)
                .sorted()
                .runLengths()
                .mapValues(l -> (int) (long) l)
                .toMap();

        ListValidationBuilder<Keyword, Defect> lvb = new ListValidationBuilder<>(validationResult);
        lvb.checkEachBy((index, keyword) -> {
            ModelItemValidationBuilder<Keyword> vb = ModelItemValidationBuilder.of(keyword);
            Integer adGroupIndex = keywordsAdGroupInfo.get(index).getAdGroupIndex();
            vb.check(maxNumberPerAdGroupIsInBounds(adGroupsKeywordsNumber.get(adGroupIndex), limit), When.isValid());
            return vb.getResult();
        });
    }

    private static Constraint<Long, Defect> adGroupIsAcceptableToAddKeywords(AdGroupType adGroupType) {
        return adGroupId -> ACCEPTABLE_AD_GROUP_TYPES_FOR_ADD_KEYWORD.contains(adGroupType)
                ? null
                : KeywordDefects.notAcceptableAdGroupType();
    }

    private static Constraint<Keyword, Defect> maxNumberPerAdGroupIsInBounds(Integer keywordsCount, Long limit) {
        return keyword -> keywordsCount <= limit
                ? null
                : maxKeywordsPerAdGroupExceeded(limit.intValue(),
                Math.min(keywordsCount - limit.intValue(), keywordsCount));
    }

    private static Constraint<Keyword, Defect> maxNumberPerAdGroupIsInBounds(
            Map<Long, Integer> adGroupIdToNewKeywordsNumber, Map<Long, Integer> adGroupIdToExistingKeywordsNumber,
            Long limit) {
        return keyword -> {
            if (keyword == null) {
                return null;
            }
            Long adGroupId = keyword.getAdGroupId();
            int newKeywordsNumber = defaultIfNull(adGroupIdToNewKeywordsNumber.get(adGroupId), 0);
            int existingKeywordsNumber = defaultIfNull(adGroupIdToExistingKeywordsNumber.get(adGroupId), 0);
            int summaryNumber = newKeywordsNumber + existingKeywordsNumber;
            if (summaryNumber <= limit) {
                return null;
            } else {
                int keywordsExceededCount = summaryNumber - limit.intValue();
                return maxKeywordsPerAdGroupExceeded(limit.intValue(),
                        Math.min(keywordsExceededCount, newKeywordsNumber));

            }
        };
    }
}
