package ru.yandex.direct.core.entity.keyword.service.validation;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.keyword.container.InternalKeyword;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;

import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordDefects.keywordAlreadySuspended;
import static ru.yandex.direct.core.entity.keyword.service.validation.KeywordDefects.keywordNotSuspended;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseSyntaxValidator.keywordSyntaxValidator;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Service
public class
UpdateKeywordValidationService {
    private static final int MAX_ELEMENTS_PER_OPERATION = 10000;

    private final ModelChangesValidationTool preValidationTool;
    private final ClientKeywordValidationService clientKeywordValidationService;

    @Autowired
    public UpdateKeywordValidationService(
            ClientKeywordValidationService clientKeywordValidationService) {
        this.clientKeywordValidationService = clientKeywordValidationService;

        preValidationTool = ModelChangesValidationTool.builder()
                .minSize(1).maxSize(MAX_ELEMENTS_PER_OPERATION).build();
    }

    public ValidationResult<List<ModelChanges<Keyword>>, Defect> preValidate(
            List<ModelChanges<Keyword>> modelChanges, Set<Long> existingKeywordIds, Set<Long> suspendedIds) {
        ValidationResult<List<ModelChanges<Keyword>>, Defect> validationResult =
                preValidationTool.validateModelChangesList(modelChanges, existingKeywordIds);

        return new ListValidationBuilder<>(validationResult).checkEachBy(mc ->
                        preValidateKeyword(mc,
                                suspendedIds),
                When.isValid()).getResult();

    }

    private ValidationResult<ModelChanges<Keyword>, Defect> preValidateKeyword(
            ModelChanges<Keyword> modelChanges,
            Set<Long> suspendedIds) {
        ItemValidationBuilder<ModelChanges<Keyword>, Defect> vb =
                ItemValidationBuilder.of(modelChanges, Defect.class);

        if (modelChanges.isPropChanged(Keyword.PHRASE)
                && modelChanges.getChangedProp(Keyword.PHRASE) != null) {
            vb.item(modelChanges.getChangedProp(Keyword.PHRASE), Keyword.PHRASE.name())
                    .check(notNull())
                    .checkBy(keywordSyntaxValidator(), When.isValid());
        }

        Long id = modelChanges.getId();
        Boolean suspendChange = modelChanges.getPropIfChanged(Keyword.IS_SUSPENDED);

        vb.item(id, Keyword.ID.name())
                .checkBy(setSameValueSuspendValidator(suspendChange, suspendedIds), When.isValid());

        return vb.getResult();
    }

    /**
     * Проверка на измненение флага c тем же значением
     */
    private Validator<Long, Defect> setSameValueSuspendValidator(Boolean suspendChange,
                                                                 Set<Long> suspendedIds) {
        return id -> {
            ItemValidationBuilder<Long, Defect> vb = ItemValidationBuilder.of(id);
            if (suspendChange != null && suspendChange) {
                vb.weakCheck(notInSet(suspendedIds), keywordAlreadySuspended());
            } else if (suspendChange != null) {
                vb.weakCheck(inSet(suspendedIds), keywordNotSuspended());
            }
            return vb.getResult();
        };
    }

    public ValidationResult<List<Keyword>, Defect> validate(
            ValidationResult<List<Keyword>, Defect> validationResult,
            Map<Integer, InternalKeyword> internalKeywordMap, long operatorUid,
            ClientId clientId) {
        return clientKeywordValidationService
                .validate(validationResult, internalKeywordMap, true, false, operatorUid, clientId);
    }
}
