package ru.yandex.direct.core.entity.keyword.service.validation.phrase;

import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ru.yandex.direct.utils.StringUtils;

import static org.apache.commons.lang3.StringUtils.countMatches;

public class CommonPhrasePredicates {

    public static final String WORD_START_REGEXP = "(?:^|[\\s\".\\[\\]'\\-+!]|-[!+]|\\[!)";
    public static final String WORD_FINISH_REGEXP = "(?:[\\s\".\\[\\]'\\-+!]|$)";

    private static final Pattern WRAPPED_QUOTES_PATTERN = Pattern.compile("^\"[^\"]+\"$");
    private static final Pattern EMPTY_QUOTES_PATTERN = Pattern.compile("\"[\\s]*\"");

    private static final Pattern EMPTY_SQUARE_BRACKETS_PATTERN = Pattern.compile("\\[\\s*]");
    private static final Pattern INNER_SQUARE_BRACKETS_PATTERN = Pattern.compile("\\[([^\\[\\]]+)]");
    private static final Pattern NO_SPACE_NEAR_SQUARE_BRACKETS_PATTERN = Pattern.compile("[^\"\\s]\\[|][^\"\\s]");

    private static final Pattern WRONG_EXCLAMATION_IN_WORD_PATTERN =
            Pattern.compile("!$|![.'\\-+!\"\\[\\]\\s]|[^\\s\\-\\[\"]!|[^\\s][-]!");

    private static final Pattern WRONG_PLUS_IN_WORD_PATTERN =
            Pattern.compile("\\+$|\\+[.'\\-+!\"\\[\\]\\s]|[^\\s\\-\\[\"]\\+|[^\\s][-]\\+");

    private static final Pattern WRONG_POINT_IN_WORD_PATTERN =
            Pattern.compile("\\.\\.|^\\.|\\.[\\[!+\\-']|[\\s\\[\\]\"!+\\-']\\.");

    private static final Pattern WRONG_APOSTROPHE_IN_WORD_PATTERN =
            Pattern.compile("''|^'|'[\\[!+\\-.]|[\\s\\[\\]\"!+\\-.]'");

    private static final Pattern WRONG_PLUS_IN_BRACKETS_PATTERN = Pattern.compile("\\[[^]]*\\+[^]]*]");

    /**
     * Проверка правильности использования кавычек
     */
    public static Predicate<String> validQuotes() {
        return keyword -> !keyword.contains("\"")
                || (matches(WRAPPED_QUOTES_PATTERN, keyword) && noMatches(EMPTY_QUOTES_PATTERN, keyword));
    }

    /**
     * Проверка максимальной длины отдельных слов без операторов
     */
    public static Predicate<String> balancedSquareBrackets() {
        return StringUtils::areSquareBracketsBalanced;
    }

    /**
     * Проверка вложенности и пустоты квадратных скобок
     */
    public static Predicate<String> noNestedOrEmptySquareBrackets() {
        return keyword -> noMatches(EMPTY_SQUARE_BRACKETS_PATTERN, keyword)
                && matchCount(INNER_SQUARE_BRACKETS_PATTERN.matcher(keyword)) == countMatches(keyword, '[')
                && noMatches(NO_SPACE_NEAR_SQUARE_BRACKETS_PATTERN, keyword);
    }

    /**
     * Проверка использования оператора "!"
     */
    public static Predicate<String> validExclamationMark() {
        return keyword -> noMatches(WRONG_EXCLAMATION_IN_WORD_PATTERN, keyword);
    }

    /**
     * Проверка использования знака "+"
     */
    public static Predicate<String> validPlusMark() {
        return keyword -> noMatches(WRONG_PLUS_IN_WORD_PATTERN, keyword);
    }

    /**
     * Проверка использования точки
     */
    public static Predicate<String> validPoint() {
        return keyword -> noMatches(WRONG_POINT_IN_WORD_PATTERN, keyword);
    }

    /**
     * Проверка использования точки
     */
    public static Predicate<String> validApostrophe() {
        return keyword -> noMatches(WRONG_APOSTROPHE_IN_WORD_PATTERN, keyword);
    }

    /**
     * Проверка отсутствия оператора "+" внутри квадратных скобок
     */
    public static Predicate<String> noPlusMarkInBrackets() {
        return keyword -> !keyword.contains("[") ||
                noMatches(WRONG_PLUS_IN_BRACKETS_PATTERN, keyword);
    }

    public static boolean noMatches(Pattern pattern, String keyword) {
        return !pattern.matcher(keyword).find();
    }

    public static boolean matches(Pattern pattern, String keyword) {
        return pattern.matcher(keyword).matches();
    }

    private static int matchCount(Matcher matcher) {
        int count = 0;
        while (matcher.find()) {
            count++;
        }

        return count;
    }
}
