package ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase;

import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.google.common.base.CharMatcher;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.utils.TextUtils;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;

import static com.google.common.base.CharMatcher.anyOf;
import static java.util.Arrays.stream;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.lang3.StringUtils.countMatches;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.cannotContainLoneDot;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.cannotContainsOnlyMinusWords;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.containsOnlyStopWords;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.emptyOrNestedSquareBrackets;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.incorrectCombinationOfSpecialSymbols;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.incorrectUseOfExclamationMarks;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.incorrectUseOfMinusSign;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.incorrectUseOfPlusSign;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.minusWordCannotStartFromDotOrApostrophe;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.minusWordNotInQuotedPhrase;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.minusWordsCannotSubtractPlusWords;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.minusWordsNoPhraseWithDot;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.noMinusPhrasesOnlyWords;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.onlySingleDotBetweenNumbers;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.tooLongKeyword;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.tooManyWords;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.unpairedQuotes;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.advqphrase.AdvqPhraseDefects.unpairedSquareBrackets;
import static ru.yandex.direct.utils.TextConstants.LETTERS_AND_NUMBERS;
import static ru.yandex.direct.utils.TextUtils.isMinusWord;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.defect.CollectionDefects.maxStringLength;
import static ru.yandex.direct.validation.defect.StringDefects.admissibleChars;

/**
 * Производит основную валидацию ключевой фразы
 */
@Service
public class AdvqPhraseValidator implements DefaultValidator<String> {

    private StopWordService stopWordService;

    /**
     * Создает экземпляр валидатора
     *
     * @param stopWordService сервис доступа к стоп-словам
     */
    @Autowired
    public AdvqPhraseValidator(StopWordService stopWordService) {
        this.stopWordService = stopWordService;
    }

    @Override
    public ValidationResult<String, Defect> apply(String keyword) {
        ItemValidationBuilder<String, Defect> builder = ItemValidationBuilder.of(keyword);
        builder.check(stringLengthNoMoreThan(MAX_CHARACTERS_COUNT))
                .check(noDisallowedCharacters(), When.isValid())
                .check(noLoneDots(), When.isValid())
                .check(noUnpairedQuotes(), When.isValid())
                .check(keywordsNoMoreThan(MAX_KEYWORDS_COUNT, stopWordService), When.isValid())
                .check(notJustStopWords(stopWordService), When.isValid())
                .check(notJustMinusWords(), When.isValid())
                .check(eachKeywordNoLongerThan(MAX_KEYWORD_LENGTH), When.isValid())
                .check(squareBrackets())
                .check(exclamationMarks())
                .check(minusSign())
                .check(plusSign())
                .check(minusWords());
        return builder.getResult();
    }

    /**
     * Максимальное количество слов в ключевой фразе (без учета минус-слов и стоп-слов)
     */
    public static final int MAX_KEYWORDS_COUNT = 7;
    /**
     * Максимальная длина ключевого слова (без учета спец-символов)
     */
    public static final int MAX_KEYWORD_LENGTH = 35;
    /**
     * Максимальная длина ключевой фразы (без учета спецкомбинаций символов)
     */
    public static final int MAX_CHARACTERS_COUNT = 4096;

    private static final Pattern loneDot = Pattern.compile("(?:^|[\\s\"])\\.+(?:$|[\\s\"])");
    private static final Pattern nestedLeftSquareBracket = Pattern.compile("\\[[^\\[]*\\[");
    private static final Pattern nestedRightSquareBracket = Pattern.compile("][^]]*]");
    private static final Pattern insideSquareBrackets = Pattern.compile("\\[([^]]+)]");
    private static final Pattern modifiersInsideSquareBrackets = Pattern.compile("[+\"]");
    private static final Pattern minusModifiersInsideSquareBrackets = Pattern.compile("(?<!\\S)-");
    private static final Pattern exclamationMarkWithOtherModifiers = Pattern.compile("[^-!\"\\s\\[]![^-!\\s]");
    private static final Pattern plusSignWithOtherModifiers = Pattern.compile("[^-+\"\\s]\\+[^-+\\s]");
    private static final Pattern wordStart = Pattern.compile("^|[\\s\".\\[\\]'\\-+!]|-[!+]|\\[!");

    private static final String ALLOW_KEYWORD_LETTERS = LETTERS_AND_NUMBERS + "[].\"-+! ";
    private static final CharMatcher ALLOW_KEYWORD_LETTERS_MATCHER = anyOf(ALLOW_KEYWORD_LETTERS);

    public static Constraint<String, Defect> stringLengthNoMoreThan(int length) {
        return keyword -> keyword.replaceAll("-[!+]", "-").length() <= length ? null : maxStringLength(length);
    }

    public static Constraint<String, Defect> noDisallowedCharacters() {
        return fromPredicate(ALLOW_KEYWORD_LETTERS_MATCHER::matchesAllOf, admissibleChars());
    }

    public static Constraint<String, Defect> noLoneDots() {
        return keyword -> !loneDot.matcher(keyword).find() ?
                null :
                cannotContainLoneDot();
    }

    public static Constraint<String, Defect> noUnpairedQuotes() {
        return keyword -> keyword.codePoints().filter(ch -> ch == '"').count() % 2 == 0
                ? null : unpairedQuotes();
    }

    public static Constraint<String, Defect> notJustStopWords(StopWordService stopWordService) {
        return keyword ->
                countKeywords(keyword, kw -> true) != countKeywords(keyword, kw -> isStopWord(kw, stopWordService))
                        || countKeywords(keyword, kw -> isStopWord(kw, stopWordService)) == 0
                        ? null
                        : containsOnlyStopWords();
    }

    public static Constraint<String, Defect> notJustMinusWords() {
        return keyword -> countKeywords(keyword, kw -> true) != countKeywords(keyword, TextUtils::isMinusWord)
                || countKeywords(keyword, TextUtils::isMinusWord) == 0
                ? null : cannotContainsOnlyMinusWords();
    }

    public static Constraint<String, Defect> keywordsNoMoreThan(int count, StopWordService stopWordService) {
        return keyword -> countKeywords(keyword, kw -> !isStopWord(kw, stopWordService) && !isMinusWord(kw)) <= count
                ? null
                : tooManyWords(count);
    }

    public static Constraint<String, Defect> eachKeywordNoLongerThan(int length) {
        return keyword -> {
            List<String> tooLongWords = stream(split(keyword))
                    .filter(kw -> !isMinusWord(kw))
                    .filter(kw -> normalize(kw).length() > length)
                    .collect(toList());
            return tooLongWords.isEmpty()
                    ? null
                    : tooLongKeyword(length);
        };
    }

    public static Constraint<String, Defect> squareBrackets() {
        return keyword -> {

            long bracketsNumberDifference = 0;
            for (char symb : keyword.toCharArray()) {
                if (symb == '[') {
                    bracketsNumberDifference += 1;
                }
                if (symb == ']') {
                    bracketsNumberDifference -= 1;
                }
                if (bracketsNumberDifference < 0) {
                    return unpairedSquareBrackets();
                }
            }

            if (bracketsNumberDifference != 0) {
                return unpairedSquareBrackets();
            }

            if (keyword.contains("[]")
                    || nestedLeftSquareBracket.matcher(keyword).find()
                    || nestedRightSquareBracket.matcher(keyword).find()) {
                return emptyOrNestedSquareBrackets();
            }

            Matcher matcher = insideSquareBrackets.matcher(keyword);
            while (matcher.find()) {
                String inside = matcher.group(1);
                if (modifiersInsideSquareBrackets.matcher(inside).find()
                        || minusModifiersInsideSquareBrackets.matcher(inside).find()) {
                    return AdvqPhraseDefects.modifiersInsideSquareBrackets();
                }
            }
            return null;
        };
    }

    public static Constraint<String, Defect> exclamationMarks() {
        return keyword -> !keyword.contains("!!")
                && !keyword.contains("! ")
                && !keyword.endsWith("!")
                && !exclamationMarkWithOtherModifiers.matcher(keyword).find()
                ? null
                : incorrectUseOfExclamationMarks();
    }

    public static Constraint<String, Defect> minusSign() {
        return keyword ->
                !keyword.contains("--")
                        && !keyword.contains("- ")
                        && !keyword.endsWith("-")
                        && !keyword.contains("!-")
                        && !keyword.contains("+-")
                        && !matches(keyword, "\\S-!")
                        && !matches(keyword, "\\S-\\+")
                        && !matches(keyword, "[-!]{3,}")
                        && !matches(keyword, "[-+]{3,}")
                        ? null
                        : incorrectUseOfMinusSign();
    }

    public static Constraint<String, Defect> plusSign() {
        return keyword -> !keyword.contains("++")
                && !keyword.contains("+ ")
                && !keyword.endsWith("+")
                && !plusSignWithOtherModifiers.matcher(keyword).find()
                ? null
                : incorrectUseOfPlusSign();
    }

    public static Constraint<String, Defect> minusWords() {
        return keyword -> {
            if (matches(keyword, "(^|\\s)\\-.*?\\s[^\\-\\s]")
                    || matches(keyword, "(^|\\s)\\-.*?\\S-")) {
                return noMinusPhrasesOnlyWords();
            }
            Pattern notNumber = Pattern.compile("[^0-9.!+]");
            if (countKeywords(keyword,
                    kw -> isMinusWord(kw) && kw.contains(".") && notNumber.matcher(kw.replace("-", "")).find()) > 0) {
                return minusWordsNoPhraseWithDot();
            }
            if (matches(keyword, "[\\.\\[\\]\\'\\-\\+\\!]{2,}")
                    && !matches(keyword, "-[!+]")
                    && !matches(keyword, "\\[!")) {
                return incorrectCombinationOfSpecialSymbols();
            }
            if (matches(keyword, "(?:" + wordStart.pattern() + ")[\\.\\']")) {
                return minusWordCannotStartFromDotOrApostrophe();
            }
            if (matches(keyword, "^\".+\"$") && matches(keyword, "(?:^\\\"|\\s)-")) {
                return minusWordNotInQuotedPhrase();
            }
            if (countKeywords(keyword, kw -> isMinusWord(kw) && countMatches(kw, '.') > 1) > 0) {
                return onlySingleDotBetweenNumbers();
            }
            Set<String> allWords = stream(split(keyword)).collect(toSet());
            for (String word : allWords) {
                if (allWords.contains("-" + word)) {
                    return minusWordsCannotSubtractPlusWords(Collections.emptyList());
                }
            }
            return null;
        };
    }

    public static long countKeywords(String keyword, Predicate<String> predicate) {
        return stream(split(keyword))
                .filter(predicate)
                .count();
    }

    public static boolean isStopWord(String word, StopWordService stopWordService) {
        return !isMinusWord(word) && stopWordService.isStopWord(normalize(word));
    }

    public static String normalize(String keyword) {
        return keyword.replaceAll("[" + Pattern.quote("[]+!\"") + "]", "")
                .replace('ё', 'е').replace('Ё', 'Е');
    }

    public static String[] split(String keyword) {
        return keyword.replace("!", "").split("\\s+");
    }

    public static boolean matches(String str, String pattern) {
        return Pattern.compile(pattern).matcher(str).find();
    }

}
