package ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase;

import java.util.Arrays;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;

import ru.yandex.advq.query.ast.Word;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.CommonPhrasePredicates;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.libs.keywordutils.StopWordMatcher;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;
import ru.yandex.direct.libs.keywordutils.model.AnyKeyword;
import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;
import ru.yandex.direct.libs.keywordutils.model.OrderedKeyword;
import ru.yandex.direct.libs.keywordutils.model.SingleKeyword;
import ru.yandex.direct.libs.keywordutils.parser.SplitCompoundWordExpressionTransform;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.libs.keywordutils.inclusion.KeywordInclusionUtils.getIntersectedPlusWords;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class PhraseConstraints {

    public static final int KEYWORD_MAX_LENGTH = 4096;
    public static final int WORD_MAX_LENGTH = 35;
    public static final int WORDS_MAX_COUNT = 7;

    private PhraseConstraints() {
    }

    static Constraint<String, Defect> maxLengthWithoutMinusExclamationAndSpaces() {
        return Constraint.fromPredicate(
                PhrasePredicates.maxLengthWithoutMinusExclamationAndSpaces(KEYWORD_MAX_LENGTH),
                PhraseDefects.tooLongKeyword(KEYWORD_MAX_LENGTH));
    }

    /**
     * Максимальная длина отдельных плюс слов.
     * <p>
     * На вход предикату должны приходить фразы, в которых минус-слова
     * состоят из единичных слов без квадратных скобок, то есть уже проверенные
     * предикатом {@link #onlySingleMinusWords(KeywordWithMinuses)}.
     * <p>
     * Проверка проводится на разобранной фразе, так как при парсинге слова разделяются
     * по особым правилам и проверку хочется сделать независимой от этих деталей.
     */
    public static Constraint<String, Defect> maxPlusWordLength(KeywordWithMinuses parsedKeyword) {
        return keyword -> {
            Keyword plusKeyword = parsedKeyword.getKeyword();

            List<AnyKeyword> singlePlusKeywords =
                    filterList(plusKeyword.getAllKeywords(), kw -> kw instanceof SingleKeyword);
            List<AnyKeyword> orderedPlusKeywords =
                    filterList(plusKeyword.getAllKeywords(), kw -> kw instanceof OrderedKeyword);
            List<String> tooLongWords = StreamEx.of(orderedPlusKeywords)
                    .map(kw -> (OrderedKeyword) kw)
                    .flatCollection(OrderedKeyword::getSingleKeywords)
                    .append(mapList(singlePlusKeywords, kw -> (SingleKeyword) kw))
                    .map(SingleKeyword::getWord)
                    .map(Word::getText)
                    .filter(s -> s.length() > WORD_MAX_LENGTH)
                    .toList();
            if (tooLongWords.isEmpty()) {
                return null;
            }
            return PhraseDefects.tooLongWord(WORD_MAX_LENGTH, tooLongWords);
        };
    }

    /**
     * Максимальная длина отдельных минус слов.
     * <p>
     * На вход предикату должны приходить фразы, в которых минус-слова
     * состоят из единичных слов без квадратных скобок, то есть уже проверенные
     * предикатом {@link #onlySingleMinusWords(KeywordWithMinuses)}.
     * <p>
     * Проверка проводится на разобранной фразе, так как при парсинге слова разделяются
     * по особым правилам и проверку хочется сделать независимой от этих деталей.
     */
    public static Constraint<String, Defect> maxMinusWordLength(KeywordWithMinuses parsedKeyword) {
        return keyword -> {
            List<AnyKeyword> singleMinusKeywords = mapList(parsedKeyword.getMinusKeywords(),
                    kw -> kw.getAllKeywords().get(0));
            List<String> tooLongWords = StreamEx.of(singleMinusKeywords)
                    .map(kw -> (SingleKeyword) kw)
                    .map(SingleKeyword::getWord)
                    .map(Word::getText)
                    .filter(s -> s.length() > WORD_MAX_LENGTH)
                    .toList();
            if (tooLongWords.isEmpty()) {
                return null;
            }
            return PhraseDefects.tooLongMinusWord(WORD_MAX_LENGTH, tooLongWords);
        };
    }

    static Constraint<String, Defect> allowedChars() {
        return keyword -> {

            List<String> incorrectWords = Arrays.stream(keyword.split("\\s+"))
                    .filter(word -> !PhrasePredicates.allowedChars().test(word))
                    .collect(Collectors.toList());

            return incorrectWords.isEmpty() ? null : PhraseDefects.illegalCharacters(incorrectWords);
        };
    }

    static Constraint<String, Defect> validQuotes() {
        return Constraint.fromPredicate(
                CommonPhrasePredicates.validQuotes(),
                PhraseDefects.invalidQuotes());
    }

    static Constraint<String, Defect> validSquareBrackets() {
        return Constraint.fromPredicate(
                CommonPhrasePredicates.balancedSquareBrackets()
                        .and(CommonPhrasePredicates.noNestedOrEmptySquareBrackets()),
                PhraseDefects.invalidBrackets());
    }

    static Constraint<String, Defect> validExclamationMark() {
        return Constraint.fromPredicate(
                CommonPhrasePredicates.validExclamationMark(),
                PhraseDefects.invalidExclamationMark());
    }

    static Constraint<String, Defect> validPlusMark() {
        return Constraint.fromPredicate(
                CommonPhrasePredicates.validPlusMark(),
                PhraseDefects.invalidPlusMark());
    }

    static Constraint<String, Defect> validMinusMark() {
        return Constraint.fromPredicate(
                PhrasePredicates.validMinusMark(),
                PhraseDefects.invalidMinusMark());
    }

    static Constraint<String, Defect> validPoint() {
        return Constraint.fromPredicate(
                CommonPhrasePredicates.validPoint(),
                PhraseDefects.invalidPoint());
    }

    static Constraint<String, Defect> validApostrophe() {
        return Constraint.fromPredicate(
                CommonPhrasePredicates.validApostrophe(),
                PhraseDefects.invalidApostrophe());
    }

    static Constraint<String, Defect> noBothQuotesAndMinusWords() {
        return Constraint.fromPredicate(
                PhrasePredicates.noBothQuotesAndMinusWords(),
                PhraseDefects.bothQuotesAndMinusWords());
    }

    static Constraint<String, Defect> noMinusWordsInsideBracketsOrQuotes() {
        return Constraint.fromPredicate(
                PhrasePredicates.noMinusWordsInsideBracketsOrQuotes(),
                PhraseDefects.minusWordInsideBracketsOrQuotes());
    }

    static Constraint<String, Defect> noPlusMarkInBrackets() {
        return Constraint.fromPredicate(
                CommonPhrasePredicates.noPlusMarkInBrackets(),
                PhraseDefects.plusMarkInBrackets());
    }

    static Constraint<String, Defect> noBracketsInMinusWords() {
        return Constraint.fromPredicate(
                PhrasePredicates.noBracketsInMinusWords(),
                PhraseDefects.bracketsInMinusWords());
    }


    /**
     * Разрешены только единичные минус-слова.
     * Проверяет с учетом разделения слов, применяемого при парсинге.
     */

    static Constraint<String, Defect> onlySingleMinusWordsRaw() {
        return keyword -> {
            final String startStr = " -";
            int startPos = keyword.indexOf(startStr);
            if (startPos < 0) {
                return null;
            }
            String minusWordsSubstring = keyword.substring(startPos + startStr.length());
            List<String> minusWords = StreamEx.of(Pattern.compile("\\s+-").split(minusWordsSubstring))
                    .map(String::trim)
                    .remove(String::isEmpty)
                    .toList();

            for (String minusWord : minusWords) {
                if (minusWord.contains(" ")) {
                    return PhraseDefects.notSingleMinusWord(minusWords);
                }
            }
            for (String minusWord : minusWords) {
                List<String> minusWordParts =
                        SplitCompoundWordExpressionTransform.splitRawWordRegexp(minusWord);
                if (minusWordParts.size() != 1) {
                    return PhraseDefects.notSingleMinusWord(minusWords);
                }
            }
            return null;
        };
    }

    static Constraint<String, Defect> containsPlusWords() {
        return Constraint.fromPredicate(
                PhrasePredicates.containsPlusWords(),
                PhraseDefects.noPlusWords());
    }

    static Constraint<String, Defect> onlySingleMinusWords(@Nullable KeywordWithMinuses parsedKeyword) {
        // по факту тут проверяется parsedKeyword, значит, если он null, то проверять нечего
        if (parsedKeyword == null) {
            return keyword -> null;
        }
        return Constraint.fromPredicate(
                PhrasePredicates.onlySingleMinusWords(parsedKeyword),
                PhraseDefects.notSingleMinusWord(
                        parsedKeyword.getMinusKeywords().stream()
                                .map(Keyword::toString).collect(Collectors.toList())));
    }

    static Constraint<String, Defect> noMinusWordsDeletingPlusWords(KeywordWithLemmasFactory keywordFactory,
                                                                    StopWordMatcher stopWordMatcher,
                                                                    KeywordWithMinuses parsedKeyword) {
        return keyword -> {
            if (keyword == null) {
                return null;
            }

            List<Keyword> intersectedPlusWords =
                    getIntersectedPlusWords(keywordFactory, stopWordMatcher, parsedKeyword);
            if (intersectedPlusWords.isEmpty()) {
                return null;
            }

            List<String> errorWords = intersectedPlusWords.stream().map(Keyword::toString).collect(Collectors.toList());

            return PhraseDefects.minusWordDeletePlusWord(errorWords);

        };
    }

    static Constraint<String, Defect> plusWordsNoMoreThanMax(StopWordService stopWordService,
                                                             KeywordWithMinuses parsedKeyword) {
        return Constraint.fromPredicate(
                PhrasePredicates.plusWordsNoMoreThanMax(WORDS_MAX_COUNT, stopWordService, parsedKeyword),
                PhraseDefects.tooManyWords(WORDS_MAX_COUNT));
    }

    static Constraint<String, Defect> notOnlyStopWords(StopWordService stopWordService,
                                                       KeywordWithMinuses parsedKeyword) {
        return Constraint.fromPredicate(
                PhrasePredicates.notOnlyStopWords(stopWordService, parsedKeyword),
                PhraseDefects.onlyStopWords());
    }
}
