package ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase;

import java.util.List;
import java.util.function.Predicate;
import java.util.regex.Pattern;

import com.google.common.base.CharMatcher;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.keyword.processing.KeywordProcessingUtils;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.libs.keywordutils.StopWordMatcher;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;
import ru.yandex.direct.libs.keywordutils.model.AnyKeyword;
import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;
import ru.yandex.direct.libs.keywordutils.model.OrderedKeyword;
import ru.yandex.direct.libs.keywordutils.model.SingleKeyword;
import ru.yandex.direct.libs.keywordutils.parser.SplitCompoundWordExpressionTransform;
import ru.yandex.direct.utils.TextConstants;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.CommonPhrasePredicates.noMatches;
import static ru.yandex.direct.libs.keywordutils.inclusion.KeywordInclusionUtils.getIntersectedPlusWords;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public final class PhrasePredicates {

    private static final Pattern WRONG_MINUS_IN_WORD_PATTERN =
            Pattern.compile("(--)|" +
                    "(-([\\s.\"'\\[\\]]|$))|" +
                    "([]'.+!]-)|" +
                    "[\\s\"\\[]-[\\s\"\\[\\]]");

    private static final Pattern MINUS_WORD_START_PATTERN = Pattern.compile("([ \\[\\]\\-\"]-)|(^-)");

    private static final Pattern MINUS_INSIDE_BRACKETS_OR_QUOTES_PATTERN =
            Pattern.compile("(\\[-)|"
                    + "(\\[[^]]*\\s-)|"
                    + "(^[^\"]*\"-)|"
                    + "(^[^\"]*\"[^\"]*\\s-)");

    private static final Pattern BRACKETS_IN_MINUS_WORDS_PATTERN =
            Pattern.compile("\\s-[^\\[\\]-]*+[\\[\\]]");

    private static final Pattern ONLY_MINUS_WORDS_PATTERN =
            Pattern.compile("^\\s*-");

    private static final Pattern SPLIT_MINUS_WORDS_PATTERN =
            Pattern.compile("\\s+-");

    private static final CharMatcher ALLOW_KEYWORD_CHARS =
            CharMatcher.anyOf(TextConstants.LETTERS + TextConstants.NUMBERS + "\"!+-[] .'");

    private PhrasePredicates() {
    }

    /**
     * Проверка максимальной длины фразы не учитывая `!` и `+` в минус-словах
     */
    public static Predicate<String> maxLengthWithoutMinusExclamationAndSpaces(int maxLength) {
        return keyword -> KeywordProcessingUtils.getLengthWithoutMinusExclamationAndSpaces(keyword) <= maxLength;
    }

    /**
     * Проверка отсутствия недопустимых символов
     */
    public static Predicate<String> allowedChars() {
        return ALLOW_KEYWORD_CHARS::matchesAllOf;
    }

    /**
     * Проверка использования знака "-"
     */
    public static Predicate<String> validMinusMark() {
        return keyword -> noMatches(WRONG_MINUS_IN_WORD_PATTERN, keyword);
    }

    /**
     * Проверка отсутствия в ключевой фразе минус-слов одновременно с кавычками
     */
    public static Predicate<String> noBothQuotesAndMinusWords() {
        return keyword -> !keyword.contains("\"") || noMatches(MINUS_WORD_START_PATTERN, keyword);
    }

    /**
     * Минус-слова запрещены внутри квадратных скобок и кавычек
     */
    public static Predicate<String> noMinusWordsInsideBracketsOrQuotes() {
        return keyword -> noMatches(MINUS_INSIDE_BRACKETS_OR_QUOTES_PATTERN, keyword);
    }

    /**
     * Минус-слова запрещены внутри квадратных скобок и кавычек
     */
    public static Predicate<String> noBracketsInMinusWords() {
        return keyword -> noMatches(BRACKETS_IN_MINUS_WORDS_PATTERN, keyword);
    }

    /**
     * Фраза должна содержать плюс-слова
     */
    public static Predicate<String> containsPlusWords() {
        return keyword -> noMatches(ONLY_MINUS_WORDS_PATTERN, keyword);
    }

    /**
     * Разрешены только отдельные единичные минус-слова без кавычек и квадратных скобок.
     * Это дополнительная проверка на разобранной фразе для гарантии, что логика
     * разделения слов в валидации и парсере не отличается.
     */
    public static Predicate<String> onlySingleMinusWords(KeywordWithMinuses parsedKeyword) {
        return keyword -> {
            List<Keyword> minusKeywords = parsedKeyword.getMinusKeywords();
            for (Keyword minusKeyword : minusKeywords) {
                if (minusKeyword.isQuoted()) {
                    return false;
                }
                //получаем список SingleKeyword и разделяем составные слова
                List<String> singleKeywordTexts = StreamEx.of(minusKeyword.getAllKeywords())
                        .filter(kw -> kw instanceof SingleKeyword)
                        .map(kw -> (SingleKeyword) kw)
                        .map(sk -> sk.getWord().getText())
                        .flatCollection(SplitCompoundWordExpressionTransform::splitRawWordRegexp)
                        .toList();
                List<AnyKeyword> otherKeywords =
                        filterList(minusKeyword.getAllKeywords(), kw -> !(kw instanceof SingleKeyword));
                if (!otherKeywords.isEmpty()) {
                    return false;
                }
                if (singleKeywordTexts.size() != 1) {
                    return false;
                }
            }
            return true;
        };
    }

    /**
     * Минус-слова не могут вычитать плюс-слова
     */
    public static Predicate<String> noMinusWordsDeletingPlusWords(KeywordWithLemmasFactory keywordFactory,
                                                                  StopWordMatcher stopWordMatcher,
                                                                  KeywordWithMinuses parsedKeyword) {
        return keyword -> getIntersectedPlusWords(keywordFactory, stopWordMatcher, parsedKeyword).isEmpty();
    }

    /**
     * Проверка ограничения количества плюс-слов
     */
    public static Predicate<String> plusWordsNoMoreThanMax(int maxWords, StopWordService stopWordService,
                                                           KeywordWithMinuses normalKeyword) {
        return keyword -> {
            Keyword plusKeyword = normalKeyword.getKeyword();

            List<AnyKeyword> singleKeywords =
                    filterList(plusKeyword.getAllKeywords(), kw -> kw instanceof SingleKeyword);
            List<AnyKeyword> orderedKeywords =
                    filterList(plusKeyword.getAllKeywords(), kw -> kw instanceof OrderedKeyword);
            checkState(singleKeywords.size() + orderedKeywords.size() == plusKeyword.getAllKeywords().size(),
                    "keyword contains unsupported sub keywords type");

            if (!plusKeyword.isQuoted()) {
                singleKeywords = filterList(singleKeywords, kw -> !stopWordService.isStopWord(kw.toString()));
            }

            long wordsCount = StreamEx.of(orderedKeywords)
                    .map(kw -> (OrderedKeyword) kw)
                    .flatCollection(OrderedKeyword::getSingleKeywords)
                    .append(mapList(singleKeywords, kw -> (SingleKeyword) kw))
                    .flatCollection(
                            kw -> SplitCompoundWordExpressionTransform.splitRawWordRegexp(kw.getWord().getText()))
                    .count();
            return wordsCount <= maxWords;
        };
    }

    /**
     * Плюс-фраза не может состоять только из стоп-слов
     */
    public static Predicate<String> notOnlyStopWords(StopWordService stopWordService,
                                                     KeywordWithMinuses parsedKeyword) {
        return keyword -> {
            Keyword plusKeyword = parsedKeyword.getKeyword();

            List<AnyKeyword> singleKeywords =
                    filterList(plusKeyword.getAllKeywords(), kw -> kw instanceof SingleKeyword);
            List<AnyKeyword> orderedKeywords =
                    filterList(plusKeyword.getAllKeywords(), kw -> kw instanceof OrderedKeyword);
            checkState(singleKeywords.size() + orderedKeywords.size() == plusKeyword.getAllKeywords().size(),
                    "keyword contains unsupported sub keywords type");

            return StreamEx.of(orderedKeywords)
                    .map(kw -> (OrderedKeyword) kw)
                    .flatCollection(OrderedKeyword::getSingleKeywords)
                    .append(mapList(singleKeywords, kw -> (SingleKeyword) kw))
                    .anyMatch(singleKeyword -> !stopWordService.isStopWord(singleKeyword.getWord().getText()));
        };
    }
}
