package ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.libs.keywordutils.StopWordMatcher;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseConstraints.maxMinusWordLength;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseConstraints.maxPlusWordLength;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseConstraints.noMinusWordsDeletingPlusWords;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseConstraints.notOnlyStopWords;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseConstraints.onlySingleMinusWords;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseConstraints.plusWordsNoMoreThanMax;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseSyntaxValidator.keywordSyntaxValidator;

/**
 * Класс проверок для самой ключевой фразы
 */
public class PhraseValidator implements DefaultValidator<String> {

    private final StopWordService stopWordService;
    private final KeywordWithLemmasFactory keywordFactory;
    private final StopWordMatcher stopWordMatcher;
    private final KeywordWithMinuses parsedKeyword;
    private final boolean weakenValidation;

    /**
     * Собраны проверки на валидность самой ключевой фразы, перечечения с минус словами,
     * проверки нормализованной фразы
     * @param weakenValidation Ослабленная валидация, пропускает проверки, которые могут не проходить для существующих
     *                         фраз
     */
    public PhraseValidator(StopWordService stopWordService,
                           KeywordWithLemmasFactory keywordFactory,
                           @Nullable KeywordWithMinuses parsedKeyword,
                           boolean weakenValidation) {
        this.stopWordService = stopWordService;
        this.keywordFactory = keywordFactory;
        this.parsedKeyword = parsedKeyword;
        this.weakenValidation = weakenValidation;
        stopWordMatcher = stopWordService::isStopWord;
    }

    /**
     * Собраны проверки на валидность самой ключевой фразы, перечечения с минус словами,
     * проверки нормализованной фразы
     */
    public PhraseValidator(StopWordService stopWordService, KeywordWithLemmasFactory keywordFactory,
                          @Nullable KeywordWithMinuses parsedKeyword) {
        this(stopWordService, keywordFactory, parsedKeyword, false);
    }

    @Override
    public ValidationResult<String, Defect> apply(String keyword) {
        ItemValidationBuilder<String, Defect> ivb = ItemValidationBuilder.of(keyword);

        ivb.checkBy(keywordSyntaxValidator());
        checkState(ivb.getResult().hasAnyErrors() || parsedKeyword != null,
                "Parsed phrase must be provided when it doesn't contain syntax errors: %s", keyword);
        ivb.check(onlySingleMinusWords(parsedKeyword), When.isValid())
                .check(maxPlusWordLength(parsedKeyword), When.isValid())
                .check(maxMinusWordLength(parsedKeyword), When.isValid())
                .check(noMinusWordsDeletingPlusWords(keywordFactory, stopWordMatcher, parsedKeyword),
                        When.isValidAnd(When.isFalse(weakenValidation)))
                .check(plusWordsNoMoreThanMax(stopWordService, parsedKeyword),
                        When.isValidAnd(When.isFalse(weakenValidation)))
                .check(notOnlyStopWords(stopWordService, parsedKeyword), When.isValid());
        return ivb.getResult();
    }
}

