package ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase;

import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import ru.yandex.direct.core.entity.keyword.service.validation.phrase.CommonPhrasePredicates;
import ru.yandex.direct.core.entity.minuskeywordspack.model.MinusKeywordsPack;
import ru.yandex.direct.core.validation.defects.Defects;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseDefects.maxCountWordsInKeyword;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseDefects.maxLengthMinusKeywords;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseDefects.maxLengthMinusWord;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

public class MinusPhraseConstraints {

    public static final int MAX_LIBRARY_PACKS_COUNT = 30;
    public static final int CAMPAIGN_MINUS_KEYWORDS_MAX_LENGTH = 20000;
    public static final int CAMPAIGN_MINUS_KEYWORDS_MAX_LENGTH_BEFORE_NORMALIZATION =
            4 * CAMPAIGN_MINUS_KEYWORDS_MAX_LENGTH;

    public static final int GROUP_MINUS_KEYWORDS_MAX_LENGTH = 4096;
    public static final int GROUP_MINUS_KEYWORDS_MAX_LENGTH_BEFORE_NORMALIZATION = 4 * GROUP_MINUS_KEYWORDS_MAX_LENGTH;

    public static final int WORD_MAX_LENGTH = 35;
    public static final int WORDS_MAX_COUNT = 7;

    public static final int MAX_LINKED_PACKS_TO_ONE_AD_GROUP = 3;

    private MinusPhraseConstraints() {
    }

    /*
     * Проверка общей максимальной длины всех строк списка без учёта вхождения операторов
     */
    public static Constraint<List<String>, Defect> maxLengthKeywordsWithoutSpecSymbolsAndSpaces(
            int maxLength) {
        return fromPredicate(MinusPhrasePredicates.maxLengthKeywordsWithoutSpecSymbolsAndSpaces(maxLength),
                maxLengthMinusKeywords(maxLength));
    }


    /**
     * Проверка общего количества библиотечных наборов у пользователя.
     *
     * @param totalLibraryPacksCount - количество существующих наборов + колиество новых
     */
    public static Constraint<MinusKeywordsPack, Defect> maxLibraryPacksCount(int totalLibraryPacksCount) {
        return fromPredicate(newPack -> totalLibraryPacksCount <= MAX_LIBRARY_PACKS_COUNT,
                Defects.negativeKeywordSetsLimitExceeded(MAX_LIBRARY_PACKS_COUNT));
    }

    /*
     * Проверка максимальной длины минус-слов разделенных пробелом в минус-фразе без учёта вхождения операторов
     */
    static Constraint<List<String>, Defect> maxLengthWordInKeyword() {
        final int maxMinusWordLength = WORD_MAX_LENGTH;
        return keywordListConstraint(MinusPhrasePredicates.maxWordLength(maxMinusWordLength),
                keywords -> maxLengthMinusWord(maxMinusWordLength, keywords));
    }

    /*
     * Проверка максимального количества слов в минус-фразе, разделение по пробелам
     */
    public static Constraint<List<String>, Defect> maxWordsInKeywords() {
        return keywordListConstraint(MinusPhrasePredicates.maxWordsInKeywords(WORDS_MAX_COUNT),
                keywords -> maxCountWordsInKeyword(WORDS_MAX_COUNT, keywords));
    }

    /*
     * Проверка на допустимые символы: буквы английского, турецкого, казахского, русского, украинского, немецкого алфавитов,
     *  цифры, кавычки, квадратные скобки, знаки -, +, !, ', пробел
     */
    static Constraint<List<String>, Defect> allowedChars() {
        return keywordListConstraint(MinusPhrasePredicates.allowedChars(),
                MinusPhraseDefects::illegalMinusKeywordChars);
    }

    /*
     * Слово не может начинаться на точку или апостроф
     */
    static Constraint<List<String>, Defect> validWordFirstCharacter() {
        return keywordListConstraint(MinusPhrasePredicates.validWordFirstCharacter(),
                MinusPhraseDefects::invalidWordFirstCharacter);
    }

    /*
     * Проверка корректного использования точки: разрешена только в одном числе с точкой на МФ
     */
    static Constraint<List<String>, Defect> numberWithPoint() {
        return keywordListConstraint(MinusPhrasePredicates.numberWithPoint(),
                MinusPhraseDefects::invalidDot);
    }

    /*
     * Проверка корректного использования точки: разрешена только в одном числе с точкой на МФ
     */
    static Constraint<List<String>, Defect> separateDot() {
        return keywordListConstraint(MinusPhrasePredicates.separateDot(),
                MinusPhraseDefects::separateDot);
    }

    /*
     * Проверка неправильного использования кавычек
     */
    static Constraint<List<String>, Defect> wrappedQuotes() {
        return keywordListConstraint(CommonPhrasePredicates.validQuotes(),
                MinusPhraseDefects::wrongQuotes);
    }

    /*
     * Минус-фраза не может содержать несбалансированные квадратные скобки
     */
    static Constraint<List<String>, Defect> balancedSquareBrackets() {
        return keywordListConstraint(CommonPhrasePredicates.balancedSquareBrackets(),
                MinusPhraseDefects::imbalancedSquareBrackets);
    }

    /*
     * Минус-фраза не может содержать пустые или вложенные квадратные скобки
     */
    static Constraint<List<String>, Defect> noNestedOrEmptySquareBrackets() {
        return keywordListConstraint(CommonPhrasePredicates.noNestedOrEmptySquareBrackets(),
                MinusPhraseDefects::nestedOrEmptySquareBrackets);
    }

    /*
     * Проверка недопустимых операторов +-" внутри квадратных скобок
     */
    static Constraint<List<String>, Defect> operatorsInsideSquareBrackets() {
        return keywordListConstraint(MinusPhrasePredicates.operatorsInsideSquareBrackets(),
                MinusPhraseDefects::invalidOperatorsInsideSquareBrackets);
    }

    /*
     * Проверка использования знака "!" (проверка !!, использование в середине или в конце слова)
     */
    static Constraint<List<String>, Defect> exclamationMark() {
        return keywordListConstraint(CommonPhrasePredicates.validExclamationMark(),
                MinusPhraseDefects::invalidExclamationMark);
    }

    /*
     * Проверка использования знака "-" (проверка вхождения в начале или в конце слова)
     */
    static Constraint<List<String>, Defect> minusMark() {
        return keywordListConstraint(MinusPhrasePredicates.minusMark(),
                MinusPhraseDefects::invalidMinusMark);
    }

    /*
     * Проверка использования знака "+" (проверка ++, использование в середине или конце слова)
     */
    static Constraint<List<String>, Defect> plusMark() {
        return keywordListConstraint(CommonPhrasePredicates.validPlusMark(),
                MinusPhraseDefects::invalidPlusMark);
    }

    /*
     * Проверка сочетания спец. символов
     */
    static Constraint<List<String>, Defect> combinationSpecialSymbols() {
        return keywordListConstraint(MinusPhrasePredicates.combinationSpecialSymbols(),
                MinusPhraseDefects::invalidCombinationSpecialSymbols);
    }

    private static Constraint<List<String>, Defect> keywordListConstraint(
            Predicate<String> validElementPredicate,
            Function<List<String>, Defect> defectDefinitionProvider) {
        return keywords -> {
            if (keywords == null) {
                return null;
            }

            List<String> defectivePhrases = keywords.stream()
                    .filter(Objects::nonNull)
                    .filter(validElementPredicate.negate())
                    .collect(Collectors.toList());

            if (!defectivePhrases.isEmpty()) {
                return defectDefinitionProvider.apply(defectivePhrases);
            }

            return null;
        };
    }
}
