package ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.google.common.base.CharMatcher;

import ru.yandex.direct.utils.TextConstants;

import static ru.yandex.direct.core.entity.keyword.processing.KeywordProcessingUtils.getLengthWithoutSpecSymbolsAndSpaces;
import static ru.yandex.direct.core.entity.keyword.processing.KeywordProcessingUtils.removeSpecSymbols;
import static ru.yandex.direct.core.entity.keyword.processing.KeywordProcessingUtils.splitBySpaces;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.CommonPhrasePredicates.WORD_FINISH_REGEXP;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.CommonPhrasePredicates.WORD_START_REGEXP;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.CommonPhrasePredicates.noMatches;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

public class MinusPhrasePredicates {

    public static final String ALLOW_MINUS_KEYWORD_CHARS = TextConstants.LETTERS_AND_NUMBERS + "\"!+-[] .";
    private static final Pattern NUMBER_WITH_DOT_PATTERN = Pattern.compile("[0-9]+\\.[0-9]+");

    private static final Pattern INVALID_WORD_FIRST_CHAR_PATTERN = Pattern.compile(WORD_START_REGEXP + "[.\\']");

    private static final Pattern WRONG_MINUS_IN_WORD_PATTERN =
            Pattern.compile(WORD_START_REGEXP + "-|-" + WORD_FINISH_REGEXP);

    private static final Pattern ALLOWED_SYMBOLS_IN_BRACKETS_PATTERN = Pattern.compile("\\[[^+\"]+]");
    private static final Pattern WORDS_IN_BRACKETS_PATTERN = Pattern.compile("\\[([^]]+)]");

    private static final Pattern SEPARATE_DOT_PATTERN = Pattern.compile("(?:^|[\\s\"])\\.+(?:$|[\\s\"])");

    private static final Pattern ILLEGAL_SPEC_SYMBOLS_COMBINATION_PATTERN = Pattern.compile("[.\\[\\]\\-+!]{2,}");
    private static final Pattern LEGAL_EXCEPTION_SPEC_SYMBOLS_COMBINATION_PATTERN = Pattern.compile("\\[!");
    private static final CharMatcher ALLOW_MINUS_KEYWORD_CHAR_MATCHER = CharMatcher.anyOf(ALLOW_MINUS_KEYWORD_CHARS);

    /**
     * Проверка общей максимальной длины всех строк списка без учёта вхождения операторов
     */
    public static Predicate<List<String>> maxLengthKeywordsWithoutSpecSymbolsAndSpaces(int maxLength) {
        return keywords -> {
            keywords = filterList(keywords, Objects::nonNull);
            return getLengthWithoutSpecSymbolsAndSpaces(keywords) <= maxLength;
        };
    }

    /**
     * Проверка максимальной длины отдельных слов без операторов
     */
    public static Predicate<String> maxWordLength(int maxWordLength) {
        return keyword -> Arrays.stream(splitBySpaces(removeSpecSymbols(keyword)))
                .allMatch(word -> word.length() <= maxWordLength);
    }

    /**
     * Проверка максимального количества слов в минус-фразе
     */
    public static Predicate<String> maxWordsInKeywords(int wordsMaxCount) {
        return keyword -> splitBySpaces(keyword).length <= wordsMaxCount;
    }

    /**
     * Проверка отсутствия недопустимых символов
     */
    public static Predicate<String> allowedChars() {
        return ALLOW_MINUS_KEYWORD_CHAR_MATCHER::matchesAllOf;
    }

    /**
     * Правильное использование точки: разрешена только одна в одном числе на всю минус-фразу
     */
    public static Predicate<String> numberWithPoint() {
        return keyword -> !keyword.contains(".")
                || (noMatches(NUMBER_WITH_DOT_PATTERN, keyword) || keyword.split("\\.+").length <= 2);
    }

    /**
     * Проверка корректного использования точки: не может стоять отдельно
     */
    public static Predicate<String> separateDot() {
        return keyword -> !keyword.contains(".") || noMatches(SEPARATE_DOT_PATTERN, keyword);
    }

    /**
     * Проверка использования оператора "-"
     */
    public static Predicate<String> minusMark() {
        return keyword -> noMatches(WRONG_MINUS_IN_WORD_PATTERN, keyword);
    }

    // todo починить. пропускает невалидные кейсы при наличии валидных из-за !noMatches

    /**
     * Проверка допустимых операторов внутри квадратных скобок
     */
    public static Predicate<String> operatorsInsideSquareBrackets() {
        return keyword -> !keyword.contains("[")
                || (!noMatches(ALLOWED_SYMBOLS_IN_BRACKETS_PATTERN, keyword) && isValidInsideBrackets(keyword));
    }

    /**
     * Проверка правильности начала слова
     */
    public static Predicate<String> validWordFirstCharacter() {
        return keyword -> noMatches(INVALID_WORD_FIRST_CHAR_PATTERN, keyword);
    }

    // todo внимательно посмотреть на полурабочий предикат, кейсы которого покрыты другими предикатами

    /**
     * Проверка сочетания спец. символов
     */
    public static Predicate<String> combinationSpecialSymbols() {
        return keyword -> noMatches(ILLEGAL_SPEC_SYMBOLS_COMBINATION_PATTERN, keyword)
                || LEGAL_EXCEPTION_SPEC_SYMBOLS_COMBINATION_PATTERN.matcher(keyword).find();
    }

    private static boolean isValidInsideBrackets(String keyword) {
        Matcher matcher = WORDS_IN_BRACKETS_PATTERN.matcher(keyword);
        while (matcher.find()) {
            String word = matcher.group(1);
            if (word.matches("(?<!\\S)-")) {
                return false;
            }
        }

        return true;
    }
}
