package ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase;

import java.util.Arrays;
import java.util.List;

import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.allowedChars;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.balancedSquareBrackets;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.combinationSpecialSymbols;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.exclamationMark;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.maxLengthWordInKeyword;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.maxWordsInKeywords;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.minusMark;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.noNestedOrEmptySquareBrackets;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.numberWithPoint;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.operatorsInsideSquareBrackets;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.plusMark;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.separateDot;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.validWordFirstCharacter;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.wrappedQuotes;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

public class MinusPhraseValidator implements DefaultValidator<List<String>> {

    public enum ValidationMode {
        ONE_ERROR_PER_TYPE {
            @Override
            protected void check(
                    ListValidationBuilder<String, Defect> validationBuilder,
                    Constraint<List<String>, Defect> constraint) {
                validationBuilder.check(constraint, When.isValid());
            }
        },
        ONE_ERROR_PER_TYPE_AND_KEYWORD {
            @Override
            protected void check(
                    ListValidationBuilder<String, Defect> validationBuilder,
                    Constraint<List<String>, Defect> constraint) {
                validationBuilder.checkEach(
                        (Constraint<String, Defect>) s -> constraint.apply(singletonList(s)),
                        When.isValid());
            }
        };

        protected abstract void check(
                ListValidationBuilder<String, Defect> validationBuilder,
                Constraint<List<String>, Defect> constraint);
    }

    private final ValidationMode validationMode;

    public static MinusPhraseValidator minusKeywordIsValid(ValidationMode validationMode) {
        return new MinusPhraseValidator(validationMode);
    }

    private MinusPhraseValidator(ValidationMode validationMode) {
        this.validationMode = validationMode;
    }

    @Override
    public ValidationResult<List<String>, Defect> apply(List<String> keywords) {
        // NB: каждая из этих проверок может вызываться на список, состоящий из одной минус-фразы,
        // так что добавлять сюда какую-нибудь проверку на суммарную длину нельзя
        List<Constraint<List<String>, Defect>> constraints = Arrays.asList(
                allowedChars(),
                wrappedQuotes(),
                separateDot(),
                validWordFirstCharacter(),
                numberWithPoint(),
                maxWordsInKeywords(),
                maxLengthWordInKeyword(),
                balancedSquareBrackets(),
                noNestedOrEmptySquareBrackets(),
                operatorsInsideSquareBrackets(),
                exclamationMark(),
                minusMark(),
                plusMark(),
                combinationSpecialSymbols());

        ListValidationBuilder<String, Defect> validationBuilder =
                ListValidationBuilder.<String, Defect>of(keywords)
                        .checkEach(notNull())
                        .checkEach(notBlank(), When.isValid());

        for (Constraint<List<String>, Defect> constraint : constraints) {
            validationMode.check(validationBuilder, constraint);
        }

        return validationBuilder.getResult();
    }

}
