package ru.yandex.direct.core.entity.landing

import java.util.regex.Pattern
import org.apache.commons.validator.routines.EmailValidator
import org.slf4j.LoggerFactory
import ru.yandex.direct.core.entity.landing.model.BizContactType

private val logger = LoggerFactory.getLogger(BizContactValueFormatter::class.java)

private val PHONE_NON_DIGIT_SYMBOLS = Pattern.compile("[\\s-()+]")
private const val MIN_PHONE_LENGTH = 9
private const val MAX_PHONE_LENGTH = 15
private const val PHONE_REGEX = "\\d{$MIN_PHONE_LENGTH,$MAX_PHONE_LENGTH}"

private const val URL_PREFIX = "(?:https?://)?(?:www\\.)"

private val EMAIL_VALIDATOR = EmailValidator.getInstance()

private val CONTACT_VALUE_PATTERN = mapOf(
    BizContactType.TELEGRAM to Pattern.compile("@?(\\w+)"),
    BizContactType.VKONTAKTE to Pattern.compile("@?(\\w+)"),
    BizContactType.WHATSAPP to Pattern.compile("([\\d+\\-\\s()]+)"),
    BizContactType.VIBER to Pattern.compile("([\\d+\\-\\s()]+)"),
)

private val CONTACT_URL_TEMPLATE = mapOf(
    BizContactType.TELEGRAM to
        listOf(
            Pattern.compile("($URL_PREFIX?t\\.me/)?([A-Za-z0-9_+]+)/?") to "https://t.me/",
            Pattern.compile("($URL_PREFIX?t\\.me/joinchat/)?([A-Za-z0-9_+]+)/?") to "https://t.me/joinchat/",
            Pattern.compile("($URL_PREFIX?telegram\\.me/)?([A-Za-z0-9_+]+)/?") to "https://telegram.me/",
            Pattern.compile("($URL_PREFIX?telegram\\.me/joinchat/)?([A-Za-z0-9_+]+)/?") to "https://telegram.me/joinchat/",
        ),
    BizContactType.VKONTAKTE to
        listOf(
            Pattern.compile("($URL_PREFIX?vk\\.com/)?(\\w+)/?") to "https://vk.com/",
            Pattern.compile("($URL_PREFIX?vk\\.me/)?(\\w+)/?") to "https://vk.me/",
            Pattern.compile("($URL_PREFIX?vk\\.cc/)?(\\w+)/?") to "https://vk.cc/",
        ),
    BizContactType.WHATSAPP to
        listOf(
            Pattern.compile("($URL_PREFIX?wa\\.me/)?($PHONE_REGEX)/?")
                to "https://wa.me/",

            Pattern.compile("($URL_PREFIX?chat\\.whatsapp\\.com/)(\\w+)/?")
                to "https://chat.whatsapp.com/",

            Pattern.compile("($URL_PREFIX?chat\\.whatsapp\\.com/invite/)(\\w+)/?")
                to "https://chat.whatsapp.com/invite/",

            Pattern.compile("($URL_PREFIX?api\\.whatsapp\\.com/send\\?phone=)($PHONE_REGEX)/?")
                to "https://api.whatsapp.com/send?phone=",
        ),
    BizContactType.VIBER to
        listOf(
            Pattern.compile("($URL_PREFIX?viber\\.click/)?($PHONE_REGEX)/?") to "https://viber.click/"
        ),

    )

class BizContactValueFormatter {
    companion object {
        /**
         * @return отформатированный контакт или `null`, если `value` невалидного формата для типа `type`
         */
        fun format(value: String, type: BizContactType): String? {
            return when (type) {
                BizContactType.EMAIL ->
                    if (isValidEmail(value)) {
                        value
                    } else {
                        null
                    }
                BizContactType.PHONE -> formatPhone(value)
                BizContactType.VKONTAKTE, BizContactType.TELEGRAM, BizContactType.VIBER, BizContactType.WHATSAPP ->
                    formatMessenger(value, type)
            }
        }

        /**
         * Возвращает телефон без спец символов, если на вход пришел телефон, иначе `null`
         */
        fun formatPhone(value: String): String? {
            val maybePhone = PHONE_NON_DIGIT_SYMBOLS.matcher(value).replaceAll("")
            return if (maybePhone.length in MIN_PHONE_LENGTH..MAX_PHONE_LENGTH && maybePhone.all { it.isDigit() }) {
                maybePhone
            } else {
                null
            }
        }

        fun isValidEmail(value: String) = EMAIL_VALIDATOR.isValid(value)

        fun formatMessenger(value: String?, type: BizContactType): String? {
            if (value.isNullOrEmpty()) {
                return null
            }
            try {
                val rawLogin = prepareIfPhone(value)
                val uri = parseUri(rawLogin, type) ?: return null
                val login = parseLogin(rawLogin, type) ?: return null
                return uri + "" + login
            } catch (e: RuntimeException) {
                logger.error("Failed to format $value as $type value")
                return null
            }
        }

        private fun parseLogin(value: String, type: BizContactType): String? {
            val loginMatcher = CONTACT_VALUE_PATTERN[type]!!.matcher(value)
            if (loginMatcher.matches()) {
                val login = loginMatcher.group(1)
                return if (type == BizContactType.VIBER || type == BizContactType.WHATSAPP) {
                    if (login.startsWith("8")) login.replace("8", "7") else login
                } else {
                    login
                }
            }
            val regex = CONTACT_URL_TEMPLATE[type]!!.find { it.first.matcher(value).matches() } ?: return null
            val urlMatcher = regex.first.matcher(value)
            if (urlMatcher.matches()) {
                return urlMatcher.group(2)
            }
            return null
        }

        private fun parseUri(value: String, type: BizContactType): String? {
            CONTACT_URL_TEMPLATE[type]!!.forEach {
                if (it.first.matcher(value).matches()) {
                    return it.second
                }
            }
            return null
        }

        private fun prepareIfPhone(value: String) = formatPhone(value) ?: value
    }
}
