package ru.yandex.direct.core.entity.landing

import maps_adv.geosmb.landlord.proto.internal.landing_details.LandingDetailsOuterClass
import maps_adv.geosmb.landlord.proto.organization_details.OrganizationDetailsOuterClass
import org.springframework.stereotype.Service
import ru.yandex.direct.bvm.client.BvmClient
import ru.yandex.direct.core.entity.landing.converter.toBlocksOptions
import ru.yandex.direct.core.entity.landing.converter.toContacts
import ru.yandex.direct.core.entity.landing.converter.toCta
import ru.yandex.direct.core.entity.landing.converter.toImageUrl
import ru.yandex.direct.core.entity.landing.converter.toPreferences
import ru.yandex.direct.core.entity.landing.model.BizContact
import ru.yandex.direct.core.entity.landing.model.BizContactType
import ru.yandex.direct.core.entity.landing.model.BizLanding
import ru.yandex.direct.core.entity.organizations.service.OrganizationService
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.core.validation.defects.RightsDefects
import ru.yandex.direct.landlord.client.LandlordClient
import ru.yandex.direct.organizations.swagger.model.CompanyEmail
import ru.yandex.direct.organizations.swagger.model.CompanyPhone
import ru.yandex.direct.organizations.swagger.model.CompanyUrl
import ru.yandex.direct.organizations.swagger.model.CompanyUrlType
import ru.yandex.direct.organizations.swagger.model.NewCompany
import ru.yandex.direct.organizations.swagger.model.UpdateOrganizationRequest
import ru.yandex.direct.result.Result
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.geosmb.bvm.model.FetchBizIdProtobuf

private const val UNDEFINED_BIZ_ID = 0L

@Service
class BizLandingService(
    private val landlordClient: LandlordClient,
    private val bvmClient: BvmClient,
    private val bizLandingValidationService: BizLandingValidationService,
    private val organizationService: OrganizationService,
    private val grutApiService: GrutApiService
) {

    fun getBizLanding(operatorUid: Long, id: Long?, permalink: Long?): BizLanding? {
        var landingId = id ?: getBizIdOrNull(permalink!!)
        if (landingId == null) {
            val hasRights = bizLandingValidationService.operatorCanEditPermalink(operatorUid, permalink!!)
            if (!hasRights) {
                return null
            }
            landingId = getBizIdOrCreate(permalink)
            landlordClient.generateLandingData(landingId) ?: return null
        }
        val output = landlordClient.showLandingDetails(landingId) ?: return null
        return toBizLanding(landingId, output.slug, output.landingDetails)
    }

    /**
     * @return идентификатор обновленного лендинга или `null`, если этот лендинг не удалось обновить
     */
    fun updateBizLanding(operatorUid: Long, landing: BizLanding): Result<Long> {
        val hasRights = bizLandingValidationService.operatorCanEditPermalink(operatorUid, landing.permalink)
        if (!hasRights) {
            return Result.broken(ValidationResult.failed(landing.id, RightsDefects.noRights()))
        }

        val (vr, formattedContacts) = bizLandingValidationService.validateAndFormatContacts(landing)
        if (vr.hasAnyErrors()) {
            return Result.broken(vr)
        }
        landing.contacts = formattedContacts

        return updateBizLanding(landing)
    }

    fun addBizLanding(operatorUid: Long, landing: BizLanding): Result<Long> {
        val hasRights = bizLandingValidationService.operatorCanEditPermalink(operatorUid, landing.permalink)
        if (!hasRights) {
            return Result.broken(ValidationResult.failed(landing.id, RightsDefects.noRights()))
        }

        val (vr, formattedContacts) = bizLandingValidationService.validateAndFormatContacts(landing)
        if (vr.hasAnyErrors()) {
            return Result.broken(vr)
        }
        landing.contacts = formattedContacts

        val bizId = getBizIdOrCreate(landing.permalink)
        landlordClient.generateLandingData(bizId)
            ?: return Result.broken(ValidationResult.failed(landing.permalink, CommonDefects.invalidValue()))
        landing.id = bizId
        return updateBizLanding(landing)
    }

    private fun toBizLanding(
        id: Long?,
        slug: String,
        outputDetails: LandingDetailsOuterClass.LandingDetails
    ) = BizLanding().apply {
        this.id = id
        permalink = outputDetails.contacts.geo.permalink.toLong()
        url = landlordClient.buildUrlFromSlug(slug)
        name = outputDetails.name
        description = outputDetails.description
        categories = outputDetails.categoriesList
        logoUrl = outputDetails.logo.templateUrl
        coverUrl = outputDetails.cover.templateUrl
        cta = toCta(outputDetails.preferences)
        contacts = toContacts(outputDetails.contacts)
    }

    private fun updateBizLanding(landing: BizLanding): Result<Long> {
        val oldLanding = landlordClient.showLandingDetails(landing.id)!!.landingDetails
        val input = buildLandingDetailsInput(oldLanding, landing)

        val orgRequest = buildUpdateOrgRequest(landing.contacts)
        organizationService.updateOrganization(landing.permalink, orgRequest)

        val id = landing.id
        val stableInput = buildUpdateInput(id, input, OrganizationDetailsOuterClass.LandingVersion.STABLE)
        val output = landlordClient.editLandingDetails(stableInput)
            ?: return Result.broken(ValidationResult.failed(id, CommonDefects.invalidValue()))

        // Легаси landlord-а, которое требует, чтобы было отправлено 2 запроса с разными version
        val unstableInput = buildUpdateInput(id, input, OrganizationDetailsOuterClass.LandingVersion.UNSTABLE)
        landlordClient.editLandingDetails(unstableInput)

        landing.url = landlordClient.buildUrlFromSlug(output.slug)
        grutApiService.bizLandingGrutApi.createOrUpdateBizLanding(landing)

        return Result.successful(id)
    }

    /**
     * При дописывании полей нужно обязательно указать их в
     * {@link LandingDetailsInputUpdateFieldsTest#DIRECT_INPUT_FIELDS}
     */
    private fun buildLandingDetailsInput(
        oldLanding: LandingDetailsOuterClass.LandingDetails,
        landing: BizLanding
    ): LandingDetailsOuterClass.LandingDetailsInput {
        return LandingDetailsOuterClass.LandingDetailsInput.newBuilder().apply {
            name = landing.name
            addAllCategories(landing.categories)
            landing.description?.let { description = it }
            landing.logoUrl?.let { logo = toImageUrl(it) }
            landing.coverUrl?.let { cover = toImageUrl(it) }
            preferences = toPreferences(oldLanding.preferences, landing.cta)
            contacts = toContacts(oldLanding.contacts, landing.contacts)
            extras = oldLanding.extras
            blocksOptions = toBlocksOptions(oldLanding.blocksOptions)
        }.build()
    }

    private fun buildUpdateOrgRequest(contacts: List<BizContact>): UpdateOrganizationRequest {
        val contactsByType = contacts.groupBy { it.type }
        return UpdateOrganizationRequest(
            newCompany = NewCompany(
                emails = contactsByType[BizContactType.EMAIL]?.map {
                    CompanyEmail().apply { this.value = it.value }
                } ?: emptyList(),
                phones = contactsByType[BizContactType.PHONE]?.map {
                    CompanyPhone().apply { formatted = it.value }
                } ?: emptyList(),
                urls = listOfNotNull(
                    toCompanyUrl(contactsByType, BizContactType.VKONTAKTE),
                    toCompanyUrl(contactsByType, BizContactType.TELEGRAM),
                    toCompanyUrl(contactsByType, BizContactType.VIBER),
                    toCompanyUrl(contactsByType, BizContactType.WHATSAPP),
                )
            )
        )
    }

    private fun toCompanyUrl(contactsByType: Map<BizContactType, List<BizContact>>, type: BizContactType) =
        contactsByType[type]?.firstOrNull()?.let {
            CompanyUrl().apply {
                this.type = CompanyUrlType.SOCIAL
                this.value = it.value
            }
        }

    private fun buildUpdateInput(
        id: Long,
        landingDetails: LandingDetailsOuterClass.LandingDetailsInput,
        version: OrganizationDetailsOuterClass.LandingVersion
    ): LandingDetailsOuterClass.EditLandingDetailsInput {

        return LandingDetailsOuterClass.EditLandingDetailsInput.newBuilder()
            .setBizId(id)
            .setVersion(version)
            .setLandingDetails(landingDetails)
            .build()
    }

    private fun getBizIdOrCreate(permalink: Long): Long {
        val input = FetchBizIdProtobuf.FetchBizIdInput.newBuilder().setPermalink(permalink).build()
        val fetchBizId = bvmClient.fetchBizId(input)
        return fetchBizId.bizId
    }

    private fun getBizIdOrNull(permalink: Long): Long? {
        val input = FetchBizIdProtobuf.FetchNoCreateBizIdInput.newBuilder().setPermalink(permalink).build()
        val fetchBizId = bvmClient.fetchNoCreateBizId(input)
        return if (fetchBizId.bizId == UNDEFINED_BIZ_ID) null else fetchBizId.bizId
    }
}
