package ru.yandex.direct.core.entity.landing

import org.springframework.stereotype.Service
import ru.yandex.common.util.collections.IdentityHashSet
import ru.yandex.direct.core.entity.landing.model.BizContact
import ru.yandex.direct.core.entity.landing.model.BizLanding
import ru.yandex.direct.core.entity.organizations.service.OrganizationService
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.D
import ru.yandex.direct.validation.util.validateObject

@Service
class BizLandingValidationService(
    private val organizationService: OrganizationService,
) {
    fun operatorCanEditPermalink(operatorUid: Long, permalink: Long) =
        organizationService.operatorEditableOrganizations(operatorUid, listOf(permalink)).contains(permalink)

    /**
     * @return пара: результат валидации контактов и переформатированный набор контактов
     */
    fun validateAndFormatContacts(landing: BizLanding): Pair<ValidationResult<BizLanding, D>, List<BizContact>> {
        val formattedContacts = mutableListOf<BizContact>()
        val invalidContacts = IdentityHashSet<BizContact>()
        for (contact in landing.contacts) {
            val contactOrNull = getFormattedContactOrNull(contact)
            if (contactOrNull == null) {
                invalidContacts.add(contact)
            } else {
                formattedContacts.add(contactOrNull)
            }
        }

        val result = validateObject(landing) {
            list(landing.contacts, BizLanding.CONTACTS.name())
                .checkEach(CommonConstraints.notInSet(invalidContacts), CommonDefects.invalidValue())
        }
        return Pair(result, formattedContacts)
    }

    private fun getFormattedContactOrNull(contact: BizContact): BizContact? {
        if (contact.value == null) {
            return contact
        }
        val formatted = BizContactValueFormatter.format(contact.value, contact.type)
        if (formatted == null) {
            // Невалидное значение
            return null
        }
        return BizContact().withType(contact.type).withValue(formatted)
    }
}
