package ru.yandex.direct.core.entity.lockobject.repository;

import java.time.Duration;
import java.time.LocalDateTime;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.impl.DSL;
import org.jooq.types.ULong;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.model.LockObjectInfo;
import ru.yandex.direct.core.entity.model.LockObjectType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapper.write.WriterFunction1;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromLongFieldToBoolean;
import static ru.yandex.direct.dbschema.ppcdict.tables.LockObject.LOCK_OBJECT;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromPropertyToField;

@Repository
@ParametersAreNonnullByDefault
public class LockObjectRepository {

    private static final Field<Long> IS_LOCKED =
            DSL.iif(LOCK_OBJECT.LOCK_TIME.ge(DSL.currentLocalDateTime()), 1L, 0L).as("is_locked");
    private static final WriterFunction1<Duration, Field<LocalDateTime>> LOCK_TIME_WRITER_FUNCTION =
            duration -> DSL.field(String.format("NOW() + interval %d second", duration.getSeconds()), LocalDateTime.class);

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<LockObjectInfo> jooqMapper;

    @Autowired
    public LockObjectRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.jooqMapper = JooqMapperWithSupplierBuilder.builder(LockObjectInfo::new)
                .map(property(LockObjectInfo.OBJECT_ID, LOCK_OBJECT.OBJECT_ID))
                .map(convertibleProperty(LockObjectInfo.OBJECT_TYPE, LOCK_OBJECT.OBJECT_TYPE,
                        LockObjectType::fromSource, LockObjectType::toSource))
                .map(convertibleProperty(LockObjectInfo.HALF_MD5_HASH, LOCK_OBJECT.HALF_MD5_HASH,
                        ULong::toBigInteger, ULong::valueOf))
                .readProperty(LockObjectInfo.LOCK_ID, fromField(LOCK_OBJECT.ID))
                .readProperty(LockObjectInfo.IS_LOCKED, fromLongFieldToBoolean(IS_LOCKED))
                .writeField(LOCK_OBJECT.LOCK_TIME, fromPropertyToField(LockObjectInfo.DURATION)
                        .by(LOCK_TIME_WRITER_FUNCTION))
                .build();
    }

    public boolean addLockObject(LockObjectInfo lockObjectInfo) {
        var insertHelper = new InsertHelper<>(dslContextProvider.ppcdict(), LOCK_OBJECT)
                .add(jooqMapper, lockObjectInfo)
                .onDuplicateKeyIgnore();

        return insertHelper.execute() > 0;
    }

    @Nullable
    public LockObjectInfo getLockObject(Long objectId, LockObjectType objectType, @Nullable Long lockId) {
        return dslContextProvider.ppcdict()
                .select(jooqMapper.getFieldsToRead())
                .from(LOCK_OBJECT)
                .where(getSearchCondition(objectId, objectType, lockId))
                .fetchOne(jooqMapper::fromDb);
    }

    public boolean deleteLockObject(Long objectId, LockObjectType objectType, @Nullable Long lockId) {
        return dslContextProvider.ppcdict()
                .deleteFrom(LOCK_OBJECT)
                .where(getSearchCondition(objectId, objectType, lockId))
                .execute() > 0;
    }

    private static Condition getSearchCondition(Long objectId, LockObjectType objectType, @Nullable Long lockId) {
        var conditions = LOCK_OBJECT.OBJECT_ID.eq(objectId)
                .and(LOCK_OBJECT.OBJECT_TYPE.eq(LockObjectType.toSource(objectType)));
        if (lockId != null) {
            return conditions.and(LOCK_OBJECT.ID.eq(lockId));
        }

        return conditions;
    }
}
