package ru.yandex.direct.core.entity.mailnotification.model;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.currency.CurrencyCode;

import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.NumberUtils.isZero;

/**
 * Событие изменения кампании
 */
@ParametersAreNonnullByDefault
public class CampaignEvent<T> extends GenericEvent {

    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("dd.MM.yyyy");

    private final EventType eventType;
    private final EventCampaignParams<T> params;

    private CampaignEvent(Long campaignId, Long uid, Long ownerUid, EventType eventType,
                          EventCampaignParams<T> params) {
        super(campaignId, uid, ownerUid, campaignId);
        this.eventType = eventType;
        this.params = params;
    }

    /**
     * @return {@link CampaignEvent} &ndash; событие изменение даты начала кампании
     */
    public static CampaignEvent<String> changedStartDateEvent(Long operatorUid, Long ownerUid, Long campaignId,
                                                              LocalDate oldValue, LocalDate newValue) {
        var params = new EventCampaignParams<>(dateFormat(oldValue), dateFormat(newValue));
        return new CampaignEvent<>(campaignId, operatorUid, ownerUid, EventType.C_START, params);
    }

    /**
     * @return {@link CampaignEvent} &ndash; событие изменение даты окончания кампании
     */
    public static CampaignEvent<String> changedEndDateEvent(Long operatorUid, Long ownerUid, Long campaignId,
                                                            String textForNullValue,
                                                            @Nullable LocalDate oldValue,
                                                            @Nullable LocalDate newValue) {
        String oldVal = nvl(dateFormat(oldValue), textForNullValue);
        String newVal = nvl(dateFormat(newValue), textForNullValue);
        var params = new EventCampaignParams<>(oldVal, newVal);
        return new CampaignEvent<>(campaignId, operatorUid, ownerUid, EventType.C_FINISH, params);
    }

    /**
     * @return {@link CampaignEvent} &ndash; событие изменение дневного бюджета кампании
     */
    public static CampaignEvent<String> changedDayBudgetEvent(Long operatorUid, Long ownerUid, Long campaignId,
                                                              CurrencyCode currency,
                                                              BigDecimal oldValue, BigDecimal newValue) {
        var params = new EventCampaignParams<>(toSumWithCurrency(oldValue, currency),
                toSumWithCurrency(newValue, currency));
        return new CampaignEvent<>(campaignId, operatorUid, ownerUid, EventType.C_DAY_BUDGET_MULTICURRENCY, params);
    }

    /**
     * @return {@link CampaignEvent} &ndash; событие остановка кампании
     */
    public static CampaignEvent<String> campaignStopped(Long operatorUid, Long ownerUid, Long campaignId) {
        var params = new EventCampaignParams<>("start", "stop");
        return new CampaignEvent<>(campaignId, operatorUid, ownerUid, EventType.C_STATUS, params);
    }


    private static String toSumWithCurrency(BigDecimal value, CurrencyCode currency) {
        return isZero(value) ? "0" : String.format("%s:%s", value.toString(), currency.name());
    }

    @Nullable
    private static String dateFormat(@Nullable LocalDate value) {
        return value == null ? null : value.format(DATE_FORMATTER);
    }

    @Override
    protected ObjectType getObjectType() {
        return ObjectType.CAMP;
    }

    @Override
    protected EventType getEventType() {
        return eventType;
    }

    @Override
    protected EventCampaignParams getJsonData() {
        return params;
    }
}
