package ru.yandex.direct.core.entity.mailnotification.model;

import java.math.BigDecimal;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.keyword.container.KeywordDeleteInfo;

import static ru.yandex.direct.core.entity.mailnotification.model.EventType.PH_CHANGE;

/**
 * Событие изменения ставки для фразы
 */
@ParametersAreNonnullByDefault
public class KeywordEvent<T> extends GenericEvent {

    private ObjectType objectType;
    private EventType eventType;
    private EventAdGroupParams<T> adGroupParams;

    private KeywordEvent(Long campaignId, Long uid, Long ownerUid, Long adGroupId,
                         ObjectType objectType, EventType eventType, EventAdGroupParams<T> adGroupParams) {
        super(campaignId, uid, ownerUid, adGroupId);
        this.objectType = objectType;
        this.eventType = eventType;
        this.adGroupParams = adGroupParams;
    }

    /**
     * @return {@link KeywordEvent} &ndash; событие изменение ставки для фразы на поиске
     */
    public static KeywordEvent<BigDecimal> changedSearchPriceEvent(Long operatorUid, Long ownerUid, AdGroup adGroup,
                                                                   BigDecimal oldValue, BigDecimal newValue) {
        return changedSearchPriceEvent(operatorUid, ownerUid, adGroup.getCampaignId(), adGroup.getId(),
                adGroup.getName(), oldValue, newValue);
    }

    /**
     * @return {@link KeywordEvent} &ndash; событие изменение ставки для фразы на сети
     */
    public static KeywordEvent<BigDecimal> changedContextPriceEvent(Long operatorUid, Long ownerUid, AdGroup adGroup,
                                                                    BigDecimal oldValue, BigDecimal newValue) {
        return changedContextPriceEvent(operatorUid, ownerUid, adGroup.getCampaignId(), adGroup.getId(),
                adGroup.getName(), oldValue, newValue);
    }

    /**
     * @return {@link KeywordEvent} &ndash; событие изменение ставки для фразы на поиске
     */
    public static KeywordEvent<BigDecimal> changedSearchPriceEvent(Long operatorUid, Long ownerUid,
                                                                   Long campaignId, Long adGroupId, String adGroupName, BigDecimal oldValue, BigDecimal newValue) {
        EventAdGroupPriceParams adGroupParams =
                new EventAdGroupPriceParams(adGroupId, adGroupName, oldValue, newValue);
        return new KeywordEvent<>(campaignId, operatorUid, ownerUid, adGroupId,
                ObjectType.PHRASE, EventType.PH_PRICE, adGroupParams);
    }

    /**
     * @return {@link KeywordEvent} &ndash; событие изменение ставки для фразы на сети
     */
    public static KeywordEvent<BigDecimal> changedContextPriceEvent(Long operatorUid, Long ownerUid,
                                                                    Long campaignId, Long adGroupId, String adGroupName, BigDecimal oldValue, BigDecimal newValue) {
        EventAdGroupPriceParams adGroupParams =
                new EventAdGroupPriceParams(adGroupId, adGroupName, oldValue, newValue);
        return new KeywordEvent<>(campaignId, operatorUid, ownerUid, adGroupId,
                ObjectType.PHRASE, EventType.PH_PRICE_CTX, adGroupParams);
    }

    /**
     * @return {@link KeywordEvent} &ndash; событие добавления фразы в группу.
     */
    public static KeywordEvent<String> addedKeywordEvent(Long operatorUid, Long ownerUid,
                                                         Long campaignId, Long adGroupId, String adGroupName, String phrase) {
        EventAdGroupParams<String> adGroupParams =
                new EventAdGroupParams<>(adGroupId, adGroupName, "", phrase);
        return new KeywordEvent<>(campaignId, operatorUid, ownerUid, adGroupId,
                ObjectType.ADGROUP, EventType.ADGR_WORD, adGroupParams);
    }

    /**
     * @return {@link KeywordEvent} &ndash; событие обновления текста фразы в группе.
     */
    public static KeywordEvent<String> changedKeywordPhraseEvent(Long operatorUid, Long ownerUid,
                                                                 Long campaignId, Long adGroupId, String adGroupName, String oldPhrase, String newPhrase) {
        EventAdGroupParams<String> adGroupParams =
                new EventAdGroupParams<>(adGroupId, adGroupName, oldPhrase, newPhrase);
        return new KeywordEvent<>(campaignId, operatorUid, ownerUid, adGroupId,
                ObjectType.PHRASE, EventType.PH_CHANGE, adGroupParams);
    }

    public static KeywordEvent<String> deletedKeywordEvent(long operatorUid, Long adGroupId,
                                                           KeywordDeleteInfo deleteInfo, String phrase) {
        EventAdGroupParams<String> eventAdGroupParams = new EventAdGroupParams<>(deleteInfo.getAdGroupId(),
                deleteInfo.getAdGroupName(), phrase, "");
        return new KeywordEvent<>(deleteInfo.getCampaignId(), operatorUid, deleteInfo.getOwnerUid(),
                adGroupId, ObjectType.PHRASE, PH_CHANGE, eventAdGroupParams);
    }

    @Override
    protected ObjectType getObjectType() {
        return objectType;
    }

    @Override
    protected EventType getEventType() {
        return eventType;
    }

    @Override
    protected EventAdGroupParams getJsonData() {
        return adGroupParams;
    }
}
