package ru.yandex.direct.core.entity.mailnotification.repository;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.mailnotification.model.EventType;
import ru.yandex.direct.core.entity.mailnotification.model.MailNotificationEvent;
import ru.yandex.direct.core.entity.mailnotification.model.ObjectType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppc.tables.Events.EVENTS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class MailNotificationEventRepository {
    private static final int INSERT_CHUNK_SIZE = 5_000;

    private final DslContextProvider dslContextProvider;

    public final JooqMapperWithSupplier<MailNotificationEvent> jooqMapper;

    @Autowired
    public MailNotificationEventRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        jooqMapper = JooqMapperWithSupplierBuilder.builder(MailNotificationEvent::new)
                .map(property(MailNotificationEvent.OBJECT_ID, EVENTS.OBJECTID))
                .map(property(MailNotificationEvent.OPERATOR_UID, EVENTS.UID))
                .map(property(MailNotificationEvent.OWNER_UID, EVENTS.OBJECTUID))
                .map(property(MailNotificationEvent.CAMPAIGN_ID, EVENTS.CID))
                .map(convertibleProperty(MailNotificationEvent.OBJECT_TYPE, EVENTS.EVENTOBJECT,
                        ObjectType::fromSource, ObjectType::toSource))
                .map(convertibleProperty(MailNotificationEvent.EVENT_TYPE, EVENTS.EVENTTYPE,
                        EventType::fromSource, EventType::toSource))
                .map(property(MailNotificationEvent.JSON_DATA, EVENTS.JSON_DATA))
                .build();
    }

    /**
     * @see #addEvents(DSLContext, List)
     */
    public int addEvents(int shard, List<MailNotificationEvent> events) {
        return addEvents(dslContextProvider.ppc(shard), events);
    }

    /**
     * Добавляет заявки на отправку писем в таблицу {@link ru.yandex.direct.dbschema.ppc.tables.Events#EVENTS EVENTS}
     *
     * @return количество добавленных строк
     */
    public int addEvents(DSLContext dslContext, List<MailNotificationEvent> events) {
        return Lists.partition(events, INSERT_CHUNK_SIZE)
                .stream()
                .mapToInt(chunk -> new InsertHelper<>(dslContext, EVENTS)
                        .addAll(jooqMapper, chunk)
                        .executeIfRecordsAdded()
                )
                .sum();
    }
}
