package ru.yandex.direct.core.entity.mailnotification.service;

import java.util.Collection;
import java.util.List;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.mailnotification.model.GenericEvent;
import ru.yandex.direct.core.entity.mailnotification.model.MailNotificationEvent;
import ru.yandex.direct.core.entity.mailnotification.repository.MailNotificationEventRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Аналог Perl-модуля {@code MailNotification.pm}
 */
@Service
@ParametersAreNonnullByDefault
public class MailNotificationEventService {
    private static final Logger logger = LoggerFactory.getLogger(MailNotificationEventService.class);

    private final ShardHelper shardHelper;
    private final MailNotificationEventRepository mailNotificationEventRepository;
    private final RbacService rbacService;

    @Autowired
    public MailNotificationEventService(ShardHelper shardHelper,
                                        MailNotificationEventRepository mailNotificationEventRepository,
                                        RbacService rbacService) {
        this.shardHelper = shardHelper;
        this.mailNotificationEventRepository = mailNotificationEventRepository;
        this.rbacService = rbacService;
    }

    /**
     * Добавляет события {@code events} в очередь на отправку уведомлений пользователям.
     */
    public void queueEvents(Long operatorUid, ClientId clientId, Collection<? extends GenericEvent> events) {
        queueEvents(operatorUid, events, notificationEvents -> {
            int shard = shardHelper.getShardByClientIdStrictly(clientId);
            return mailNotificationEventRepository.addEvents(shard, notificationEvents);
        });
    }

    /**
     * Добавляет события {@code events} в очередь на отправку уведомлений пользователям в рамках переданного dslContext
     */
    public void queueEvents(DSLContext dslContext, Long operatorUid, Collection<? extends GenericEvent> events) {
        queueEvents(operatorUid, events,
                notificationEvents -> mailNotificationEventRepository.addEvents(dslContext, notificationEvents));
    }

    private void queueEvents(Long operatorUid, Collection<? extends GenericEvent> events,
                             Function<List<MailNotificationEvent>, Integer> addEventsFunction) {
        //посылаем письма только, если изменение делает менеджер
        RbacRole operatorRole = rbacService.getUidRole(operatorUid);
        if (operatorRole != RbacRole.MANAGER) {
            if (rbacService.getManagersOfUser(operatorUid).isEmpty()) {
                return;
            }
        }

        List<MailNotificationEvent> notificationEvents = mapList(events, GenericEvent::toMailNotificationEvent);
        int addedEventsCount = addEventsFunction.apply(notificationEvents);
        logger.debug("{} notification events added", addedEventsCount);
    }

}

