package ru.yandex.direct.core.entity.markupcondition.repository;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.pricepackage.model.MarkupCondition;
import ru.yandex.direct.core.entity.pricepackage.model.MarkupConditionTargeting;
import ru.yandex.direct.core.entity.pricepackage.model.MarkupConditionsOperator;
import ru.yandex.direct.dbschema.ppcdict.tables.records.MarkupConditionsRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.dbschema.ppcdict.tables.MarkupConditions.MARKUP_CONDITIONS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class MarkupConditionRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<MarkupCondition> markupConditionMapper;

    @Autowired
    public MarkupConditionRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.markupConditionMapper = createMarkupConditionMapper();
    }

    /**
     * Возвращает список всех условий наценок за таргеты
     */
    public List<MarkupCondition> getAllMarkupConditions() {
        return dslContextProvider.ppcdict()
                .select(markupConditionMapper.getFieldsToRead())
                .from(MARKUP_CONDITIONS)
                .fetch(markupConditionMapper::fromDb);
    }

    /**
     * Добавление новых и изменение существующих записей в таблице MARKUP_CONDITIONS
     */
    public int updateMarkupConditions(Collection<MarkupCondition> markupConditions) {
        if (markupConditions.isEmpty()) {
            return 0;
        }

        InsertHelper<MarkupConditionsRecord> helper = new InsertHelper<>(dslContextProvider.ppcdict(), MARKUP_CONDITIONS);
        helper.addAll(markupConditionMapper, markupConditions);

        if (helper.hasAddedRecords()) {
            helper.onDuplicateKeyUpdate()
                    .set(MARKUP_CONDITIONS.CONDITION_ID, MySQLDSL.values(MARKUP_CONDITIONS.CONDITION_ID))
                    .set(MARKUP_CONDITIONS.IS_ARCHIVED, MySQLDSL.values(MARKUP_CONDITIONS.IS_ARCHIVED))
                    .set(MARKUP_CONDITIONS.GROUP_KEY, MySQLDSL.values(MARKUP_CONDITIONS.GROUP_KEY))
                    .set(MARKUP_CONDITIONS.PRIORITY, MySQLDSL.values(MARKUP_CONDITIONS.PRIORITY))
                    .set(MARKUP_CONDITIONS.OPERATOR, MySQLDSL.values(MARKUP_CONDITIONS.OPERATOR))
                    .set(MARKUP_CONDITIONS.TARGETINGS_CUSTOM, MySQLDSL.values(MARKUP_CONDITIONS.TARGETINGS_CUSTOM))
                    .set(MARKUP_CONDITIONS.DESCRIPTION, MySQLDSL.values(MARKUP_CONDITIONS.DESCRIPTION));
        }
        return helper.executeIfRecordsAdded();
    }

    public List<Long> addMarkupConditions(List<MarkupCondition> markupConditions) {
        DSLContext dslContext = dslContextProvider.ppcdict();

        return InsertHelper.addModelsAndReturnIds(dslContext, MARKUP_CONDITIONS,
                markupConditionMapper, markupConditions, MARKUP_CONDITIONS.CONDITION_ID);
    }

    /**
     * Поиск по префиксу названия
     */
    public List<MarkupCondition> findMarkupConditionsByNamePrefix(String markupConditionDescriptionPrefix, int limit) {
        return dslContextProvider.ppcdict()
                .select(markupConditionMapper.getFieldsToRead())
                .from(MARKUP_CONDITIONS)
                .where(MARKUP_CONDITIONS.DESCRIPTION.startsWith(markupConditionDescriptionPrefix))
                .orderBy(MARKUP_CONDITIONS.DESCRIPTION)
                .limit(limit)
                .fetch(markupConditionMapper::fromDb);
    }

    public void deleteMarkupConditions(Collection<Long> conditionIds) {
        if (conditionIds.isEmpty()) {
            return;
        }
        dslContextProvider.ppcdict()
                .deleteFrom(MARKUP_CONDITIONS)
                .where(MARKUP_CONDITIONS.CONDITION_ID.in(conditionIds))
                .execute();
    }

    private static JooqMapperWithSupplier<MarkupCondition> createMarkupConditionMapper() {
        return JooqMapperWithSupplierBuilder.builder(MarkupCondition::new)
                .map(property(MarkupCondition.ID, MARKUP_CONDITIONS.CONDITION_ID))
                .map(convertibleProperty(MarkupCondition.IS_ARCHIVED, MARKUP_CONDITIONS.IS_ARCHIVED,
                        RepositoryUtils::booleanFromLong, RepositoryUtils::booleanToLong))
                .map(property(MarkupCondition.GROUP, MARKUP_CONDITIONS.GROUP_KEY))
                .map(property(MarkupCondition.PRIORITY, MARKUP_CONDITIONS.PRIORITY))
                .map(convertibleProperty(MarkupCondition.OPERATOR, MARKUP_CONDITIONS.OPERATOR,
                        MarkupConditionsOperator::fromSource, MarkupConditionsOperator::toSource))
                .map(property(MarkupCondition.DESCRIPTION, MARKUP_CONDITIONS.DESCRIPTION))
                .map(convertibleProperty(MarkupCondition.TARGETING, MARKUP_CONDITIONS.TARGETINGS_CUSTOM,
                        json -> JsonUtils.fromJson(json, MarkupConditionTargeting.class),
                        JsonUtils::toJson))
                .build();
    }
}
